/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Reginald Stadlbauer <reggie@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#ifndef docstruct_h
#define docstruct_h

#include <klistview.h>
#include <tqwidget.h>

class KWDocument;
class KWFormulaFrameSet;
class KWPictureFrameSet;
class KWPartFrameSet;
class KWGUI;
class TQWidget;
class TQVBoxLayout;
class KWFrame;
class KWFrameSet;
class KWTextParag;

/******************************************************************/
/* Class: KWOrderedFrameSet                                       */
/******************************************************************/

/** Provides a way to sort framesets using a TQValueList based on their top left corner. */
class KWOrderedFrameSet
{
public:
    KWOrderedFrameSet(KWFrameSet* fs);
    KWOrderedFrameSet();    // default constructor
    bool operator<( KWOrderedFrameSet ofs);
    KWFrameSet* frameSet() { return m_frameset; }

private:
    KWFrameSet* m_frameset;
};

class KWDocListViewItem : public KListViewItem
{
public:
    enum ItemType {
        IT_BASE = FT_BASE,
        IT_TEXT = FT_TEXT,
        IT_PICTURE = FT_PICTURE,
        IT_PART = FT_PART,
        IT_FORMULA = FT_FORMULA,
        IT_CLIPART = FT_CLIPART,
        IT_TABLE = FT_TABLE,
        IT_ROOT = 11,
        IT_TEXTFRAME = 12,
        IT_PARAG = 13
    };

    KWDocListViewItem(TQListViewItem *_parent, const TQString &_text);
    KWDocListViewItem(TQListViewItem *_parent, TQListViewItem *_after, const TQString &_text);
    KWDocListViewItem(TQListView *_parent, const TQString &_text);
    virtual ItemType itemType() { return IT_BASE; }
    virtual void selectItem() {}
    virtual void editItem() {}
    virtual void deleteItem() {}
    virtual void editProperties() {}
    virtual void speakItem() {}
    virtual void contextMenu(TQListViewItem *, const TQPoint &, int) {}

protected:
    KWDocument* doc();
    KWGUI* gui();
    void deleteAllChildren();
};

/******************************************************************/
/* Class: KWDocStructParagItem                                    */
/******************************************************************/

class KWDocStructParagItem : public KWDocListViewItem
{
public:
    KWDocStructParagItem(TQListViewItem* parent, const TQString& text, KWTextParag* parag);
    KWDocStructParagItem(TQListViewItem* parent, TQListViewItem* after, const TQString& text, KWTextParag* parag);
    virtual ItemType itemType() { return IT_PARAG; }
    virtual void selectItem();
    virtual void editItem();
    virtual void deleteItem();
    virtual void editProperties();
    virtual void speakItem();
    virtual void contextMenu(TQListViewItem* item, const TQPoint& p, int);

    KWTextParag* parag() const { return m_parag; }

protected:
    KWTextParag* m_parag;
};

/******************************************************************/
/* Class: KWDocStructTextFrameItem                                */
/******************************************************************/

class KWDocStructTextFrameItem : public KWDocListViewItem
{
public:
    KWDocStructTextFrameItem(TQListViewItem* parent, const TQString& text,
        KWTextFrameSet* frameset, KWFrame* frame );
    KWDocStructTextFrameItem(TQListViewItem* parent, TQListViewItem* after, const TQString& text,
        KWTextFrameSet* frameset, KWFrame* frame );
    virtual int ItemType() { return IT_TEXTFRAME; }
    virtual void selectItem();
    virtual void editItem();
    virtual void deleteItem();
    virtual void editProperties();
    virtual void speakItem();
    virtual void contextMenu(TQListViewItem* item, const TQPoint& p, int);

    KWFrame* frame() const { return m_frame; }
    KWTextFrameSet* frameSet() const { return m_frameset; };
    void setupTextParags();
    KWDocStructParagItem* findTextParagItem(const KWTextParag* parag);

protected:
    KWFrame* m_frame;
    KWTextFrameSet* m_frameset;
};


/******************************************************************/
/* Class: KWDocStructTextFrameSetItem                                 */
/******************************************************************/

class KWDocStructTextFrameSetItem : public KWDocListViewItem
{
public:
    KWDocStructTextFrameSetItem(TQListViewItem* parent, const TQString& text, KWTextFrameSet* frameset);
    KWDocStructTextFrameSetItem(TQListViewItem* parent, TQListViewItem* after, const TQString& text,
        KWTextFrameSet* frameset );
    virtual int ItemType() { return IT_TEXT; }
    virtual void selectItem();
    virtual void editItem();
    virtual void deleteItem();
    virtual void editProperties();
    virtual void speakItem();
    virtual void contextMenu(TQListViewItem* item, const TQPoint& p, int);

    KWTextFrameSet* frameSet() const { return m_frameset; };
    void setupTextFrames();
    KWDocStructTextFrameItem* findTextFrameItem(const KWFrame* frame);

protected:
    KWTextFrameSet* m_frameset;

};

/******************************************************************/
/* Class: KWDocStructTableItem                                    */
/******************************************************************/

class KWDocStructTableItem : public KWDocListViewItem
{
public:
    KWDocStructTableItem(TQListViewItem* parent, const TQString& text, KWTableFrameSet* table );
    KWDocStructTableItem(TQListViewItem* parent, TQListViewItem* after, const TQString& text,
        KWTableFrameSet* table );
    virtual int ItemType() { return IT_TABLE; }
    virtual void selectItem();
    virtual void editItem();
    virtual void deleteItem();
    virtual void editProperties();
    virtual void speakItem();
    virtual void contextMenu(TQListViewItem* item, const TQPoint& p, int);

    KWTableFrameSet* table() const { return m_table; };
    void setupCells();
    KWDocStructTextFrameItem* findCellItem(const KWTextFrameSet* cell);

protected:
    KWTableFrameSet* m_table;
};

/******************************************************************/
/* Class: KWDocStructFormulaItem                                  */
/******************************************************************/

class KWDocStructFormulaItem : public KWDocListViewItem
{
public:
    KWDocStructFormulaItem(TQListViewItem* parent, const TQString& text, KWFormulaFrameSet* form);
    virtual int ItemType() { return IT_FORMULA; }
    virtual void selectItem();
    virtual void editItem();
    virtual void deleteItem();
    virtual void editProperties();
    virtual void speakItem();
    virtual void contextMenu(TQListViewItem* item, const TQPoint& p, int);

protected:
    KWFormulaFrameSet* m_form;
};


/******************************************************************/
/* Class: KWDocStructPictureItem                                  */
/******************************************************************/

class KWDocStructPictureItem : public KWDocListViewItem
{
public:
    KWDocStructPictureItem(TQListViewItem* parent, const TQString& text, KWPictureFrameSet* pic);
    virtual int ItemType() { return IT_PICTURE; }
    virtual void selectItem();
    virtual void editItem();
    virtual void deleteItem();
    virtual void editProperties();
    virtual void contextMenu(TQListViewItem* item, const TQPoint& p, int);

protected:
    KWPictureFrameSet* m_pic;
};

/******************************************************************/
/* Class: KWDocStructPartItem                                     */
/******************************************************************/

class KWDocStructPartItem : public KWDocListViewItem
{
public:
    KWDocStructPartItem(TQListViewItem* parent, const TQString& text, KWPartFrameSet* part);
    virtual int ItemType() { return IT_PART; }
    virtual void selectItem();
    virtual void editItem();
    virtual void deleteItem();
    virtual void editProperties();
    virtual void contextMenu(TQListViewItem* item, const TQPoint& p, int);

protected:
    KWPartFrameSet* m_part;
};

/******************************************************************/
/* Class: KWDocStructRootItem                                     */
/******************************************************************/

class KWDocStructRootItem : public KWDocListViewItem
{
public:
    KWDocStructRootItem(TQListView* parent, const TQString& text, TypeStructDocItem type);

    virtual int ItemType() { return IT_ROOT; }
    void setupArrangement();
    void setupTextFrameSets();
    void setupTables();
    void setupPictures();
    void setupEmbedded();
    void setupFormulaFrames();
    virtual void setOpen(bool o);

protected:
    KWDocStructTextFrameSetItem* findTextFrameSetItem(const KWFrameSet* frameset);
    KWDocStructTableItem* findTableItem(const KWFrameSet* frameset);

    TypeStructDocItem m_type;

};

/******************************************************************/
/* Class: KWDocStructTree                                         */
/******************************************************************/

class KWDocStructTree : public KListView
{
    Q_OBJECT
  TQ_OBJECT

public:
    KWDocStructTree( TQWidget* parent, KWDocument* doc, KWGUI* gui );
    ~KWDocStructTree();

    void setup();
    void refreshTree(int _type);
    TQSize minimumSizeHint() const {
        return TQSize( 0, 0 );
    }
    void selectItem();
    void editItem();
    void deleteItem();
    void editProperties();
    void speakItem();

    KWDocument* doc() { return m_doc; };
    KWGUI* gui() { return m_gui; };

public slots:
    void slotContextMenu(KListView* l, TQListViewItem* i, const TQPoint& p);
    void slotRightButtonClicked(TQListViewItem*, const TQPoint&, int);
    void slotDoubleClicked(TQListViewItem* item);
    void slotReturnPressed(TQListViewItem* item);

protected:
    KWDocument* m_doc;
    KWGUI* m_gui;

    KWDocStructRootItem *arrangement, *tables, *pictures, *textfrms, *embedded, *formulafrms;
};

/******************************************************************/
/* Class: KWDocStruct                                             */
/******************************************************************/

class KWDocStruct : public TQWidget
{
    Q_OBJECT
  TQ_OBJECT

public:
    KWDocStruct( TQWidget* parent, KWDocument* doc, KWGUI* gui );
    void selectItem();
    void editItem();
    void deleteItem();
    void editProperties();
    void speakItem();
    /** Refresh the specified branches of the tree.  Actual refreshing is deferred
        until the paint event. */
    void refreshTree(int _type);
    /** Set focus to the tree, if visible. */
    void setFocusHere();

public slots:
    /** Refresh the entire tree. */
    void refreshEntireTree();

protected:
    KWDocStructTree* m_tree;
    TQVBoxLayout* m_layout;

    virtual void paintEvent(TQPaintEvent* ev);

    KWDocument* m_doc;
    KWGUI* m_gui;
    int dirtyTreeTypes;
};

#endif
