/***************************************************************************
 * This file is part of the KDE project
 * copyright (C) 2005 by Sebastian Sauer (mail@dipe.org)
 * copyright (C) 2005 by Tobi Krebs (tobi.krebs@gmail.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * You should have received a copy of the GNU Library General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "variable.h"
#include "exception.h"

#include <kdebug.h>

using namespace KoMacro;

namespace KoMacro {

	/**
	* @internal d-pointer class to be more flexible on future extension of the
	* functionality without to much risk to break the binary compatibility.
	*/
	class Variable::Private
	{
		public:

			/**
			 * The name this @a Variable has.
			 */
			TQString name;

			/**
			 * The i18n-caption used for display purposes only
			 * this @a Variable has.
			 */
			TQString text;

			/**
			* If @a Variable::Type is @a Variable::TypeVariant this TQVariant
			* holds the value else it's invalid.
			*/
			TQVariant variant;

			/**
			* If @a Variable::Type is @a Variable::TypeObject this TQObject is
			* the value else it's NULL.
			*/
			const TQObject* object;

			/**
			* Optional list of children this @a Variable has.
			*/
			// TODO Dow we use this or is it for the future??
			Variable::List children;

			/**
			* Defines if the variable is enabled or disabled.
			*/
			bool enabled;

			explicit Private()
				: enabled(true)
			{
			}
	};

}

Variable::Variable()
	: MetaParameter()
	, d( new Private() ) // create the private d-pointer instance.
{
	setType(TypeNone);
	d->object = 0;
}

Variable::Variable(const TQVariant& variant, const TQString& name, const TQString& text)
	: MetaParameter()
	, d( new Private() ) // create the private d-pointer instance.
{
	setVariantType(variant.type());
	d->variant = variant;
	d->object = 0;
	d->name = name;
	d->text = text;
}

Variable::Variable(const TQObject* object)
	: MetaParameter()
	, d( new Private() ) // create the private d-pointer instance.
{
	setType(TypeObject);
	d->object = object;
}

Variable::Variable(const TQDomElement& element)
	: MetaParameter()
	, d( new Private() ) // create the private d-pointer instance.
{

	TQString typesignature = element.attribute("type", "const TQString&");
	TQString value = element.text();

	setSignatureArgument( typesignature );

	switch( type() ) {
		case KoMacro::MetaParameter::TypeVariant: {
			//kdDebug() << TQString("KoMacro::Variable(TQDomElement) KoMacro::MetaParameter::TypeVariant") << endl;
			// Set the variant without overwritting the previously detected varianttype.
			setVariant( TQVariant(value), false );
		} break;
		case KoMacro::MetaParameter::TypeObject: {
			//kdDebug() << TQString("KoMacro::Variable(TQDomElement) KoMacro::MetaParameter::TypeObject") << endl;
			//TODO setObject();
		} break;
		default: {
			kdWarning() << TQString("KoMacro::Variable(TQDomElement) KoMacro::MetaParameter::TypeNone") << endl;
		} break;
	}
}

Variable::~Variable()
{
	delete d;
}

TQString Variable::name() const
{
	return d->name;
}

void Variable::setName(const TQString& name)
{
	d->name = name;
}

TQString Variable::text() const
{
	return d->text;
}

void Variable::setText(const TQString& text)
{
	d->text = text;
}

const TQVariant Variable::variant() const
{
	//Q_ASSERT( type() == MetaParameter::TypeVariant );
	//Q_ASSERT( variantType() != TQVariant::Invalid );
	//if(variantType() == TQVariant::Invalid) return TQVariant();
	return d->variant;
}

void Variable::setVariant(const TQVariant& variant, bool detecttype)
{
	if(detecttype) {
		setVariantType( variant.type() );
	}
	d->variant = variant;
}

const TQObject* Variable::object() const
{
	Q_ASSERT( ! d->object );
	return d->object;
}

void Variable::setObject(const TQObject* object)
{
	setType(TypeObject);
	d->object = object;
}

Variable::operator TQVariant () const
{
	return variant();
}

Variable::operator const TQObject* () const
{
	return object();
}

const TQString Variable::toString() const
{
	switch( type() ) {
		case KoMacro::MetaParameter::TypeVariant: {
			return variant().toString();
		} break;
		case KoMacro::MetaParameter::TypeObject: {
			return TQString("[%1]").arg( object()->name() );
		} break;
		default: {
			throw Exception("Type is undefined.");
		} break;
	}
	return TQString();
}

int Variable::toInt() const
{
	return variant().toInt();
}

Variable::List Variable::children() const
{
	return d->children;
}

void Variable::appendChild(KSharedPtr<Variable> variable)
{
	d->children.append(variable);
}

void Variable::clearChildren()
{
	d->children.clear();
}

void Variable::setChildren(const Variable::List& children)
{
	d->children = children;
}

/*
bool Variable::isEnabled() const
{
	return d->enabled;
}

void Variable::setEnabled(const bool enabled)
{
	d->enabled = enabled;
}
*/
