/***************************************************************************
 * This file is part of the KDE project
 * copyright (C) 2006 by Sebastian Sauer (mail@dipe.org)
 * copyright (C) 2006 by Bernd Steindorff (bernd@itii.de)
 * copyright (C) 2006 by Sascha Kupper (kusato@kfnv.de)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * You should have received a copy of the GNU Library General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "openaction.h"

#include <core/kexi.h>
#include <core/kexiproject.h>
#include <core/kexipartmanager.h>
#include <core/kexipartinfo.h>
#include <core/kexipart.h>
#include <core/keximainwindow.h>

#include <klocale.h>

using namespace KexiMacro;

namespace KexiMacro {

	static const TQString DATAVIEW = "data";
	static const TQString DESIGNVIEW = "design";
	static const TQString TEXTVIEW = "text";
	
	static const TQString OBJECT = "object";
	static const TQString NAME = "name";
	static const TQString VIEW = "view";

	/**
	* The ViewVariable class provide a list of viewmodes supported
	* by a KexiPart::Part as @a KoMacro::Variable .
	*/
	template<class ACTIONIMPL>
	class ViewVariable : public KexiVariable<ACTIONIMPL>
	{
		public:
			ViewVariable(ACTIONIMPL* actionimpl, const TQString& objectname = TQString(), const TQString& viewname = TQString())
				: KexiVariable<ACTIONIMPL>(actionimpl, VIEW, i18n("View"))
			{
				TQStringList namelist;
				KexiPart::Part* part = Kexi::partManager().partForMimeType( TQString("kexi/%1").arg(objectname) );
				if(part) {
					int viewmodes = part->supportedViewModes();
					if(viewmodes & Kexi::DataViewMode)
						namelist << DATAVIEW;
					if(viewmodes & Kexi::DesignViewMode)
						namelist << DESIGNVIEW;
					if(viewmodes & Kexi::TextViewMode)
						namelist << TEXTVIEW;
					for(TQStringList::Iterator it = namelist.begin(); it != namelist.end(); ++it)
						this->children().append( KSharedPtr<KoMacro::Variable>(new KoMacro::Variable(*it)) );
				}
				const TQString n =
					namelist.contains(viewname)
						? viewname
						: namelist.count() > 0 ? namelist[0] : "";

				this->setVariant(n);
			}
	};

}

OpenAction::OpenAction()
	: KexiAction("open", i18n("Open"))
{
	const int conditions = ObjectVariable<OpenAction>::VisibleInNav;
	
	KSharedPtr<KoMacro::Variable> objvar = new ObjectVariable<OpenAction>(this, conditions);
	setVariable(objvar);

	setVariable(KSharedPtr<KoMacro::Variable>( new ObjectNameVariable<OpenAction>(this, objvar->variant().toString()) ));
	setVariable(KSharedPtr<KoMacro::Variable>( new ViewVariable<OpenAction>(this, objvar->variant().toString()) ));
}

OpenAction::~OpenAction()
{
}

bool OpenAction::notifyUpdated(KSharedPtr<KoMacro::MacroItem> macroitem, const TQString& name)
{
	kdDebug()<<"OpenAction::notifyUpdated() name="<<name<<" macroitem.action="<<(macroitem->action() ? macroitem->action()->name() : "NOACTION")<<endl;
	KSharedPtr<KoMacro::Variable> variable = macroitem->variable(name, false);
	if(! variable) {
		kdWarning()<<"OpenAction::notifyUpdated() No such variable="<<name<<" in macroitem."<<endl;
		return false;
	}

	variable->clearChildren();
	if(name == OBJECT) {
		const TQString objectvalue = macroitem->variant(OBJECT, true).toString(); // e.g. "table" or "query"
		const TQString objectname = macroitem->variant(NAME, true).toString(); // e.g. "table1" or "table2" if objectvalue above is "table"
		const TQString viewname = macroitem->variant(VIEW, true).toString(); // "data", "design" or "text"

		macroitem->variable(NAME, true)->setChildren(
			KoMacro::Variable::List() << KSharedPtr<KoMacro::Variable>(new ObjectNameVariable<OpenAction>(this, objectvalue, objectname)) );
		macroitem->variable(VIEW, true)->setChildren(
			KoMacro::Variable::List() << KSharedPtr<KoMacro::Variable>(new ViewVariable<OpenAction>(this, objectvalue, viewname)) );
	}

	return true;
}

void OpenAction::activate(KSharedPtr<KoMacro::Context> context)
{
	if(! mainWin()->project()) {
		throw KoMacro::Exception(i18n("No project loaded."));
	}

	const TQString objectname = context->variable(OBJECT)->variant().toString();
	const TQString name = context->variable(NAME)->variant().toString();
	KexiPart::Item* item = mainWin()->project()->itemForMimeType( TQString("kexi/%1").arg(objectname).latin1(), name );
	if(! item) {
		throw KoMacro::Exception(i18n("No such object \"%1.%2\".").arg(objectname).arg(name));
	}

	// Determinate the viewmode.
	const TQString view = context->variable(VIEW)->variant().toString();
	int viewmode;
	if(view == DATAVIEW)
		viewmode = Kexi::DataViewMode;
	else if(view == DESIGNVIEW)
		viewmode = Kexi::DesignViewMode;
	else if(view == TEXTVIEW)
		viewmode = Kexi::TextViewMode;
	else {
		throw KoMacro::Exception(i18n("No such viewmode \"%1\" in object \"%2.%3\".").arg(view).arg(objectname).arg(name));
	}

	// Try to open the object now.
	bool openingCancelled;
	if(! mainWin()->openObject(item, viewmode, openingCancelled)) {
		if(! openingCancelled) {
			throw KoMacro::Exception(i18n("Failed to open object \"%1.%2\".").arg(objectname).arg(name));
		}
	}
}

//#include "openaction.moc"
