/* This file is part of the KDE project
   Copyright (C) 2004 Cedric Pasteur <cedric.pasteur@free.fr>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "scriptIO.h"
#include "formscript.h"
#include "kexievents.h"

#include "form.h"
#include "objecttree.h"
// Kross includes
#include "main/scriptcontainer.h"

bool
ScriptIO::saveFormEvents(TQDomNode &parentNode, FormScript *formScript)
{
	TQDomDocument domDoc = parentNode.ownerDocument();

	// Save the form's code
	if(!formScript->getCode().isEmpty()) {
		TQDomElement  script = domDoc.createElement("script");
		script.setAttribute("language", formScript->scriptContainer()->getInterpreterName());
		parentNode.appendChild(script);
		TQDomText scriptCode = domDoc.createTextNode(formScript->getCode());
		script.appendChild(scriptCode);
	}

	// Save all form events
	if(!formScript->eventList()->isEmpty())
		saveEventList(formScript->eventList(), parentNode);
	return true;
}

bool
ScriptIO::loadFormEvents(TQDomNode &parentNode, Form *form, ScriptManager *manager)
{
	TQDomElement script = parentNode.namedItem("script").toElement();
	TQDomElement events = parentNode.namedItem("events").toElement();

	// Load script code
	FormScript *formScript = new FormScript(form, manager);
	if(!script.isNull()) {
		formScript->scriptContainer()->setInterpreterName(script.attribute("language"));
		formScript->setCode(script.text());
	}

	// Load all events in the EventList
	if(!events.isNull()) {
		for(TQDomNode n = events.firstChild(); !n.isNull(); n = n.nextSibling())
			loadEvent(n, formScript->eventList(), form);
	}
	return true;
}

bool
ScriptIO::saveAllEventsForWidget(TQObject *widget, FormScript *formScript, TQDomNode &node)
{
	EventList *l = formScript->eventList()->allEventsForObject(widget);
	saveEventList(l, node);
	return true;
}

void
ScriptIO::saveEvent(Event *event, TQDomNode &parentNode)
{
	if(!event)
		return;

	TQDomDocument domDoc = parentNode.ownerDocument();
	TQDomElement eventNode = domDoc.createElement("event");
	eventNode.setAttribute("type", event->type());
	parentNode.appendChild(eventNode);

	switch(event->type()) {
		case Event::Slot: {
			TQDomElement sender = domDoc.createElement("sender");
			eventNode.appendChild(sender);
			TQDomText senderText = domDoc.createTextNode(event->sender() ? event->sender()->name() : "");
			sender.appendChild(senderText);

			TQDomElement signal = domDoc.createElement("signal");
			eventNode.appendChild(signal);
			TQDomText signalText = domDoc.createTextNode(event->signal());
			signal.appendChild(signalText);

			TQDomElement receiver = domDoc.createElement("receiver");
			eventNode.appendChild(receiver);
			TQDomText receiverText = domDoc.createTextNode(event->receiver() ? event->receiver()->name() : "");
			receiver.appendChild(receiverText);

			TQDomElement slot = domDoc.createElement("slot");
			eventNode.appendChild(slot);
			TQDomText slotText = domDoc.createTextNode(event->slot());
			slot.appendChild(slotText);
			break;
		}

		case Event::UserFunction: {
			TQDomElement sender = domDoc.createElement("sender");
			eventNode.appendChild(sender);
			TQDomText senderText = domDoc.createTextNode(event->sender() ? event->sender()->name() : "");
			sender.appendChild(senderText);

			TQDomElement signal = domDoc.createElement("signal");
			eventNode.appendChild(signal);
			TQDomText signalText = domDoc.createTextNode(event->signal());
			signal.appendChild(signalText);

			TQDomElement function = domDoc.createElement("function");
			eventNode.appendChild(function);
			TQDomText functionText = domDoc.createTextNode(event->slot());
			function.appendChild(functionText);
			break;
		}

		case Event::Action:
			// !\todo
			break;

		default:
			break;
	}
}

void
ScriptIO::saveEventList(EventList *list, TQDomNode &parentNode)
{
	if(!list || list->isEmpty())
		return;

	TQDomDocument domDoc = parentNode.ownerDocument();
	TQDomElement events = domDoc.createElement("events");
	parentNode.appendChild(events);

	TQValueListConstIterator<Event*> endIt = list->constEnd();
	for(TQValueListConstIterator<Event*> it =  list->constBegin(); it != endIt; ++it)
		saveEvent((*it), events);
}

void
ScriptIO::loadEvent(TQDomNode &node, EventList *list, Form *form)
{
	int type = node.toElement().attribute("type").toInt();
	Event *event = new Event();

	switch(type) {
		case Event::Slot: {
			ObjectTreeItem *sender = form->objectTree()->lookup(node.namedItem("sender").toElement().text());
			event->setSender(sender ? sender->widget() : 0);
			event->setSignal(node.namedItem("signal").toElement().text().local8Bit());
			ObjectTreeItem *receiver = form->objectTree()->lookup(node.namedItem("receiver").toElement().text());
			event->setReceiver(receiver ? receiver->widget() : 0);
			event->setSlot(node.namedItem("slot").toElement().text().local8Bit());
			event->setType(Event::Slot);
			break;
		}

		case Event::UserFunction: {
			ObjectTreeItem *sender = form->objectTree()->lookup(node.namedItem("sender").toElement().text());
			event->setSender(sender ? sender->widget() : 0);
			event->setSignal(node.namedItem("signal").toElement().text().local8Bit());
			event->setSlot(node.namedItem("function").toElement().text().local8Bit());
			event->setType(Event::UserFunction);
			break;
		}
		default:  break;
	}

	list->addEvent(event);
}

