/* This file is part of the KDE project
   Copyright (C) 2001, 2002, 2003 The Karbon Developers

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/


#include "vpolyline.h"
#include <tqdom.h>

#include "vglobal.h"
#include <klocale.h>
#include <vdocument.h>
#include "vtransformcmd.h"

#include <KoStore.h>
#include <KoXmlWriter.h>
#include <KoXmlNS.h>

VPolyline::VPolyline( VObject* parent, VState state )
	: VPath( parent, state )
{
}

/*VPolyline::VPolyline( VObject* parent, VState state ) 
	: VPath( parent, state )
{
}*/

/*VPolyline::VPolyline( VObject* parent, const TQString &points ) 
	: VPath( parent ), m_points( points )
{
	init();
}*/

void
VPolyline::init()
{
	bool bFirst = true;

	TQString points = m_points.simplifyWhiteSpace();
	points.replace( ',', ' ' );
	points.remove( '\r' );
	points.remove( '\n' );
	TQStringList pointList = TQStringList::split( ' ', points );
	TQStringList::Iterator end(pointList.end());
	for( TQStringList::Iterator it = pointList.begin(); it != end; ++it )
	{
		KoPoint point;
		point.setX( (*it).toDouble() );
		point.setY( (*++it).toDouble() );
		if( bFirst )
		{
			moveTo( point );
			bFirst = false;
		}
		else
			lineTo( point );
	}
}

TQString
VPolyline::name() const
{
	TQString result = VObject::name();
	return !result.isEmpty() ? result : i18n( "Polyline" );
}

void
VPolyline::save( TQDomElement& element ) const
{
	VDocument *doc = document();
	if( doc && doc->saveAsPath() )
	{
		VPath::save( element );
		return;
	}

	if( state() != deleted )
	{
		TQDomElement me = element.ownerDocument().createElement( "POLYLINE" );
		element.appendChild( me );

		// save fill/stroke untransformed
		VPath path( *this );
		VTransformCmd cmd( 0L, m_matrix.invert() );
		cmd.visit( path );
		path.VObject::save( me );
		//VObject::save( me );

		me.setAttribute( "points", m_points );

		TQString transform = buildSvgTransform();
		if( !transform.isEmpty() )
			me.setAttribute( "transform", transform );
	}
}

void
VPolyline::saveOasis( KoStore *store, KoXmlWriter *docWriter, KoGenStyles &mainStyles, int &index ) const
{
	// do not save deleted objects
	if( state() == deleted )
		return;

	docWriter->startElement( "draw:polyline" );

	docWriter->addAttribute( "svg:points", m_points );

	VObject::saveOasis( store, docWriter, mainStyles, index );

	docWriter->endElement();
}

void
VPolyline::load( const TQDomElement& element )
{
	setState( normal );

	TQDomNodeList list = element.childNodes();
	for( uint i = 0; i < list.count(); ++i )
		if( list.item( i ).isElement() )
			VObject::load( list.item( i ).toElement() );

	m_points = element.attribute( "points" );

	init();

	TQString trafo = element.attribute( "transform" );
	if( !trafo.isEmpty() )
		transform( trafo );
}

bool
VPolyline::loadOasis( const TQDomElement &element, KoOasisLoadingContext &context )
{
	setState( normal );

	if( element.localName() == "line" )
	{
		KoPoint p1, p2;
		p1.setX( KoUnit::parseValue( element.attributeNS( KoXmlNS::svg, "x1", TQString() ) ) );
		p1.setY( KoUnit::parseValue( element.attributeNS( KoXmlNS::svg, "y1", TQString() ) ) );
		p2.setX( KoUnit::parseValue( element.attributeNS( KoXmlNS::svg, "x2", TQString() ) ) );
		p2.setY( KoUnit::parseValue( element.attributeNS( KoXmlNS::svg, "y2", TQString() ) ) );
		
		m_points = TQString( "%1,%2 %3,%4" ).arg( p1.x() ).arg( p1.y() ).arg( p2.x() ).arg( p2.y() );

		moveTo( p1 );
		lineTo( p2 );
	}
	else if( element.localName() == "polyline" )
	{
		m_points = element.attributeNS( KoXmlNS::draw, "points", TQString() );
		init();
	}

	transformByViewbox( element, element.attributeNS( KoXmlNS::svg, "viewBox", TQString() ) );

	TQString trafo = element.attributeNS( KoXmlNS::draw, "transform", TQString() );
	if( !trafo.isEmpty() )
		transformOasis( trafo );

	return VObject::loadOasis( element, context );
}

VPath* 
VPolyline::clone() const
{
	return new VPolyline( *this );
}
