/*
 * Copyright (c) 2002-2003 Nicolas HADACEK (hadacek@kde.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "dialog.h"
#include "dialog.moc"

#include <tqhbox.h>
#include <tqvgroupbox.h>
#include <tqregexp.h>
#include <tqapplication.h>
#include <tqgrid.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqbuttongroup.h>
#include <tqradiobutton.h>
#include <tqwhatsthis.h>
#include <tqcheckbox.h>

#include <klocale.h>
#include <kdebug.h>
#include <klineedit.h>


//-----------------------------------------------------------------------------
SelectionRange::SelectionRange(const TQString &s)
{
    // fill
    TQValueVector<TQPair<uint, uint> > r;
    TQStringList list = TQStringList::split(',', s);
    TQRegExp range("^([0-9]+)\\-([0-9]+)$");
    TQRegExp one("^[0-9]+$");
    for (TQStringList::iterator it = list.begin(); it!=list.end(); ++it) {
        if ( one.exactMatch(*it) ) {
            uint p = (*it).toUInt();
            r.push_back( tqMakePair(p, p) );
        } else if ( range.exactMatch(*it) ) {
            uint p1 = range.cap(1).toUInt();
            uint p2 = range.cap(2).toUInt();
            if ( p1>p2 ) continue;
            r.push_back( tqMakePair(p1, p2) );
        }
    }

    // order
    TQPair<uint, uint> tmp;
    for (uint i=1; i<r.size(); i++)
        if ( r[i].first<r[i-1].first )
            tqSwap(r[i-1], r[i]);

    // coalesce
    for (uint i=0; i<r.size(); i++)
        if ( i!=0 && r[i].first<=tmp.second )
            tmp.second = kMax(tmp.second, r[i].second);
        else {
            _ranges.push_back(r[i]);
            tmp = r[i];
            kdDebug(30516) << "selection range: (" << tmp.first << ","
                           << tmp.second << ") " << endl;
        }
}

uint SelectionRange::nbPages() const
{
    uint nb = 0;
    for (uint i=0; i<_ranges.size(); i++)
        nb += _ranges[i].second - _ranges[i].first + 1;
    return nb;
}

SelectionRangeIterator::SelectionRangeIterator(const SelectionRange &range)
    : _ranges(range._ranges)
{
    toFirst();
}

int SelectionRangeIterator::toFirst()
{
    if ( _ranges.size()==0 ) _current = -1;
    else {
        _index = 0;
        _current = _ranges[0].first;
    }
    return _current;
}

int SelectionRangeIterator::next()
{
    if ( _current==-1 ) return -1;
    if ( _current==int(_ranges[_index].second) ) {
        _index++;
        _current = (_index==_ranges.size() ? -1
                    : int(_ranges[_index].first));
    } else _current++;
    return _current;
}

//-----------------------------------------------------------------------------
namespace PDFImport
{

Dialog::Dialog(uint nbPages, bool isEncrypted, TQWidget *widget)
    : KDialogBase(Plain, i18n("KWord's PDF Import Filter"), Ok|Cancel, Ok,
                  widget, "pdf_import_dialog"), _nbPages(nbPages)
{
    TQApplication::restoreOverrideCursor();

    TQVBoxLayout *top = new TQVBoxLayout(plainPage(), KDialogBase::marginHint(),
                                       KDialogBase::spacingHint());

    // page selection
    TQVGroupBox *gbox = new TQVGroupBox(i18n("Page Selection"), plainPage());
    gbox->setInsideSpacing(KDialogBase::spacingHint());
    top->addWidget(gbox);
    _group = new TQButtonGroup;
    _allButton = new TQRadioButton(i18n("All (%1 pages)").arg(nbPages), gbox);
    _allButton->setChecked(true);
    _group->insert(_allButton);
    TQHBox *hbox = new TQHBox(gbox);
    _rangeButton = new TQRadioButton(i18n("Range:"), hbox);
    _group->insert(_rangeButton);
    _range = new KLineEdit(hbox);
    _range->setFocus();
    connect(_range, TQT_SIGNAL(textChanged(const TQString &)),
            TQT_SLOT(rangeChanged(const TQString &)));

    // options
    _images = new TQCheckBox(i18n("Import images"), plainPage());
    _images->setChecked(true);
    top->addWidget(_images);
    _smart = new TQCheckBox(i18n("\"Smart\" mode"), plainPage());
    _smart->setChecked(true);
    TQWhatsThis::add(_smart,
                    i18n("Removes returns and hyphens at end of line. "
                         "Also tries to compute the paragraph alignment. "
                         "Note that the layout of some pages can "
                         "get messed up."));
    top->addWidget(_smart);

    // passwords
    gbox = new TQVGroupBox(i18n("Passwords"), plainPage());
    top->addWidget(gbox);
    TQGrid *grid = new TQGrid(2, gbox);
    grid->setSpacing(KDialogBase::spacingHint());
    (void)new TQLabel(i18n("Owner:"), grid);
    _owner = new KLineEdit(grid);
    _owner->setEchoMode(TQLineEdit::Password);
    (void)new TQLabel(i18n("User:"), grid);
    _user = new KLineEdit(grid);
    _user->setEchoMode(TQLineEdit::Password);
    grid->setEnabled(isEncrypted);
}

Dialog::~Dialog()
{
    delete _group;
    TQApplication::setOverrideCursor(TQt::waitCursor);
}

void Dialog::rangeChanged(const TQString &)
{
    _rangeButton->setChecked(true);
}

Options Dialog::options() const
{
    Options o;
    o.range = SelectionRange( (_allButton->isChecked() ?
                             TQString("1-%1").arg(_nbPages) : _range->text()) );
    o.ownerPassword = _owner->text();
    o.userPassword = _user->text();
    o.importImages = _images->isChecked();
    o.smart = _smart->isChecked();
    return o;
}

} // namespace
