/*
 *  Copyright (c) 2002 Patrick Julien <freak@codepimps.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
#include <limits.h>

#include <kglobalsettings.h>
#include <kconfig.h>
#include <kglobal.h>
#include <kdebug.h>
#include <config.h>

#include LCMS_HEADER

#include "kis_global.h"
#include "kis_config.h"

namespace {
    const double IMG_DEFAULT_RESOLUTION = 100.0;
    const TQ_INT32 IMG_DEFAULT_WIDTH = 512;
    const TQ_INT32 IMG_DEFAULT_HEIGHT = 512;
    const enumCursorStyle DEFAULT_CURSOR_STYLE = CURSOR_STYLE_OUTLINE;
    const TQ_INT32 DEFAULT_MAX_THREADS = 4;
    const TQ_INT32 DEFAULT_MAX_TILES_MEM = 500; // 8192 kilobytes given 64x64 tiles with 32bpp
    const TQ_INT32 DEFAULT_SWAPPINESS = 100;
    const TQ_INT32 DEFAULT_PRESSURE_CORRECTION = 50;
    const TQ_INT32 DEFAULT_DOCKABILITY = 0;
    const TQ_INT32 DEFAULT_UNDO_LIMIT = 50;
}

KisConfig::KisConfig()
{

    m_cfg = KGlobal::config();
    if (!m_cfg) {
        // Allow unit tests to test parts of the code without having to run the
        // full application.
        m_cfg = new KConfig();
    }
    m_cfg->setGroup("");
}

KisConfig::~KisConfig()
{
    m_cfg->sync();
}


bool KisConfig::fixDockerWidth() const
{
    return m_cfg->readBoolEntry("fixDockerWidth", true);
}

void KisConfig::setFixedDockerWidth(bool fix)
{
    m_cfg->writeEntry("fixDockerWidth", fix);
}

bool KisConfig::undoEnabled() const
{
    return m_cfg->readBoolEntry("undoEnabled", true);
}

void KisConfig::setUndoEnabled(bool undo)
{
    m_cfg->writeEntry("undoEnabled", undo);
}


TQ_INT32 KisConfig::defUndoLimit() const
{
    return m_cfg->readNumEntry("undolimit", DEFAULT_UNDO_LIMIT);
}

void KisConfig::defUndoLimit(TQ_INT32 limit)
{
    m_cfg->writeEntry("undolimit", limit);
}

TQ_INT32 KisConfig::defImgWidth() const
{
    return m_cfg->readNumEntry("imgWidthDef", IMG_DEFAULT_WIDTH);
}

TQ_INT32 KisConfig::defImgHeight() const
{
    return m_cfg->readNumEntry("imgHeightDef", IMG_DEFAULT_HEIGHT);
}

double KisConfig::defImgResolution() const
{
    return m_cfg->readDoubleNumEntry("imgResolutionDef", IMG_DEFAULT_RESOLUTION);
}

void KisConfig::defImgWidth(TQ_INT32 width)
{
    m_cfg->writeEntry("imgWidthDef", width);
}

void KisConfig::defImgHeight(TQ_INT32 height)
{
    m_cfg->writeEntry("imgHeightDef", height);
}

void KisConfig::defImgResolution(double res)
{
    m_cfg->writeEntry("imgResolutionDef", res);
}

enumCursorStyle KisConfig::cursorStyle() const
{
    return (enumCursorStyle) m_cfg->readNumEntry("cursorStyleDef", DEFAULT_CURSOR_STYLE);
}

enumCursorStyle KisConfig::getDefaultCursorStyle() const
{
    return DEFAULT_CURSOR_STYLE;
}

void KisConfig::setCursorStyle(enumCursorStyle style)
{
    m_cfg->writeEntry("cursorStyleDef", style);
}


TQString KisConfig::monitorProfile() const
{
    return m_cfg->readEntry("monitorProfile", "");
}

void KisConfig::setMonitorProfile(TQString monitorProfile)
{
    m_cfg->writeEntry("monitorProfile", monitorProfile);
}


TQString KisConfig::workingColorSpace() const
{
    return m_cfg->readEntry("workingColorSpace", "RGBA");
}

void KisConfig::setWorkingColorSpace(TQString workingColorSpace)
{
    m_cfg->writeEntry(workingColorSpace, workingColorSpace);
}


TQString KisConfig::printerColorSpace() const
{
    return m_cfg->readEntry("printerColorSpace", "CMYK");
}

void KisConfig::setPrinterColorSpace(TQString printerColorSpace)
{
    m_cfg->writeEntry("printerColorSpace", printerColorSpace);
}


TQString KisConfig::printerProfile() const
{
    return m_cfg->readEntry("printerProfile", "");
}

void KisConfig::setPrinterProfile(TQString printerProfile)
{
    m_cfg->writeEntry("printerProfile", printerProfile);
}


bool KisConfig::useBlackPointCompensation() const
{
    return m_cfg->readBoolEntry("useBlackPointCompensation", false);
}

void KisConfig::setUseBlackPointCompensation(bool useBlackPointCompensation)
{
    m_cfg->writeEntry("useBlackPointCompensation", useBlackPointCompensation);
}


bool KisConfig::showRulers() const
{
    return m_cfg->readBoolEntry("showrulers", false);
}

void KisConfig::setShowRulers(bool rulers)
{
    m_cfg->writeEntry("showrulers", rulers);
}


TQ_INT32 KisConfig::pasteBehaviour() const
{
    return m_cfg->readNumEntry("pasteBehaviour", 2);
}

void KisConfig::setPasteBehaviour(TQ_INT32 renderIntent)
{
    m_cfg->writeEntry("pasteBehaviour", renderIntent);
}


TQ_INT32 KisConfig::renderIntent() const
{
    return m_cfg->readNumEntry("renderIntent", INTENT_PERCEPTUAL);
}

void KisConfig::setRenderIntent(TQ_INT32 renderIntent)
{
    m_cfg->writeEntry("renderIntent", renderIntent);
}

bool KisConfig::useOpenGL() const
{
    return m_cfg->readBoolEntry("useOpenGL", false);
}

void KisConfig::setUseOpenGL(bool useOpenGL)
{
    m_cfg->writeEntry("useOpenGL", useOpenGL);
}

bool KisConfig::useOpenGLShaders() const
{
    return m_cfg->readBoolEntry("useOpenGLShaders", false);
}

void KisConfig::setUseOpenGLShaders(bool useOpenGLShaders)
{
    m_cfg->writeEntry("useOpenGLShaders", useOpenGLShaders);
}

TQ_INT32 KisConfig::maxNumberOfThreads()
{
    return m_cfg->readNumEntry("maxthreads", DEFAULT_MAX_THREADS);
}

void KisConfig::setMaxNumberOfThreads(TQ_INT32 maxThreads)
{
    m_cfg->writeEntry("maxthreads", maxThreads);
}

TQ_INT32 KisConfig::maxTilesInMem() const
{
    return m_cfg->readNumEntry("maxtilesinmem", DEFAULT_MAX_TILES_MEM);
}

void KisConfig::setMaxTilesInMem(TQ_INT32 tiles)
{
    m_cfg->writeEntry("maxtilesinmem", tiles);
}

TQ_INT32 KisConfig::swappiness() const
{
    return m_cfg->readNumEntry("swappiness", DEFAULT_SWAPPINESS);
}

void KisConfig::setSwappiness(TQ_INT32 swappiness)
{
    m_cfg->writeEntry("swappiness", swappiness);
}

TQ_INT32 KisConfig::getPressureCorrection()
{
    return m_cfg->readNumEntry( "pressurecorrection", DEFAULT_PRESSURE_CORRECTION );
}

void KisConfig::setPressureCorrection( TQ_INT32 correction )
{
    m_cfg->writeEntry( "pressurecorrection",  correction );
}

TQ_INT32 KisConfig::getDefaultPressureCorrection()
{
    return DEFAULT_PRESSURE_CORRECTION;
}

bool KisConfig::tabletDeviceEnabled(const TQString& tabletDeviceName) const
{
    return m_cfg->readBoolEntry("TabletDevice" + tabletDeviceName + "Enabled", false);
}

void KisConfig::setTabletDeviceEnabled(const TQString& tabletDeviceName, bool enabled)
{
    m_cfg->writeEntry("TabletDevice" + tabletDeviceName + "Enabled", enabled);
}

TQ_INT32 KisConfig::tabletDeviceAxis(const TQString& tabletDeviceName, const TQString& axisName, TQ_INT32 defaultAxis) const
{
    return m_cfg->readNumEntry("TabletDevice" + tabletDeviceName + axisName, defaultAxis);
}

void KisConfig::setTabletDeviceAxis(const TQString& tabletDeviceName, const TQString& axisName, TQ_INT32 axis) const
{
    m_cfg->writeEntry("TabletDevice" + tabletDeviceName + axisName, axis);
}

void KisConfig::setDockability( TQ_INT32 dockability )
{
    m_cfg->writeEntry( "palettesdockability", dockability );
}

TQ_INT32 KisConfig::dockability()
{
    return m_cfg->readNumEntry("palettesdockability", DEFAULT_DOCKABILITY);
}

TQ_INT32 KisConfig::getDefaultDockability()
{
    return DEFAULT_DOCKABILITY;
}

float KisConfig::dockerFontSize()
{
    return (float) m_cfg->readNumEntry("palettefontsize", (int)getDefaultDockerFontSize());
}

float KisConfig::getDefaultDockerFontSize()
{
    float ps = TQMIN(9, KGlobalSettings::generalFont().pointSize() * 0.8);
    if (ps < 6) ps = 6;
    return ps;
}

void KisConfig::setDockerFontSize(float size)
{
    m_cfg->writeEntry("palettefontsize", size);
}

TQ_UINT32 KisConfig::getGridMainStyle()
{
    TQ_UINT32 v = m_cfg->readNumEntry("gridmainstyle", 0);
    if (v > 2)
        v = 2;
    return v;
}

void KisConfig::setGridMainStyle(TQ_UINT32 v)
{
    m_cfg->writeEntry("gridmainstyle", v);
}

TQ_UINT32 KisConfig::getGridSubdivisionStyle()
{
    TQ_UINT32 v = m_cfg->readNumEntry("gridsubdivisionstyle", 1);
    if (v > 2) v = 2;
    return v;
}

void KisConfig::setGridSubdivisionStyle(TQ_UINT32 v)
{
    m_cfg->writeEntry("gridsubdivisionstyle", v);
}

TQColor KisConfig::getGridMainColor()
{
    return m_cfg->readColorEntry("gridmaincolor", new TQColor(99,99,99));
}

void KisConfig::setGridMainColor(TQColor v)
{
    m_cfg->writeEntry("gridmaincolor", v);
}

TQColor KisConfig::getGridSubdivisionColor()
{
    return m_cfg->readColorEntry("gridsubdivisioncolor", new TQColor(150,150,150));
}

void KisConfig::setGridSubdivisionColor(TQColor v)
{
    m_cfg->writeEntry("gridsubdivisioncolor", v);
}

TQ_UINT32 KisConfig::getGridHSpacing()
{
    TQ_INT32 v = m_cfg->readNumEntry("gridhspacing", 10);
    return (TQ_UINT32)TQMAX(1, v );
}

void KisConfig::setGridHSpacing(TQ_UINT32 v)
{
    m_cfg->writeEntry("gridhspacing", v);
}

TQ_UINT32 KisConfig::getGridVSpacing()
{
    TQ_INT32 v = m_cfg->readNumEntry("gridvspacing", 10);
    return (TQ_UINT32)TQMAX(1, v );
}

void KisConfig::setGridVSpacing(TQ_UINT32 v)
{
    m_cfg->writeEntry("gridvspacing", v);
}

TQ_UINT32 KisConfig::getGridSubdivisions()
{
    TQ_INT32 v = m_cfg->readNumEntry("gridsubsivisons", 2);
    return (TQ_UINT32)TQMAX(1, v );
}

void KisConfig::setGridSubdivisions(TQ_UINT32 v)
{
    return m_cfg->writeEntry("gridsubsivisons", v);
}

TQ_UINT32 KisConfig::getGridOffsetX()
{
    TQ_INT32 v = m_cfg->readNumEntry("gridoffsetx", 0);
    return (TQ_UINT32)TQMAX(0, v );
}

void KisConfig::setGridOffsetX(TQ_UINT32 v)
{
    m_cfg->writeEntry("gridoffsetx", v);
}

TQ_UINT32 KisConfig::getGridOffsetY()
{
    TQ_INT32 v = m_cfg->readNumEntry("gridoffsety", 0);
    return (TQ_UINT32)TQMAX(0, v );
}

void KisConfig::setGridOffsetY(TQ_UINT32 v)
{
    m_cfg->writeEntry("gridoffsety", v);
}

