/*
 *  dlg_layersize.cc - part of Chalk
 *
 *  Copyright (c) 2004 Boudewijn Rempt <boud@valdyas.org>
 *  Copyright (c) 2005 Sven Langkamp <longamp@reallygood.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <config.h>

#include <math.h>

#include <iostream>

using namespace std;

#include <tqradiobutton.h>
#include <tqcheckbox.h>
#include <tqbuttongroup.h>
#include <tqlabel.h>
#include <tqcombobox.h>

#include <klocale.h>
#include <knuminput.h>
#include <kdebug.h>

#include <kis_cmb_idlist.h>
#include <kis_filter_strategy.h>

#include "dlg_layersize.h"
#include "wdg_layersize.h"


// XXX: I'm really real bad at arithmetic, let alone math. Here
// be rounding errors. (Boudewijn)
DlgLayerSize::DlgLayerSize( TQWidget *  parent,
                const char * name)
    : super (parent, name, true, i18n("Scale Layer"), Ok | Cancel, Ok)
{
    m_lock = false;

    m_page = new WdgLayerSize(this, "layer_size");
    Q_CHECK_PTR(m_page);
    
    m_page->cmbFilterType->setIDList(KisFilterStrategyRegistry::instance()->listKeys());
    m_page->cmbFilterType->setCurrentText("Mitchell");

    setMainWidget(m_page);
    resize(m_page->sizeHint());

    unblockAll();


    connect(this, TQT_SIGNAL(okClicked()),
        this, TQT_SLOT(okClicked()));

}

DlgLayerSize::~DlgLayerSize()
{
    delete m_page;
}

void DlgLayerSize::setWidth(TQ_UINT32 w)
{
    blockAll();

    m_page->lblWidthOriginal->setNum((int)w);
    m_page->intWidth->setValue(w);
    m_oldW = w;
    m_origW = w;

    unblockAll();
}

void DlgLayerSize::setWidthPercent(TQ_UINT32 w)
{
    blockAll();

    m_page->intWidthPercent->setValue(w);
    m_oldWPercent = w;

    unblockAll();
}


void DlgLayerSize::setMaximumWidth(TQ_UINT32 w)
{
    m_page->intWidth->setMaxValue(w);
    m_maxW = w;
}

TQ_INT32 DlgLayerSize::width()
{
    //return (TQ_INT32)tqRound(m_oldW);
    return (TQ_INT32)tqRound(m_page->intWidth->value());
}

void DlgLayerSize::setHeight(TQ_UINT32 h)
{
    blockAll();

    m_page->lblHeightOriginal->setNum((int)h);
    m_page->intHeight->setValue(h);
    m_oldH = h;
    m_origH = h;

    unblockAll();
}


void DlgLayerSize::setHeightPercent(TQ_UINT32 h)
{
    blockAll();

    m_page->intHeightPercent->setValue(h);
    m_oldHPercent = h;

    unblockAll();
}

void DlgLayerSize::setMaximumHeight(TQ_UINT32 h)
{
    m_page->intHeight->setMaxValue(h);
    m_maxH = h;
}

TQ_INT32 DlgLayerSize::height()
{
    //return (TQ_INT32)tqRound(m_oldH);
    return (TQ_INT32)tqRound(m_page->intHeight->value());
}

KisFilterStrategy *DlgLayerSize::filterType()
{
    KisID filterID = m_page->cmbFilterType->currentItem();
    KisFilterStrategy *filter = KisFilterStrategyRegistry::instance()->get(filterID);
    return filter;
}


// SLOTS

void DlgLayerSize::okClicked()
{
    accept();
}

void DlgLayerSize::slotWidthPixelsChanged(int w)
{
    blockAll();

    double wPercent = double(w) * 100 / double(m_origW);

    m_page->intWidthPercent->setValue(tqRound(wPercent));

    // Set height in pixels and percent of necessary
    if (m_page->chkConstrain->isChecked()) {
        m_page->intHeightPercent->setValue(tqRound(wPercent));

        m_oldH = tqRound(m_origH * wPercent / 100);
        m_page->intHeight->setValue(tqRound(m_oldH));

    }
    m_oldW = w;

    unblockAll();
}

void DlgLayerSize::slotHeightPixelsChanged(int h)
{
    blockAll();

    double hPercent = double(h) * 100 / double(m_origH);

    m_page->intHeightPercent->setValue(tqRound(hPercent));

    // Set width in pixels and percent of necessary
    if (m_page->chkConstrain->isChecked()) {
        m_page->intWidthPercent->setValue(tqRound(hPercent));

        m_oldW = tqRound(m_origW * hPercent / 100);
        m_page->intWidth->setValue(tqRound(m_oldW));

    }
    m_oldH = h;

    unblockAll();
}

void DlgLayerSize::slotWidthPercentChanged(int w)
{
    blockAll();

    m_page->intWidth->setValue(tqRound(w * m_origW / 100));

    if (m_page->chkConstrain->isChecked()) {
        m_page->intHeightPercent->setValue(w);
        m_page->intHeight->setValue(tqRound( w * m_origH / 100));
    }

    unblockAll();
}

void DlgLayerSize::slotHeightPercentChanged(int h)
{
    blockAll();

    m_page->intHeight->setValue(tqRound(h * m_origH / 100));
    if (m_page->chkConstrain->isChecked()) {
        m_page->intWidthPercent->setValue(h);
        m_page->intWidth->setValue(tqRound( h * m_origW / 100));
    }

    unblockAll();

}


void DlgLayerSize::blockAll()
{
    // XXX: more efficient to use blockSignals?
    m_page->intWidth->disconnect();
    m_page->intHeight->disconnect();
    m_page->intWidthPercent->disconnect();
    m_page->intHeightPercent->disconnect();

}

void DlgLayerSize::unblockAll()
{
    // XXX: more efficient to use blockSignals?
    connect (m_page->intWidth, TQT_SIGNAL(valueChanged(int)),
         this, TQT_SLOT(slotWidthPixelsChanged(int)));

    connect (m_page->intHeight, TQT_SIGNAL(valueChanged(int)),
         this, TQT_SLOT(slotHeightPixelsChanged(int)));

    connect (m_page->intWidthPercent, TQT_SIGNAL(valueChanged(int)),
         this, TQT_SLOT(slotWidthPercentChanged(int)));

    connect (m_page->intHeightPercent, TQT_SIGNAL(valueChanged(int)),
         this, TQT_SLOT(slotHeightPercentChanged(int)));


}

#include "dlg_layersize.moc"
