/*
 *  Copyright (c) 2002 Patrick Julien  <freak@codepimps.org>
 *  Copyright (c) 2005 Adrian Page <adrian@pagenet.plus.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
#ifndef KIS_STRATEGY_COLORSPACE_LAB_H_
#define KIS_STRATEGY_COLORSPACE_LAB_H_

#include <tqcolor.h>

#include <klocale.h>

#include "kis_global.h"
#include "kis_integer_maths.h"
#include "kis_u16_base_colorspace.h"

class KisLabColorSpace : public KisU16BaseColorSpace {
public:
    KisLabColorSpace(KisColorSpaceFactoryRegistry * parent, KisProfile *p);
    virtual ~KisLabColorSpace();

public:

    /**
     * Return a COPY of the provided data. This method is provided to provide consistency,
     * but you really don't want to be calling it.
     */
    virtual TQ_UINT8 * toLabA16(const TQ_UINT8 * data, const TQ_UINT32 nPixels) const;

    /**
     * Return a COPY of the provided data. This method is provided for consistency,
     * but you really don't want to call it.
     */
    virtual TQ_UINT8 * fromLabA16(const TQ_UINT8 * labData, const TQ_UINT32 nPixels) const;



    virtual bool willDegrade(ColorSpaceIndependence independence)
        {
            if (independence == TO_RGBA8) 
                return true;
            else
                return false;
        };

    virtual TQValueVector<KisChannelInfo *> channels() const;
    virtual TQ_UINT32 nChannels() const;
    virtual TQ_UINT32 nColorChannels() const;
    virtual TQ_UINT32 pixelSize() const;

    virtual TQString channelValueText(const TQ_UINT8 *pixel, TQ_UINT32 channelIndex) const;
    virtual TQString normalisedChannelValueText(const TQ_UINT8 *pixel, TQ_UINT32 channelIndex) const;
    virtual void getSingleChannelPixel(TQ_UINT8 *dstPixel, const TQ_UINT8 *srcPixel, TQ_UINT32 channelIndex);

    virtual TQ_UINT8 difference(const TQ_UINT8 *src1, const TQ_UINT8 *src2);
    virtual void mixColors(const TQ_UINT8 **colors, const TQ_UINT8 *weights, TQ_UINT32 nColors, TQ_UINT8 *dst) const;
    virtual void invertColor(TQ_UINT8 * src, TQ_INT32 nPixels);
    virtual void convolveColors(TQ_UINT8** colors, TQ_INT32 * kernelValues, KisChannelInfo::enumChannelFlags channelFlags, TQ_UINT8 *dst, TQ_INT32 factor, TQ_INT32 offset, TQ_INT32 nColors) const;
    
    virtual void darken(const TQ_UINT8 * src, TQ_UINT8 * dst, TQ_INT32 shade, bool compensate, double compensation, TQ_INT32 nPixels) const;

    virtual KisCompositeOpList userVisiblecompositeOps() const;

protected:

    virtual void bitBlt(TQ_UINT8 *dst,
                TQ_INT32 dstRowStride,
                const TQ_UINT8 *src,
                TQ_INT32 srcRowStride,
                const TQ_UINT8 *srcAlphaMask,
                TQ_INT32 maskRowStride,
                TQ_UINT8 opacity,
                TQ_INT32 rows,
                TQ_INT32 cols,
                const KisCompositeOp& op);

    void compositeOver(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, TQ_UINT16 opacity);
/*
    void compositeMultiply(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, TQ_UINT16 opacity);
    void compositeDivide(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, TQ_UINT16 opacity);
    void compositeScreen(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, TQ_UINT16 opacity);
    void compositeOverlay(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, TQ_UINT16 opacity);
    void compositeDodge(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, TQ_UINT16 opacity);
    void compositeBurn(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, TQ_UINT16 opacity);
    void compositeDarken(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, TQ_UINT16 opacity);
    void compositeLighten(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, TQ_UINT16 opacity);
    void compositeHue(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, TQ_UINT16 opacity);
    void compositeSaturation(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, TQ_UINT16 opacity);
    void compositeValue(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, TQ_UINT16 opacity);
    void compositeColor(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, TQ_UINT16 opacity);
*/
    void compositeErase(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, TQ_UINT16 opacity);

private:
    struct Pixel {
        TQ_UINT16 lightness;
        TQ_UINT16 a;
        TQ_UINT16 b;
        TQ_UINT16 alpha;
    };
    static const TQ_UINT16 U16_OPACITY_OPAQUE = UINT16_MAX;
    static const TQ_UINT16 U16_OPACITY_TRANSPARENT = UINT16_MIN;

    static const TQ_UINT32 NUM_CHANNELS = 4;
    static const TQ_UINT32 NUM_COLOR_CHANNELS = 3;

    static const TQ_UINT32 CHANNEL_L = 0;
    static const TQ_UINT32 CHANNEL_A = 1;
    static const TQ_UINT32 CHANNEL_B = 2;
    static const TQ_UINT32 CHANNEL_ALPHA = 3;

    static const TQ_UINT32 MAX_CHANNEL_L = 0xff00;
    static const TQ_UINT32 MAX_CHANNEL_AB = 0xffff;
    static const TQ_UINT32 CHANNEL_AB_ZERO_OFFSET = 0x8000;

    friend class KisLabColorSpaceTester;
};

class KisLabColorSpaceFactory : public KisColorSpaceFactory
{
public:
    /**
     * Chalk definition for use in .kra files and internally: unchanging name +
     * i18n'able description.
     */
    virtual KisID id() const { return KisID("LABA", i18n("L*a*b* (16-bit integer/channel)")); };

    /**
     * lcms colorspace type definition.
     */
    virtual TQ_UINT32 colorSpaceType() { return (COLORSPACE_SH(PT_Lab)|CHANNELS_SH(3)|BYTES_SH(2)|EXTRA_SH(1)); };

    virtual icColorSpaceSignature colorSpaceSignature() { return icSigLabData; };
    
    virtual KisColorSpace *createColorSpace(KisColorSpaceFactoryRegistry * parent, KisProfile *p) { return new KisLabColorSpace(parent, p); };

    virtual TQString defaultProfile() { return "Lab built-in - (lcms internal)"; };
};

#endif // KIS_STRATEGY_COLORSPACE_LAB_H_
