/* ============================================================
 * Author: Gilles Caulier <caulier dot gilles at gmail dot com>
 *         from digiKam project.
 * Date  : 2003-10-01
 * Description : a kipi plugin to export image collections
 *               on CD/DVD.
 *
 * Copyright 2003-2005 by Gilles Caulier <caulier dot gilles at gmail dot com>
 * Copyright 2003-2004 by Gregory Kokanosky
 *                        <gregory dot kokanosky at free.fr>
 *                        for HTML interface navigation mode
 * Copyright 2005      by Owen Hirst <n8rider at sbcglobal.net>
 *                        about bugfix.
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// TQt includes

#include <tqcheckbox.h>
#include <tqcombobox.h>
#include <tqfileinfo.h>
#include <tqgroupbox.h>
#include <tqheader.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqlineedit.h>
#include <tqlistview.h>
#include <tqprogressdialog.h>
#include <tqpushbutton.h>
#include <tqspinbox.h>
#include <tqvbox.h>
#include <tqwhatsthis.h>

// KDE includes

#include <kaboutdata.h>
#include <kapplication.h>
#include <kbuttonbox.h>
#include <kcolorbutton.h>
#include <kconfig.h>
#include <kdebug.h>
#include <kdirsize.h>
#include <kfontdialog.h>
#include <kglobalsettings.h>
#include <khelpmenu.h>
#include <kiconloader.h>
#include <klineedit.h>
#include <klistview.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <knuminput.h>
#include <kpopupmenu.h>
#include <ksqueezedtextlabel.h>
#include <kstandarddirs.h>
#include <kurl.h>
#include <kurlrequester.h>

// KIPI includes

#include <libkipi/imagecollection.h>
#include <libkipi/imagecollectionselector.h>
#include <libkipi/imageinfo.h>

// Local includes

#include "cdarchivingdialog.h"
#include "kpaboutdata.h"
#include "pluginsversion.h"


namespace KIPICDArchivingPlugin
{

KIO::filesize_t TargetMediaSize;

CDArchivingDialog::CDArchivingDialog( KIPI::Interface* interface, TQWidget *parent)
        : KDialogBase( IconList, i18n("Configure Archive to CD"), Help|Ok|Cancel, Ok,
                       parent, "CDArchivingDialog", true, false ), m_interface( interface )
{
    setCaption(i18n("Create CD/DVD Archive"));
    setupSelection();
    setupLookPage();
    setupCDInfos();
    setupBurning();
    page_setupSelection->setFocus();
    resize(650, 450);

    // About data and help button.

    m_about = new KIPIPlugins::KPAboutData(I18N_NOOP("CD/DVD Archiving"),
                                           0,
                                           KAboutData::License_GPL,
                                           I18N_NOOP("An Album CD/DVD Archiving Kipi plugin.\n"
                                                     "This plugin use K3b CD/DVD burning software available at\n"
                                                     "http://www.k3b.org"),
                                           "(c) 2003-2005, Gilles Caulier");

    m_about->addAuthor("Gilles Caulier", I18N_NOOP("Author"),
                       "caulier dot gilles at gmail dot com");

    m_about->addAuthor("Angelo Naselli", I18N_NOOP("Contributor"),
                       "anaselli at linux.it");

    m_about->addAuthor("Gregory Kokanosky", I18N_NOOP("Image navigation mode patches"),
                       "gregory dot kokanosky at free.fr");

    m_about->addAuthor("Owen Hirst", I18N_NOOP("Bugfix"),
                       "n8rider at sbcglobal.net");

    m_helpButton = actionButton( Help );
    KHelpMenu* helpMenu = new KHelpMenu(this, m_about, false);
    helpMenu->menu()->removeItemAt(0);
    helpMenu->menu()->insertItem(i18n("Plugin Handbook"), this, TQT_SLOT(slotHelp()), 0, -1, 0);
    m_helpButton->setPopup( helpMenu->menu() );
}

CDArchivingDialog::~CDArchivingDialog()
{
    delete m_about;
}

void CDArchivingDialog::slotHelp()
{
    KApplication::kApplication()->invokeHelp("cdarchiving", "kipi-plugins");
}

void CDArchivingDialog::setupSelection(void)
{
    page_setupSelection = addPage(i18n("Selection"), i18n("Album Selection"),
                                  BarIcon("folder_image", KIcon::SizeMedium));

    TQVBoxLayout *layout = new TQVBoxLayout(page_setupSelection, 0, spacingHint() );
    m_imageCollectionSelector = new KIPI::ImageCollectionSelector(page_setupSelection, m_interface);
    layout->addWidget(m_imageCollectionSelector);

    //---------------------------------------------

    TQGroupBox * groupBox3 = new TQGroupBox( 2, Qt::Horizontal,
                                           i18n("Target Media Information"),
                                           page_setupSelection );
    groupBox3->layout()->setSpacing( 6 );
    groupBox3->layout()->setMargin( 11 );
    TQWhatsThis::add( groupBox3, i18n("<p>Information about the backup medium.") );

    m_mediaSize = new TQLabel( groupBox3 );
    m_mediaSize->setAlignment( int( TQLabel::WordBreak | TQLabel::AlignVCenter ) );

    m_mediaFormat = new TQComboBox(false, groupBox3);
    m_mediaFormat->insertItem(i18n("CD (650Mb)"));
    m_mediaFormat->insertItem(i18n("CD (700Mb)"));
    m_mediaFormat->insertItem(i18n("CD (880Mb)"));
    m_mediaFormat->insertItem(i18n("DVD (4,7Gb)"));
    m_mediaFormat->setCurrentText (i18n("CD (650Mb)"));
    mediaFormatActived(m_mediaFormat->currentText());
    TQWhatsThis::add( m_mediaFormat, i18n("<p>Select here the backup media format."));

    layout->addWidget( groupBox3 );

    //---------------------------------------------

    connect( m_mediaFormat, TQT_SIGNAL( highlighted( const TQString & ) ),
             this, TQT_SLOT( mediaFormatActived( const TQString & ) ) );

    connect( m_imageCollectionSelector, TQT_SIGNAL( selectionChanged() ),
             this, TQT_SLOT( slotAlbumSelected() ) );
}

void CDArchivingDialog::setupLookPage(void)
{
    TQString whatsThis;
    page_setupLook = addPage( i18n("HTML Interface"), i18n("HTML Interface Look"),
                              BarIcon("html", KIcon::SizeMedium ) );

    TQVBoxLayout *vlay = new TQVBoxLayout( page_setupLook, 0, spacingHint() );

    //---------------------------------------------

    m_useHTMLInterface = new TQCheckBox( i18n("Build CD HTML interface"), page_setupLook);
    m_useHTMLInterface->setChecked( true );
    vlay->addWidget( m_useHTMLInterface );
    TQWhatsThis::add( m_useHTMLInterface,
                     i18n("<p>This option adds a HTML interface to browse the CD's contents.") );

    //---------------------------------------------

    m_useAutoRunWin32 = new TQCheckBox( i18n("Add \"autorun\" functionality"), page_setupLook);
    m_useAutoRunWin32->setChecked( true );
    vlay->addWidget( m_useAutoRunWin32 );
    TQWhatsThis::add( m_useAutoRunWin32,
                     i18n("<p>This option adds MS Windows(tm) autorunning capability to the CD.") );

    //---------------------------------------------

    m_labelTitle = new TQLabel( i18n("Archive title:"), page_setupLook);
    vlay->addWidget( m_labelTitle );

    m_title = new TQLineEdit(i18n("Album Archiving"), page_setupLook);
    vlay->addWidget( m_title );
    m_labelTitle->setBuddy(m_title);
    TQWhatsThis::add( m_title, i18n("<p>Enter here the title of the CD archive.") );

    //---------------------------------------------

    m_imagesPerRow = new KIntNumInput(4, page_setupLook);
    m_imagesPerRow->setRange(1, 8, 1, true );
    m_imagesPerRow->setLabel( i18n("I&mages per row:") );
    TQWhatsThis::add( m_imagesPerRow, i18n("<p>Enter here the number of images per row on the album page. "
                                          "A good value is '4'.") );
    vlay->addWidget( m_imagesPerRow );

    TQGridLayout *grid = new TQGridLayout( 2, 2 );
    vlay->addLayout( grid );

    //---------------------------------------------

    m_thumbnailsSize = new KIntNumInput(140, page_setupLook);
    m_thumbnailsSize->setRange(10, 1000, 1, true );
    m_thumbnailsSize->setLabel( i18n("Thumbnail size:") );
    vlay->addWidget( m_thumbnailsSize );
    TQWhatsThis::add( m_thumbnailsSize, i18n("<p>The new size of thumbnails in pixels") );

    //---------------------------------------------

    TQHBoxLayout *hlay3 = new TQHBoxLayout( spacingHint() );
    vlay->addLayout( hlay3 );
    m_imageFormat = new TQComboBox(false, page_setupLook);
    m_imageFormat->insertItem("JPEG");
    m_imageFormat->insertItem("PNG");
    m_imageFormat->setCurrentText ("JPEG");
    whatsThis = i18n("<p>Select here the image file format for thumbnails.<p>");
    whatsThis = whatsThis + i18n("<b>JPEG</b>: The Joint Photographic Experts Group's file format is a "
                                 "good Web file format but it uses lossy data compression.<p>"
                                 "<b>PNG</b>: the Portable Network Graphics format is an extensible file format for "
                                 "the lossless, portable, well-compressed storage of raster images. PNG provides a "
                                 "patent-free replacement for GIF and can also replace many common uses of TIFF. "
                                 "PNG is designed to work well in online viewing applications, such as the World "
                                 "Wide Web, so it is fully streamable with a progressive display option. Also, "
                                 "PNG can store gamma and chromaticity data for improved color matching on "
                                 "heterogeneous platforms.");
    TQWhatsThis::add( m_imageFormat, whatsThis );

    m_labelThumbsFileFormat = new TQLabel( i18n("Thumbnail file format:"), page_setupLook);
    hlay3->addWidget( m_labelThumbsFileFormat );
    m_labelThumbsFileFormat->setBuddy( m_imageFormat );
    hlay3->addStretch( 1 );
    hlay3->addWidget( m_imageFormat );

    //---------------------------------------------

    TQHBoxLayout *hlay11  = new TQHBoxLayout( );
    vlay->addLayout( hlay11 );

    m_fontName = new TQComboBox( false, page_setupLook );
    TQStringList standardFonts;
    KFontChooser::getFontList(standardFonts, 0);
    m_fontName->insertStringList( standardFonts );
    m_fontName->setCurrentText( KGlobalSettings::generalFont().family());
    TQWhatsThis::add( m_fontName, i18n("<p>Select here the font name used for the pages.") );

    m_labelFontName = new TQLabel( i18n("Fon&t name:"), page_setupLook );
    m_labelFontName->setBuddy( m_fontName );
    hlay11->addWidget( m_labelFontName );
    hlay11->addStretch( 1 );
    hlay11->addWidget( m_fontName );

    //---------------------------------------------

    TQHBoxLayout *hlay12  = new TQHBoxLayout( );
    vlay->addLayout( hlay12 );

    m_fontSize = new TQSpinBox( 6, 15, 1, page_setupLook );
    m_fontSize->setValue( 14 );
    TQWhatsThis::add( m_fontSize, i18n("<p>Select here the font size used for the pages.") );

    m_labelFontSize = new TQLabel( i18n("Font si&ze:"), page_setupLook );
    m_labelFontSize->setBuddy( m_fontSize );
    hlay12->addWidget( m_labelFontSize );
    hlay12->addStretch( 1 );
    hlay12->addWidget( m_fontSize );

    //---------------------------------------------

    TQHBoxLayout *hlay1  = new TQHBoxLayout( spacingHint() );
    vlay->addLayout( hlay1 );

    m_foregroundColor = new KColorButton(page_setupLook);
    m_foregroundColor->setColor(TQColor("#d0ffd0"));
    TQWhatsThis::add( m_foregroundColor, i18n("<p>Select here the foreground color used for the pages.") );

    m_labelForegroundColor = new TQLabel( i18n("&Foreground color:"), page_setupLook);
    m_labelForegroundColor->setBuddy( m_foregroundColor );
    hlay1->addWidget( m_labelForegroundColor );
    hlay1->addStretch( 1 );
    hlay1->addWidget(m_foregroundColor);

    //---------------------------------------------

    TQHBoxLayout *hlay2 = new TQHBoxLayout( spacingHint() );
    vlay->addLayout( hlay2 );

    m_backgroundColor = new KColorButton(page_setupLook);
    m_backgroundColor->setColor(TQColor("#333333"));
    TQWhatsThis::add( m_backgroundColor, i18n("<p>Select here the background color used for the pages.") );

    m_labelBackgroundColor = new TQLabel( i18n("&Background color:"), page_setupLook);
    hlay2->addWidget( m_labelBackgroundColor );
    m_labelBackgroundColor->setBuddy( m_backgroundColor );
    hlay2->addStretch( 1 );
    hlay2->addWidget(m_backgroundColor);

    //---------------------------------------------

    TQHBoxLayout *hlay13  = new TQHBoxLayout( );
    vlay->addLayout( hlay13 );

    m_bordersImagesSize = new TQSpinBox( 1, 20, 1, page_setupLook );
    m_bordersImagesSize->setValue( 1 );
    TQWhatsThis::add( m_bordersImagesSize, i18n("<p>Select here the image border's size in pixels.") );

    m_labelImageBorderSize = new TQLabel( i18n("Image border s&ize:"), page_setupLook );
    m_labelImageBorderSize->setBuddy( m_bordersImagesSize );
    hlay13->addWidget( m_labelImageBorderSize );
    hlay13->addStretch( 1 );
    hlay13->addWidget( m_bordersImagesSize );

    //---------------------------------------------

    TQHBoxLayout *hlay4 = new TQHBoxLayout( spacingHint() );
    vlay->addLayout( hlay4 );

    m_bordersImagesColor = new KColorButton(page_setupLook);
    m_bordersImagesColor->setColor(TQColor("#d0ffd0"));
    TQWhatsThis::add( m_bordersImagesColor, i18n("<p>Select here the color used "
                     "for the image borders.") );

    m_labelImageBorderSizeColor = new TQLabel( i18n("Image bo&rder color:"), page_setupLook);
    hlay4->addWidget( m_labelImageBorderSizeColor );
    m_labelImageBorderSizeColor->setBuddy( m_bordersImagesColor );
    hlay4->addStretch( 1 );
    hlay4->addWidget(m_bordersImagesColor);

    //---------------------------------------------

    vlay->addStretch(1);

    connect(m_useHTMLInterface, TQT_SIGNAL(toggled(bool)),
            m_useAutoRunWin32, TQT_SLOT(setEnabled(bool)));

    connect(m_useHTMLInterface, TQT_SIGNAL(toggled(bool)),
            m_labelTitle, TQT_SLOT(setEnabled(bool)));

    connect(m_useHTMLInterface, TQT_SIGNAL(toggled(bool)),
            m_labelThumbsFileFormat, TQT_SLOT(setEnabled(bool)));

    connect(m_useHTMLInterface, TQT_SIGNAL(toggled(bool)),
            m_labelFontName, TQT_SLOT(setEnabled(bool)));

    connect(m_useHTMLInterface, TQT_SIGNAL(toggled(bool)),
            m_labelFontSize, TQT_SLOT(setEnabled(bool)));

    connect(m_useHTMLInterface, TQT_SIGNAL(toggled(bool)),
            m_labelForegroundColor, TQT_SLOT(setEnabled(bool)));

    connect(m_useHTMLInterface, TQT_SIGNAL(toggled(bool)),
            m_labelBackgroundColor, TQT_SLOT(setEnabled(bool)));

    connect(m_useHTMLInterface, TQT_SIGNAL(toggled(bool)),
            m_title, TQT_SLOT(setEnabled(bool)));

    connect(m_useHTMLInterface, TQT_SIGNAL(toggled(bool)),
            m_imagesPerRow, TQT_SLOT(setEnabled(bool)));

    connect(m_useHTMLInterface, TQT_SIGNAL(toggled(bool)),
            m_fontSize, TQT_SLOT(setEnabled(bool)));

    connect(m_useHTMLInterface, TQT_SIGNAL(toggled(bool)),
            m_fontName, TQT_SLOT(setEnabled(bool)));

    connect(m_useHTMLInterface, TQT_SIGNAL(toggled(bool)),
            m_imageFormat, TQT_SLOT(setEnabled(bool)));

    connect(m_useHTMLInterface, TQT_SIGNAL(toggled(bool)),
            m_thumbnailsSize, TQT_SLOT(setEnabled(bool)));

    connect(m_useHTMLInterface, TQT_SIGNAL(toggled(bool)),
            m_foregroundColor, TQT_SLOT(setEnabled(bool)));

    connect(m_useHTMLInterface, TQT_SIGNAL(toggled(bool)),
            m_backgroundColor, TQT_SLOT(setEnabled(bool)));

    connect(m_useHTMLInterface, TQT_SIGNAL(toggled(bool)),
            m_bordersImagesSize, TQT_SLOT(setEnabled(bool)));

    connect(m_useHTMLInterface, TQT_SIGNAL(toggled(bool)),
            m_labelImageBorderSize, TQT_SLOT(setEnabled(bool)));

    connect(m_useHTMLInterface, TQT_SIGNAL(toggled(bool)),
            m_bordersImagesColor, TQT_SLOT(setEnabled(bool)));

    connect(m_useHTMLInterface, TQT_SIGNAL(toggled(bool)),
            m_labelImageBorderSizeColor, TQT_SLOT(setEnabled(bool)));

}

void CDArchivingDialog::setupCDInfos(void)
{
    page_CDInfos = addPage( i18n("Volume Descriptor"), i18n("Media Volume Descriptor"),
                            BarIcon("cd", KIcon::SizeMedium ) );

    TQVBoxLayout *vlay = new TQVBoxLayout( page_CDInfos, 0, spacingHint() );

    TQLabel *label;

    //---------------------------------------------

    label = new TQLabel( i18n("Volume name:"), page_CDInfos);
    vlay->addWidget( label );

    m_volume_id = new TQLineEdit(i18n("CD Album"), page_CDInfos);
    vlay->addWidget( m_volume_id );
    m_labelTitle->setBuddy(m_volume_id);
    m_volume_id->setMaxLength(32);
    TQWhatsThis::add( m_volume_id, i18n("<p>Enter here the media volume name (32 characters max.)") );

    //---------------------------------------------

    label = new TQLabel( i18n("Volume set name:"), page_CDInfos);
    vlay->addWidget( label );

    m_volume_set_id = new TQLineEdit(i18n("Album CD archive"), page_CDInfos);
    vlay->addWidget( m_volume_set_id );
    m_labelTitle->setBuddy(m_volume_set_id);
    m_volume_set_id->setMaxLength(128);
    TQWhatsThis::add( m_volume_set_id,
                     i18n("<p>Enter here the media volume global name (128 characters max.)") );

    //---------------------------------------------

    label = new TQLabel( i18n("System:"), page_CDInfos);
    vlay->addWidget( label );

    m_system_id = new TQLineEdit(i18n("LINUX"), page_CDInfos);
    vlay->addWidget( m_system_id );
    m_labelTitle->setBuddy(m_system_id);
    m_system_id->setMaxLength(32);
    TQWhatsThis::add( m_system_id,
                     i18n("<p>Enter here the media burning system name (32 characters max.)") );

    //---------------------------------------------

    label = new TQLabel( i18n("Application:"), page_CDInfos);
    vlay->addWidget( label );

    m_application_id = new TQLineEdit(i18n("K3b CD-DVD Burning application"), page_CDInfos);
    vlay->addWidget( m_application_id );
    m_labelTitle->setBuddy(m_application_id);
    m_application_id->setMaxLength(128);
    TQWhatsThis::add( m_application_id,
                     i18n("<p>Enter here the media burning application name (128 characters max.).") );


    //---------------------------------------------

    label = new TQLabel( i18n("Publisher:"), page_CDInfos);
    vlay->addWidget( label );

    m_publisher = new TQLineEdit(i18n("KIPI [KDE Images Program Interface]"), page_CDInfos);
    vlay->addWidget( m_publisher );
    m_labelTitle->setBuddy(m_publisher);
    m_publisher->setMaxLength(128);
    TQWhatsThis::add( m_publisher,
                     i18n("<p>Enter here the media publisher name (128 characters max.).") );

    //---------------------------------------------

    label = new TQLabel( i18n("Preparer:"), page_CDInfos);
    vlay->addWidget( label );

    m_preparer = new TQLineEdit(i18n("KIPI CD Archiving plugin"), page_CDInfos);
    vlay->addWidget( m_preparer );
    m_labelTitle->setBuddy(m_preparer);
    m_preparer->setMaxLength(128);
    TQWhatsThis::add( m_preparer, i18n("<p>Enter here the media preparer name (128 characters max.).") );

    vlay->addStretch(1);
}

void CDArchivingDialog::setupBurning(void)
{
    page_burning = addPage( i18n("Media Burning"),
                            i18n("CD/DVD Burning Setup"),
                            BarIcon("cdwriter_unmount", KIcon::SizeMedium ) );

    TQVBoxLayout *vlay = new TQVBoxLayout( page_burning, 0, spacingHint() );

    TQLabel *label;

    //---------------------------------------------

    label = new TQLabel(i18n("&K3b binary path:"), page_burning);
    vlay->addWidget( label );

    m_K3bBinPath = new KURLRequester( "k3b", page_burning);
    label->setBuddy( m_K3bBinPath );
    vlay->addWidget(m_K3bBinPath);

    connect( m_K3bBinPath, TQT_SIGNAL(textChanged(const TQString&)),
             this, TQT_SLOT(UrlChanged(const TQString&)));

    TQWhatsThis::add( m_K3bBinPath, i18n("<p>The path name to the K3b binary program.") );

    //---------------------------------------------

    label = new TQLabel( i18n("Application parameters:"), page_burning);
    vlay->addWidget( label );

    m_K3bParameters = new TQLineEdit("--nofork", page_burning);
    vlay->addWidget( m_K3bParameters );
    m_labelTitle->setBuddy(m_K3bParameters);
    m_K3bParameters->setMaxLength(128);
    TQWhatsThis::add( m_K3bParameters,
                     i18n("<p>Enter parameters which will be used when starting the "
                          "burning application. Newer versions of K3b might need "
                          "--nofork, older versions might not need it. "
                          "(128 characters max.).") );

    //---------------------------------------------

    TQGroupBox * groupBoxAdvancedOptions = new TQGroupBox( i18n("Advanced Burning Options"), page_burning );
    groupBoxAdvancedOptions->setColumnLayout(0, Qt::Vertical );
    groupBoxAdvancedOptions->layout()->setSpacing( 6 );
    groupBoxAdvancedOptions->layout()->setMargin( 11 );

    TQVBoxLayout * groupBoxAOLayout = new TQVBoxLayout( groupBoxAdvancedOptions->layout() );
    groupBoxAOLayout->setAlignment( TQt::AlignTop );

    m_burnOnTheFly = new TQCheckBox( i18n("Media burning On-The-Fly"), groupBoxAdvancedOptions);
    m_burnOnTheFly->setChecked( false );
    TQWhatsThis::add( m_burnOnTheFly, i18n("<p>This option uses the \"On-The-Fly\" "
                                          "media burning capability; this does not use a media image.") );
    groupBoxAOLayout->addWidget( m_burnOnTheFly );

    m_checkCDBurn = new TQCheckBox( i18n("Check media"), groupBoxAdvancedOptions);
    m_checkCDBurn->setChecked( false );
    TQWhatsThis::add( m_checkCDBurn, i18n("<p>This option verifies the media after the burning process. "
                                         "You must use K3b release >= 0.10.0") );
    groupBoxAOLayout->addWidget( m_checkCDBurn );

    m_startBurningProcess = new TQCheckBox( i18n("Start burning process automatically"), groupBoxAdvancedOptions);
    m_startBurningProcess->setChecked( false );
    m_startBurningProcess->hide();
    TQWhatsThis::add( m_startBurningProcess, i18n("<p>This option start automatically the burning process "
                     "when K3b is loaded.") );
    groupBoxAOLayout->addWidget( m_startBurningProcess );

    vlay->addWidget( groupBoxAdvancedOptions );

    vlay->addStretch(1);
}

void CDArchivingDialog::slotAlbumSelected()
{
    TQValueList<KIPI::ImageCollection> ListAlbums(m_imageCollectionSelector->selectedImageCollections());
    double size = 0;

    for ( TQValueList<KIPI::ImageCollection>::Iterator it = ListAlbums.begin(); it != ListAlbums.end(); ++it )
    {
        KURL::List images = (*it).images();

        for ( KURL::List::Iterator urlIt = images.begin() ; urlIt != images.end() ; ++urlIt )
        {
            KIPI::ImageInfo info = m_interface->info( *urlIt );
            size += info.size();
        }
    }

    TargetMediaSize = (int)(size/1024.0);

    ShowMediaCapacity();
}


void CDArchivingDialog::mediaFormatActived (const TQString & item )
{
    TQString Color;

    if (item == i18n("CD (650Mb)"))
        MaxMediaSize = 665600;

    if (item == i18n("CD (700Mb)"))
        MaxMediaSize = 716800;

    if (item == i18n("CD (880Mb)"))
        MaxMediaSize = 901120;

    if (item == i18n("DVD (4,7Gb)"))
        MaxMediaSize = 4928307;

    ShowMediaCapacity();
}

void CDArchivingDialog::ShowMediaCapacity(void)
{
    TQString Color = "<font color=\"blue\">";;

    if (TargetMediaSize >= MaxMediaSize - (MaxMediaSize*0.1))
        Color = "<font color=\"orange\">";

    if (TargetMediaSize >= MaxMediaSize)
        Color = "<font color=\"red\">";

    m_mediaSize->setText( i18n("Total size: ") + Color +
                          i18n("<b>%1</b></font> / <b>%2</b>").arg(KIO::convertSizeFromKB(TargetMediaSize))
                          .arg(KIO::convertSizeFromKB (MaxMediaSize)) );
}

void CDArchivingDialog::slotOk()
{
    m_selectedAlbums = m_imageCollectionSelector->selectedImageCollections();

    if (m_selectedAlbums.size() == 0)
    {
        KMessageBox::sorry(this, i18n("You must selected at least one Album to archive."));
        return;
    }

    TQFile fileK3b(getK3bBinPathName());

    if ( !KStandardDirs::findExe( getK3bBinPathName() ))
    {
        KMessageBox::sorry(this, i18n("K3b binary path is not valid. Please check it."));
        return;
    }

    if (TargetMediaSize >= MaxMediaSize)
    {
        KMessageBox::sorry(this, i18n("Target media size is too big. Please change your album selection."));
        return;
    }

    accept();
}

void CDArchivingDialog::UrlChanged(const TQString &url )
{
    enableButtonOK( !url.isEmpty());
}

TQString CDArchivingDialog::getK3bBinPathName() const
{
    return m_K3bBinPath->url();
}

TQString CDArchivingDialog::getK3bParameters() const
{
    return m_K3bParameters->text();
}

void CDArchivingDialog::setK3bBinPathName(const TQString &Value)
{
    m_K3bBinPath->setURL( Value );
}

void CDArchivingDialog::setK3bParameters(const TQString &Value)
{
    m_K3bParameters->setText( Value );
}

int CDArchivingDialog::getImagesPerRow() const
{
    return m_imagesPerRow->value();
}

void CDArchivingDialog::setImagesPerRow(int Value)
{
    m_imagesPerRow->setValue(Value);
}

int CDArchivingDialog::getThumbnailsSize() const
{
    return m_thumbnailsSize->value();
}

void CDArchivingDialog::setThumbnailsSize(int Value)
{
    m_thumbnailsSize->setValue( Value );
}

const TQString CDArchivingDialog::getFontName() const
{
    return m_fontName->currentText();
}

void CDArchivingDialog::setFontName(TQString Value)
{
    m_fontName->setCurrentText (Value);
}

const TQString CDArchivingDialog::getFontSize() const
{
    return m_fontSize->text();
}

void CDArchivingDialog::setFontSize(int Value)
{
    m_fontSize->setValue( Value );
}

const TQColor CDArchivingDialog::getBackgroundColor() const
{
    return m_backgroundColor->color();
}

void CDArchivingDialog::setBackgroundColor(TQColor Value)
{
    m_backgroundColor->setColor( Value );
}

const TQColor CDArchivingDialog::getForegroundColor() const
{
    return m_foregroundColor->color();
}

void CDArchivingDialog::setForegroundColor(TQColor Value)
{
    m_foregroundColor->setColor( Value );
}

const TQString CDArchivingDialog::getImageFormat() const
{
    return m_imageFormat->currentText();
}

void CDArchivingDialog::setImageFormat(TQString Value)
{
    return m_imageFormat->setCurrentText( Value );
}

const TQString CDArchivingDialog::getMainTitle() const
{
    return m_title->text();
}

void CDArchivingDialog::setMainTitle(TQString Value)
{
    return m_title->setText( Value );
}

bool CDArchivingDialog::getUseHTMLInterface() const
{
    return m_useHTMLInterface->isChecked();
}

void CDArchivingDialog::setUseHTMLInterface(bool Value)
{
    m_useHTMLInterface->setChecked(Value);
}

bool CDArchivingDialog::getUseAutoRunWin32() const
{
    return m_useAutoRunWin32->isChecked();
}

void CDArchivingDialog::setUseAutoRunWin32(bool Value)
{
    m_useAutoRunWin32->setChecked(Value);
}

const TQString CDArchivingDialog::getVolumeID() const
{
    return m_volume_id->text();
}

void CDArchivingDialog::setVolumeID(TQString Value)
{
    return m_volume_id->setText( Value );
}

const TQString CDArchivingDialog::getVolumeSetID() const
{
    return m_volume_set_id->text();
}

void CDArchivingDialog::setVolumeSetID(TQString Value)
{
    return m_volume_set_id->setText( Value );
}

const TQString CDArchivingDialog::getSystemID() const
{
    return m_system_id->text();
}

void CDArchivingDialog::setSystemID(TQString Value)
{
    return m_system_id->setText( Value );
}

const TQString CDArchivingDialog::getApplicationID() const
{
    return m_application_id->text();
}

void CDArchivingDialog::setApplicationID(TQString Value)
{
    return m_application_id->setText( Value );
}

const TQString CDArchivingDialog::getPublisher() const
{
    return m_publisher->text();
}

void CDArchivingDialog::setPublisher(TQString Value)
{
    return m_publisher->setText( Value );
}

const TQString CDArchivingDialog::getPreparer() const
{
    return m_preparer->text();
}

void CDArchivingDialog::setPreparer(TQString Value)
{
    return m_preparer->setText( Value );
}

const TQString CDArchivingDialog::getMediaFormat() const
{
    return m_mediaFormat->currentText();
}

void CDArchivingDialog::setMediaFormat(TQString Value)
{
    return m_mediaFormat->setCurrentText( Value );
}

bool CDArchivingDialog::getUseOnTheFly() const
{
    return m_burnOnTheFly->isChecked();
}

void CDArchivingDialog::setUseUseOnTheFly(bool Value)
{
    m_burnOnTheFly->setChecked(Value);
}

bool CDArchivingDialog::getUseCheckCD() const
{
    return m_checkCDBurn->isChecked();
}

void CDArchivingDialog::setUseCheckCD(bool Value)
{
    m_checkCDBurn->setChecked(Value);
}

bool CDArchivingDialog::getUseStartBurningProcess() const
{
    return m_startBurningProcess->isChecked();
}

void CDArchivingDialog::setUseStartBurningProcess(bool Value)
{
    m_startBurningProcess->setChecked(Value);
}

const TQColor CDArchivingDialog::getBordersImagesColor() const
{
    return m_bordersImagesColor->color();
}

void CDArchivingDialog::setBordersImagesColor(TQColor Value)
{
    m_bordersImagesColor->setColor( Value );
}

const TQString CDArchivingDialog::getBordersImagesSize() const
{
    return m_bordersImagesSize->text();
}

void CDArchivingDialog::setBordersImagesSize(int Value)
{
    m_bordersImagesSize->setValue( Value );
}

}  // NameSpace KIPICDArchivingPlugin

#include "cdarchivingdialog.moc"
