/***************************************************************************
 *   Copyright (C) 2006 by Ken Werner                                      *
 *   ken.werner@web.de                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#include "source.h"
#include <math.h>
#include <tqlayout.h>
#include <tqlineedit.h>
#include <tqcheckbox.h>
#include <tqlabel.h>
#include <klocale.h>
#include <kglobal.h>

//#include "kdebug.h"

Source::Source(TQWidget* inParent):	
		mID(""),
		mPosition(0),
		mName(""),
		mDescription(""),
		mEnabled(true),
		mMaybeEnabled(true),
		mShowOnApplet(true),
		mMaybeShowOnApplet(true),
		mShowName(true),
		mToolTipEnabled(true),
		mSourcePrefs(0) {
}

Source::~Source(){
}

const TQString& Source::getID() const{
	return mID;
}


int Source::getPosition() const{
	return mPosition;
}

void Source::setPosition(int inPosition, KConfig* inKConfig){
	mPosition = inPosition;
	inKConfig->writeEntry(mID + "_position", mPosition);
}

const TQString& Source::getName() const{
	return mName;
}

const TQString& Source::getDescription() const{
	return mDescription;
}

bool Source::isMetric() const{
	return mIsMetric;
}

bool Source::isEnabled() const{
	return mEnabled;
}

bool Source::showOnApplet() const{
	return mShowOnApplet;
}

bool Source::showName() const{
	return mShowName;
}

bool Source::isToolTipEnabled() const{
	return mToolTipEnabled;
}

void Source::setMaybeEnabled(bool inMaybeEnabled){
	if(inMaybeEnabled != mMaybeEnabled){
		mMaybeEnabled = inMaybeEnabled;
		// disable/enable some widgets if source is disabled/enabled
		setPrefsWidgetsEnabled(mMaybeEnabled, mSourcePrefs->taskbarCheckBox->isChecked());
	}
}

TQWidget* Source::createPrefs(TQWidget* inParent){
	if(!mSourcePrefs){
		mSourcePrefs = new SourcePrefs(inParent, "sourceprefsui");

		// disable nameCheckBox if taskbarCheckBox is disabled
		connect(mSourcePrefs->taskbarCheckBox, TQT_SIGNAL(toggled(bool)), mSourcePrefs->nameCheckBox, TQT_SLOT(setEnabled(bool)));
		
		// add prefs widgets from sub classes
		createSubPrefs(mSourcePrefs);
			
		// add bottom vspacer
		mSourcePrefs->layout()->addItem(new TQSpacerItem(0, 0, TQSizePolicy::Minimum, TQSizePolicy::Expanding) );

		updatePrefsGUI(); // fill the prefs gui
	}
	return mSourcePrefs;
}

SourcePrefs* Source::getPrefs(){
	return mSourcePrefs;
}

void Source::updatePrefsGUI(){
	// set values
	mSourcePrefs->nameLineEdit->setText(mName);
	mSourcePrefs->descriptionLabel->setText(mDescription);
	mSourcePrefs->taskbarCheckBox->setChecked(mShowOnApplet);
	mSourcePrefs->nameCheckBox->setChecked(mShowName);
	mSourcePrefs->tooltipCheckBox->setChecked(mToolTipEnabled);

	// disable/enable some widgets if source is disabled/enabled
	setPrefsWidgetsEnabled(mEnabled, mShowOnApplet);
}

void Source::setPrefsWidgetsEnabled(bool isEnabled, bool isShownOnApplet){
	mSourcePrefs->nameLabel->setEnabled(isEnabled);
	mSourcePrefs->nameLineEdit->setEnabled(isEnabled);
	mSourcePrefs->taskbarCheckBox->setEnabled(isEnabled);
	mSourcePrefs->nameCheckBox->setEnabled(isEnabled && isShownOnApplet);
	mSourcePrefs->tooltipCheckBox->setEnabled(isEnabled);
}

void Source::addPrefs(TQWidget* inParent){
	if(inParent != NULL)
		mSourcePrefs->layout()->add(inParent);
}

void Source::applyPrefs(){
	mMaybeShowOnApplet = mSourcePrefs->taskbarCheckBox->isChecked();
	mShowName = mSourcePrefs->nameCheckBox->isChecked();
	mName = mSourcePrefs->nameLineEdit->text();
	mToolTipEnabled = mSourcePrefs->tooltipCheckBox->isChecked();

	//kdDebug() << "Source::applyPrefs() mEnabled: " << mEnabled << ", mMaybeEnabled: " << mMaybeEnabled << endl;
	if(mEnabled != mMaybeEnabled){
		mEnabled = mMaybeEnabled;
		//kdDebug() << "Source::applyPrefs() emit enabledChanged: " << mEnabled << endl;
		emit enabledChanged(mEnabled, this);
        // force hide/show on kicker. if the user just 
        // disabled/enabled the source, we want to show / hide
        // the source too, also if the "show on kicker" property
        // did not changed. so, force this here.
        mShowOnApplet = !mMaybeShowOnApplet;
	}

    if(!mEnabled)
        emit displaySource(false, this);
    else if(mMaybeShowOnApplet != mShowOnApplet) {
        emit displaySource(mMaybeShowOnApplet, this);
    }
    mShowOnApplet = mMaybeShowOnApplet;
}

void Source::savePrefs(KConfig* inKConfig){
	inKConfig->writeEntry(mID + "_position", mPosition);
	inKConfig->writeEntry(mID + "_enabled", mEnabled);
	inKConfig->writeEntry(mID + "_showOnApplet", mShowOnApplet);
	inKConfig->writeEntry(mID + "_showName", mShowName);
	inKConfig->writeEntry(mID + "_name", mName);
	inKConfig->writeEntry(mID + "_toolTipEnabled", mToolTipEnabled);
}

void Source::loadPrefs(KConfig* inKConfig){
	mPosition = inKConfig->readNumEntry(mID + "_position", mPosition);
	mEnabled = inKConfig->readBoolEntry(mID + "_enabled", mEnabled);
	mMaybeEnabled = mEnabled;
	mShowOnApplet = inKConfig->readBoolEntry(mID + "_showOnApplet", mShowOnApplet);
	mMaybeShowOnApplet = mShowOnApplet;
	mShowName = inKConfig->readBoolEntry(mID + "_showName", mShowName);
	mName = inKConfig->readEntry(mID + "_name", mName);
	mToolTipEnabled = inKConfig->readBoolEntry(mID + "_toolTipEnabled", mToolTipEnabled);

	// initializing
	// this signal is usually catched by the ThreadedTrigger who enables or disables the fetch loop
	emit enabledChanged(mEnabled, this);
}

// utility methods
TQString Source::formatTemperature(const TQString& temp) const {
	if(mIsMetric) {
		return temp + TQString::fromUtf8(" °C");
	} else {
		return TQString::number(celsiusToFahrenheit(temp.toInt())).append(TQString::fromUtf8(" °F"));
	}
}

TQString Source::KHzinHumanReadable( uint value ) const{
	if( value >= 1000000 )
		return TQString::number( round(value/1000000.0, 1) ) + " GHz";
	else if( value >= 1000 )
		return TQString::number( round(value/1000.0, -1) ) + " MHz";
	else
		return TQString::number( value ) + " KHz";
}
double Source::round(double inValue, int inDigits) const{
    return floor(inValue * pow( 10, inDigits) + 0.5) * pow(10, -inDigits);
}
int Source::celsiusToFahrenheit(int inCelsius) const{
	return tqRound(1.8 * inCelsius + 32);
}

void Source::realizeWidget(){
	mIsMetric = KGlobal::locale()->measureSystem() == KLocale::Metric;
}
