/***************************************************************************
                         latexcmd.cpp
                         ------------
    date                 : Nov 26 2005
    version              : 0.22
    copyright            : (C) 2005 by Holger Danielsson
    email                : holger.danielsson@t-online.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "latexcmd.h"

#include <klocale.h>
#include "kiledebug.h"

namespace KileDocument
{

// BEGIN LatexCommands

LatexCommands::LatexCommands(KConfig *config, KileInfo *info) : m_config(config), m_ki(info)
{
	m_envGroupName = "Latex Environments";
	m_cmdGroupName = "Latex Commands";
	
	LatexCommands::resetCommands();
}

void LatexCommands::resetCommands()
{
	// description of the fields for environments
	//  0: standard entry (+,-)
	//  1: environmenty type (a,m,l,t,v)
	//  2: including starred version (*)
	//  3: eol character (\\\\)
	//  4: need mathmode ($) or displaymathmode ($$)
	//  5: standard tabulator (tabulator string, f.e. &=& or &=)
	//  6: optional parameter
	//  7: parameter group(s) 
	
	TQStringList envlist;
	envlist
	   // list environments
	   << "itemize,+,l,*,,,,,"  
	   << "enumerate,+,l,*,,,,,"  
	   << "description,+,l,*,,,,,"
	   << "Bitemize,+,l,,,,,,"   
	   << "Benumerate,+,l,,,,,,"  
	   << "Bdescription,+,l,,,,,,"
	   << "labeling,+,l,,,,,[ ],{ }"
	   // tabular environments
	   << "tabular,+,t,*,\\\\,,&,[tcb],"  
	   << "tabularx,+,t,,\\\\,,&,,{w}"
	   << "tabbing,+,t,,\\\\,,\\>,," 
	   << "longtable,+,t,,\\\\,,&,[tcb],"  
	   << "ltxtable,+,t,,\\\\,,&,[tcb],{w}"
	   << "supertabular,+,t,*,\\\\,,&,,"  
	   << "mpsupertabular,+,t,*,\\\\,,&,,"  
	   << "xtabular,+,t,*,\\\\,,&,," 
	   << "mpxtabular,+,t,*,\\\\,,&,,"
	   // math environments
	   << "displaymath,+,m,,,,,,"
	   << "equation,+,m,*,,,,," 
	   << "eqnarray,+,m,*,\\\\,,&=&,,"
	   << "array,+,m,,\\\\,$,&,[tcb],"
	   << "matrix,+,m,,\\\\,$,&,,"         
	   << "pmatrix,+,m,,\\\\,$,&,,"        
	   << "bmatrix,+,m,,\\\\,$,&,,"       
	   << "Bmatrix,+,m,,\\\\,$,&,,"       
	   << "vmatrix,+,m,,\\\\,$,&,,"        
	   << "Vmatrix,+,m,,\\\\,$,&,,"        
	   // amsmath environments
	   << "multline,+,a,*,\\\\,,,," 
	   << "gather,+,a,*,\\\\,,,,"
	   << "split,+,a,,\\\\,$$,,,"          // needs surrounding environment
	   << "align,+,a,*,\\\\,,&=,,"           
	   << "flalign,+,a,*,\\\\,,&=,,"         
	   << "alignat,+,a,*,\\\\,,&=,,{n}"    
	   << "aligned,+,a,,\\\\,$,&=,[tcb],"  
	   << "gathered,+,a,,\\\\,$,,[tcb],"   
	   << "alignedat,+,a,,\\\\,$,&=,[tcb],{n}" 
	   //<< "xalignat,+,a,*,\\\\,,&=,,{n}"   // obsolet
	   //<< "xxalignat,+,a,*,\\\\,,&=,,{n}"  // obsolet
	   << "cases,+,a,,\\\\,$,&,,"           
	   // verbatim environments
	   << "verbatim,+,v,*,,,,," 
	   << "boxedverbatim,+,v,,,,,,"
	   << "Bverbatim,+,v,,,,,[ ],"
	   << "Lverbatim,+,v,,,,,[ ],"
	   << "lstlisting,+,v,,,,,[ ],"
	   ;

	// description of the fields for commands
	//  0: standard entry (+,-)
	//  1: command type (L,R,C,I)
	//  2: including starred version (*)
	//  3: optional parameter
	//  4: parameter 
	
	TQStringList cmdlist;
	cmdlist
		// Labels
	   << "\\label,+,L,,,{ }"
		// References
	   << "\\ref,+,R,,,{ }"
	   << "\\pageref,+,R,,,{ }"
	   << "\\vref,+,R,,,{ }"
	   << "\\vpageref,+,R,,[ ],{ }"
	   << "\\fref,+,R,,,{ }"
	   << "\\Fref,+,R,,,{ }"
	   << "\\eqref,+,R,,,{ }"
	   << "\\autoref,+,R,,,{ }"
		// Citations
	   << "\\cite,+,C,,,{ }"
		// Includes
	   << "\\include,+,I,,,{ }"
	   << "\\input,+,I,,,{ }"
	   << "\\Input,+,I,,,{ }"
	   ;
	
	// first clear the dictionary
	m_latexCommands.clear();
	
	// insert environments
	addUserCommands(m_envGroupName,envlist);
	insert(envlist);
	
	// insert commands
	addUserCommands(m_cmdGroupName,cmdlist);
	insert(cmdlist);
}


// add user defined environments/commands 

void LatexCommands::addUserCommands(const TQString &name, TQStringList &list)
{
	if ( m_config->hasGroup(name) ) 
	{
		KILE_DEBUG() << name << endl;
		TQMap<TQString,TQString> map = m_config->entryMap(name);
		if ( ! map.empty() ) 
		{
			TQMapConstIterator<TQString,TQString> it;
			for ( it=map.begin(); it!=map.end(); ++it) 
			{
				list << it.key() + ",-," + it.data();
				KILE_DEBUG() << "\tadd: " <<  it.key() + " --> " + it.data() << endl;
			}
		}
	}
}

// insert all entries into the dictionary

void LatexCommands::insert(const TQStringList &list)
{
	// now insert new entries, if they have the right number of attributes
	TQStringList::ConstIterator it;
	for ( it=list.begin(); it!=list.end(); ++it ) 
	{
		int pos = (*it).find(',');
		if ( pos >= 0 ) 
		{
			TQString key = (*it).left(pos);
			TQString value = (*it).right( (*it).length()-pos-1 ); 
			TQStringList valuelist = TQStringList::split(',',value,true);
			uint attributes = ( key.at(0)=='\\' ) ? MaxCmdAttr : MaxEnvAttr;
			if ( valuelist.count() == attributes ) 
				m_latexCommands[key] = value;
			else
			   KILE_DEBUG() << "\tLatexCommands error: wrong number of attributes (" << key << " ---> " << value << ")" << endl;
		} 
		else 
		{
			KILE_DEBUG() << "\tLatexCommands error: no separator found (" << (*it) << ")"  << endl;
		}
	}
}

//////////////////// get value from dictionary  ////////////////////

// Get value of a key. A star at the end is stripped.

TQString LatexCommands::getValue(const TQString &name)
{
	TQString key = ( name.find('*',-1) >= 0 ) ? name.left(name.length()-1) : name;
	return ( m_latexCommands.contains(key) ) ? m_latexCommands[key] : TQString();
}

//////////////////// internal functions  ////////////////////

// get parameter at index

TQString LatexCommands::getAttrAt(const TQString &name, uint index)
{
	uint attributes = ( name.at(0)=='\\' ) ? MaxCmdAttr : MaxEnvAttr;
	TQStringList list = TQStringList::split(',',getValue(name),true);
	return ( index<attributes && list.count()==attributes ) ? list[index] : TQString();
}

// check for a standard environment 

bool LatexCommands::isUserDefined(const TQString &name)
{
	return ( getValue(name).at(0) == '-' );          
}

// check for a special environment type

bool LatexCommands::isType(const TQString &name, TQChar ch)
{
	if ( name.find('*',-1) >= 0 )
	{
		TQString envname = name.left( name.length()-1 );
		return ( getValue(envname).at(2)==ch && isStarredEnv(envname) );
	}
	else
	{
		return ( getValue(name).at(2) == ch );          
	}
}

//////////////////// attributes and characters ////////////////////

// convert attribute to character

TQChar LatexCommands::getAttrChar(CmdAttribute attr)
{
	TQChar ch;
	switch ( attr ) 
	{
		case CmdAttrAmsmath:   ch = 'a'; break;
		case CmdAttrMath:      ch = 'm'; break;
		case CmdAttrList:      ch = 'l'; break;
		case CmdAttrVerbatim:  ch = 'v'; break;
		case CmdAttrTabular:   ch = 't'; break;
		case CmdAttrLabel:     ch = 'L'; break;
		case CmdAttrReference: ch = 'R'; break;
		case CmdAttrCitations: ch = 'C'; break;
		case CmdAttrIncludes:  ch = 'I'; break;
		default:
		     KILE_DEBUG() << "\tLatexCommands error: unknown type of env/cmd: code " << attr << endl;
			  return '?';
	}	
	
	return ch;
}

// convert character to attribute  

CmdAttribute LatexCommands::getCharAttr(TQChar ch)
{
	CmdAttribute attr;
	switch ( ch ) 
	{
		case 'a': attr = CmdAttrAmsmath;   break;
		case 'm': attr = CmdAttrMath;      break;
		case 'l': attr = CmdAttrList;      break;
		case 'v': attr = CmdAttrVerbatim;  break;
		case 't': attr = CmdAttrTabular;   break;
		case 'L': attr = CmdAttrLabel;     break;
		case 'R': attr = CmdAttrReference; break;
		case 'C': attr = CmdAttrCitations; break;
		case 'I': attr = CmdAttrIncludes;  break;
		default:
		     KILE_DEBUG() << "\tLatexCommands error: unknown type of env/cmd: " << static_cast<char>(ch) << endl;
			  return CmdAttrNone;
	}	
	
	return attr;
}	

//////////////////// public attributes  ////////////////////

// check for environment types

bool LatexCommands::isMathEnv(const TQString &name)
{
	TQChar ch = getValue(name).at(2);
	return ( ch=='m' || ch=='a' );
}

// check for some special attributes

bool LatexCommands::isStarredEnv(const TQString &name)
{
	return ( getAttrAt(name,2) == "*" );
}

bool LatexCommands::isCrEnv(const TQString &name)
{
	return ( getAttrAt(name,3) == "\\\\" );
}

bool LatexCommands::isMathModeEnv(const TQString &name)
{
	return ( getAttrAt(name,4) == "$" );
}

bool LatexCommands::isDisplaymathModeEnv(const TQString &name)
{
	return ( getAttrAt(name,4) == "$$" );
}

bool LatexCommands::needsMathMode(const TQString &name)
{
	return ( isMathModeEnv(name) || isDisplaymathModeEnv(name) );
}

TQString LatexCommands::getTabulator(const TQString &name)
{
	TQString tab = getAttrAt(name,5);
	return ( tab.find('&') >= 0 ) ? tab : TQString();
}

//////////////////// environments and commands ////////////////////

// get a list of environments and commands. The search can be restricted
// to given attributes and userdefined environments and commands

void LatexCommands::commandList(TQStringList &list, uint attr, bool userdefined)
{
	list.clear();
	
	TQMapConstIterator<TQString,TQString> it;
	for ( it=m_latexCommands.begin(); it!=m_latexCommands.end(); ++it) 
	{
		// first check, if we need really need all environments and commands
		// or if a restriction to some attributes is given
		if ( attr != (uint)CmdAttrNone ) 
		{
			if ( ! ( attr & (uint)getCharAttr( it.data().at(2) ) ) )            
				continue;
		}
		
		// second check, if we need only user defined environments or commands
		if ( ! userdefined )
			list.append( it.key() );
		else if ( it.data().at(0) == '-' )
			list.append( it.key() );
	}
}

// get all attributes for a given environment and command

bool LatexCommands::commandAttributes(const TQString &name, LatexCmdAttributes &attr)
{
	uint attributes = ( name.at(0)=='\\' ) ? MaxCmdAttr : MaxEnvAttr;
	
	// split attribute list
	TQStringList list = TQStringList::split(',',getValue(name),true);
	
	// check number of attributes
	if (  list.count() != attributes ) 
		return false;
		
	// check for a standard environment/command
	attr.standard = ( list[0] == "+" );
	
	// most important: type of environment or command
	attr.type = getCharAttr( list[1].at(0) );
	if ( attr.type == CmdAttrNone )
		return false;
		
	// all environments/commands have starred attribute 
	attr.starred = ( list[2] == "*" ) ;
	
	// next attributes differ for environments and commands
	if ( attributes == MaxEnvAttr ) 
	{
		attr.cr = ( list[3] == "\\\\" ) ;
		attr.mathmode = ( list[4] == "$" ) ;
		attr.displaymathmode = ( list[4] == "$$" ) ;
		attr.tabulator = list[5];
		attr.option = list[6];
		attr.parameter = list[7];
	}
	else
	{
		attr.cr = false;
		attr.mathmode = false;
		attr.displaymathmode = false;
		attr.tabulator = TQString();
		attr.option = list[3];
		attr.parameter = list[4];
	}
	
	return true;
}

//////////////////// determine config string ////////////////////

TQString LatexCommands::configString(LatexCmdAttributes &attr,bool env)
{
	// most important: type of environment or command
	TQChar ch = getAttrChar( attr.type );
	if ( ch == '?' )
		return TQString();
	TQString s = TQString("%1,").arg(ch);
	
	// all environments/commands have starred attribute 
	if ( attr.starred )
		s += "*,";
	else
		s += ',';
	
	// next attributes are only valid for environments
	if ( env ) 
	{
		if ( attr.cr )
			s += "\\\\,";
		else
			s += ',';
		if ( attr.mathmode )
			s += "$,";
		else if ( attr.displaymathmode )
			s += "$$";
		else
			s += ',';
		s += attr.tabulator + ',';
	}
	
	// option and parameter are for both types again
	s += attr.option + ',';
	s += attr.parameter;
	
	return s;    // s.left(s.length()-1);
}
 
// END LatexCommands

}
#include "latexcmd.moc"
