/***************************************************************************
    begin                : Thu Jul 17 2003
    copyright            : (C) 2003 by Jeroen Wijnhout
    email                : Jeroen.Wijnhout@kdemail.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// 2007-03-12 dani
//  - use KileDocument::Extensions

#include "kileinfo.h"

#include <tqwidget.h>
#include <tqfileinfo.h>
#include <tqobject.h>

#include <kate/document.h>
#include <kate/view.h>
#include <klocale.h>
#include <kmessagebox.h>

#include "kilestructurewidget.h"
#include "kiledocmanager.h"
#include "kileviewmanager.h"
#include "kiledocumentinfo.h"
#include "kileproject.h"
#include "kileuntitled.h"
#include "kilejscript.h"
#include "editorkeysequencemanager.h"
#include "templates.h"

#include <kstandarddirs.h>
#include <tqstringlist.h>
#include <tqstring.h>

KileInfo::KileInfo(TQWidget *parent) :
	m_manager(0L),
	m_jScriptManager(0L),
	m_toolFactory(0L),
	m_texKonsole(0L),
	m_edit(0L),
	m_parentWidget(parent),
	m_currentTarget(TQString())
{
	m_docManager = new KileDocument::Manager(this, TQT_TQOBJECT(parent), "KileDocument::Manager");
	m_viewManager= new KileView::Manager(this, TQT_TQOBJECT(parent), "KileView::Manager");
	m_templateManager = new KileTemplate::Manager(this, TQT_TQOBJECT(parent), "KileTemplate::Manager");
	m_editorKeySequenceManager = new KileEditorKeySequence::Manager(this, TQT_TQOBJECT(parent), "KileEditorKeySequence::Manager");
	TQObject::connect(m_docManager, TQT_SIGNAL(documentStatusChanged(Kate::Document*, bool, unsigned char)), m_viewManager, TQT_SLOT(reflectDocumentStatus(Kate::Document*, bool, unsigned char)));
}

KileInfo::~KileInfo()
{
}

Kate::Document * KileInfo::activeTextDocument() const
{
	Kate::View *view = viewManager()->currentTextView();
	if (view) return view->getDoc(); else return 0L;
}

TQString KileInfo::getName(Kate::Document *doc, bool shrt)
{
	KILE_DEBUG() << "===KileInfo::getName(Kate::Document *doc, bool " << shrt << ")===" << endl;
	TQString title=TQString();
	
	if (!doc)
		doc = activeTextDocument();
	if (doc)
	{
		KILE_DEBUG() << "url " << doc->url().path() << endl;
		title = shrt ? doc->url().fileName() : doc->url().path();
               //work around for bug in KatePart, use docName and not url
               //reloading the file after is it changed on disc by another application
               //cause the URL to be empty for a short while
		if ( title.isEmpty() )
			title = shrt ? TQFileInfo(doc->docName()).fileName() : doc->docName();
	}

	return title;
}

TQString KileInfo::getCompileName(bool shrt /* = false */)
{
	KileProject *project = docManager()->activeProject();

	if (m_singlemode)
	{
		if (project)
		{
			if (project->masterDocument().length() > 0)
			{
				KURL master = KURL::fromPathOrURL(project->masterDocument());
				if (shrt) return master.fileName();
				else return master.path();
			}
			else
			{
				KileProjectItem *item = project->rootItem(docManager()->activeProjectItem());
				if (item)
				{
					KURL url = item->url();
					if (shrt) return url.fileName();
					else return url.path();
				}
				else
					return TQString();
			}
		}
		else
			return getName(activeTextDocument(), shrt);
	}
	else
	{
		TQFileInfo fi(m_masterName);
		if (shrt)
			return fi.fileName();
		else
			return m_masterName;
	}
}

TQString KileInfo::getFullFromPrettyName(const TQString & name)
{
	if(name.isNull())
		return name;

	TQString file = name;

	if (file.left(2) == "./" )
	{
		file = TQFileInfo(outputFilter()->source()).dirPath(true) + '/' + file.mid(2);
	}

	if (file[0] != '/' )
	{
		file = TQFileInfo(outputFilter()->source()).dirPath(true) + '/' + file;
	}

	TQFileInfo fi(file);
	if ( file.isNull() || fi.isDir() || (! fi.exists()) || (! fi.isReadable()))
	{
		// - call from logwidget or error handling, which 
		//   tries to determine the LaTeX source file
		bool found = false;
		TQStringList extlist = TQStringList::split( " ", m_extensions->latexDocuments() );
		for ( TQStringList::Iterator it=extlist.begin(); it!=extlist.end(); ++it )
		{
			TQString name = file + (*it);
			if ( TQFileInfo(name).exists() )
			{
				file = name;
				fi.setFile(name);
				found = true;
				break;
			}
		}
		if ( ! found )
			file = TQString();
	}

	if ( ! fi.isReadable() ) return TQString();

	return file;
}

KURL::List KileInfo::getParentsFor(KileDocument::Info *info)
{
	KileProjectItemList *items = docManager()->itemsFor(info);
	KURL::List list;
	for ( uint i = 0; i < items->count(); ++i)
		if (items->at(i)->parent()) list.append(items->at(i)->parent()->url());

	return list;
}

const TQStringList* KileInfo::retrieveList(const TQStringList* (KileDocument::Info::*getit)() const, KileDocument::Info * docinfo /* = 0L */)
{
	m_listTemp.clear();

	if (docinfo == 0L) docinfo = docManager()->getInfo();
	KileProjectItem *item = docManager()->itemFor(docinfo, docManager()->activeProject());

	KILE_DEBUG() << "Kile::retrieveList()" << endl;
	if (item)
	{
		const KileProject *project = item->project();
		const KileProjectItem *root = project->rootItem(item);
		if (root)
		{
			KILE_DEBUG() << "\tusing root item " << root->url().fileName() << endl;

			TQPtrList<KileProjectItem> children;
			children.append(root);
			root->allChildren(&children);

			const TQStringList *list;

			for (uint i=0; i < children.count(); ++i)
			{
				KILE_DEBUG() << "\t" << children.at(i)->url().fileName() << endl;
				list = (children.at(i)->getInfo()->*getit)();
				if (list)
				{
					for (uint i=0; i < list->count(); ++i)
						m_listTemp << (*list)[i];
				}
			}

			return &m_listTemp;
		}
		else
			return &m_listTemp;
	}
	else	if (docinfo)
	{
		m_listTemp = *((docinfo->*getit)());
		return &m_listTemp;
	}
	else
		return &m_listTemp;
}

const TQStringList* KileInfo::allLabels(KileDocument::Info * info)
{
	KILE_DEBUG() << "Kile::allLabels()" << endl;
	const TQStringList* (KileDocument::Info::*p)() const=&KileDocument::Info::labels;
	const TQStringList* list = retrieveList(p, info);
	return list;
}

const TQStringList* KileInfo::allBibItems(KileDocument::Info * info)
{
	KILE_DEBUG() << "Kile::allBibItems()" << endl;
	const TQStringList* (KileDocument::Info::*p)() const=&KileDocument::Info::bibItems;
	const TQStringList* list = retrieveList(p, info);
	return list;
}

const TQStringList* KileInfo::allBibliographies(KileDocument::Info * info)
{
	KILE_DEBUG() << "Kile::bibliographies()" << endl;
	const TQStringList* (KileDocument::Info::*p)() const=&KileDocument::Info::bibliographies;
	const TQStringList* list = retrieveList(p, info);
	return list;
}

const TQStringList* KileInfo::allDependencies(KileDocument::Info * info)
{
	KILE_DEBUG() << "Kile::dependencies()" << endl;
	const TQStringList* (KileDocument::Info::*p)() const=&KileDocument::Info::dependencies;
	const TQStringList* list = retrieveList(p, info);
	return list;
}

const TQStringList* KileInfo::allNewCommands(KileDocument::Info * info)
{
	KILE_DEBUG() << "Kile::newCommands()" << endl;
	const TQStringList* (KileDocument::Info::*p)() const=&KileDocument::Info::newCommands;
	const TQStringList* list = retrieveList(p, info);
	return list;
}

const TQStringList* KileInfo::allPackages(KileDocument::Info * info)
{
	KILE_DEBUG() << "Kile::allPackages()" << endl;
	const TQStringList* (KileDocument::Info::*p)() const=&KileDocument::Info::packages;
	const TQStringList* list = retrieveList(p, info);
	return list;
}

TQString KileInfo::lastModifiedFile(KileDocument::Info * info)
{
	if (info == 0) info = docManager()->getInfo();
	const TQStringList *list = allDependencies(info);
	return info->lastModifiedFile(list);
}

TQString KileInfo::documentTypeToString(KileDocument::Type type)
{
	switch(type) {
		case KileDocument::Undefined:
			return i18n("Undefined");
		case KileDocument::Text:
			return i18n("Text");
		case KileDocument::LaTeX:
			return i18n("LaTeX");
		case KileDocument::BibTeX:	
			return i18n("BibTeX");
		case KileDocument::Script:
			return i18n("Script");
	}
	return TQString();
}

bool KileInfo::similarOrEqualURL(const KURL &validurl, const KURL &testurl)
{
	if ( testurl.isEmpty() || testurl.path().isEmpty() ) return false;

	bool absolute = testurl.path().startsWith("/");
	return (
		     (validurl == testurl) ||
		     (!absolute && validurl.path().endsWith(testurl.path()))
		   );
}

bool KileInfo::isOpen(const KURL & url)
{
	KILE_DEBUG() << "==bool KileInfo::isOpen(const KURL & url)=============" << endl;
	uint cnt = viewManager()->textViews().count();
	
	for ( uint i = 0; i < cnt; ++i)
	{
		if ( viewManager()->textView(i)->getDoc() && similarOrEqualURL(viewManager()->textView(i)->getDoc()->url(), url) )
			return true;
	}

	return false;
}

bool KileInfo::projectIsOpen(const KURL & url)
{
	KileProject *project = docManager()->projectFor(url);

	return project != 0 ;
}

TQString KileInfo::getSelection() const
{
	Kate::Document *doc = activeTextDocument();
	
	if (doc && doc->hasSelection())
	{
		return doc->selection();
	}
	
	return TQString();
}

void KileInfo::clearSelection() const
{
	Kate::Document *doc = activeTextDocument();
	
	if (doc && doc->hasSelection())
	{
		doc->removeSelectedText();
	}
}

TQString KileInfo::expandEnvironmentVars(const TQString &str)
{
	static TQRegExp reEnvVars("\\$(\\w+)");
	TQString result = str;
	int index = -1;
	while ( (index = str.find(reEnvVars, index + 1)) != -1 )
		result.replace(reEnvVars.cap(0),getenv(reEnvVars.cap(1).local8Bit()));

	return result;
}

TQString KileInfo::checkOtherPaths(const TQString &path,const TQString &file, int type)
{
	KILE_DEBUG() << "TQString KileInfo::checkOtherPaths(const TQString &path,const TQString &file, int type)" << endl;
	TQStringList inputpaths;
	TQString configpaths;
	TQFileInfo info;

	switch(type)
	{
		case bibinputs:
			configpaths = KileConfig::bibInputPaths() + ":$BIBINPUTS";
			break;
		case texinputs:
			configpaths = KileConfig::teXPaths() + ":$TEXINPUTS";
			break;
		case bstinputs:
			configpaths = KileConfig::bstInputPaths() + ":$BSTINPUTS";
			break;
		default:
			KILE_DEBUG() << "Unknown type in checkOtherPaths" << endl;
			return TQString();
			break;
	}

	inputpaths = TQStringList::split( ":",  expandEnvironmentVars(configpaths));
	inputpaths.prepend(path);

		// the first match is supposed to be the correct one
	for ( TQStringList::Iterator it = inputpaths.begin(); it != inputpaths.end(); ++it )
	{
		KILE_DEBUG() << "path is " << *it << "and file is " << file << endl;
		info.setFile((*it) + '/' + file);
		if(info.exists())
		{
			KILE_DEBUG() << "filepath after correction is: " << info.dirPath() << endl;
			return info.absFilePath();
		}
	}
	return TQString();
}

TQString KileInfo::relativePath(const TQString basepath, const TQString & file)
{
	KURL url = KURL::fromPathOrURL(file);
	TQString path = url.directory();
	TQString filename = url.fileName();

	KILE_DEBUG() <<"===findRelativeURL==================" << endl;
	KILE_DEBUG() << "\tbasepath : " <<  basepath << " path: " << path << endl;

	TQStringList basedirs = TQStringList::split("/", basepath, false);
	TQStringList dirs = TQStringList::split("/", path, false);

	uint nDirs = dirs.count();
	//uint nBaseDirs = basedirs.count();

	while ( dirs.count() > 0 && basedirs.count() > 0 &&  dirs[0] == basedirs[0] )
	{
		dirs.pop_front();
		basedirs.pop_front();
	}

	/*KILE_DEBUG() << "\tafter" << endl;
	for (uint i=0; i < basedirs.count(); ++i)
	{
		KILE_DEBUG() << "\t\tbasedirs " << i << ": " << basedirs[i] << endl;
	}

	for (uint i=0; i < dirs.count(); ++i)
	{
		KILE_DEBUG() << "\t\tdirs " << i << ": " << dirs[i] << endl;
	}*/

	if (nDirs != dirs.count() )
	{
		path = dirs.join("/");

		//KILE_DEBUG() << "\tpath : " << path << endl;

		if (basedirs.count() > 0)
		{
			for (uint j=0; j < basedirs.count(); ++j)
			{
				path = "../" + path;
			}
		}

		if ( path.length()>0 && path.right(1) != "/" ) path = path + '/';

		path = path+filename;
	}
	else //assume an absolute path was requested
	{
		path = url.path();
	}

	KILE_DEBUG() << "\trelative path : " << path << endl;

	return path;
}
