/*
    This file is part of KOrganizer.
    Copyright (c) 2002 Cornelius Schumacher <schumacher@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "entry.h"

#include <kglobal.h>
#include <klocale.h>

using namespace KNS;

Entry::Entry() :
  mRelease( 0 ), mReleaseDate( TQDate::currentDate() ), mRating( 0 ),
  mDownloads( 0 )
{
}

Entry::Entry( const TQDomElement &e )
{
  parseDomElement( e );
}

Entry::~Entry()
{
}


void Entry::setName( const TQString &name )
{
  mName = name;
}

TQString Entry::name() const
{
  return mName;
}


void Entry::setType( const TQString &type )
{
  mType = type;
}

TQString Entry::type() const
{
  return mType;
}


void Entry::setAuthor( const TQString &author )
{
  mAuthor = author;
}

TQString Entry::author() const
{
  return mAuthor;
}


void Entry::setLicence( const TQString &license )
{
  mLicence = license;
}

TQString Entry::license() const
{
  return mLicence;
}


void Entry::setSummary( const TQString &text, const TQString &lang )
{
  mSummaryMap.insert( lang, text );

  if ( mLangs.find( lang ) == mLangs.end() ) mLangs.append( lang );
}

TQString Entry::summary( const TQString &lang ) const
{
  if ( mSummaryMap.isEmpty() ) return TQString();

  if ( !mSummaryMap[ lang ].isEmpty() ) return mSummaryMap[ lang ];
  else {
    TQStringList langs = KGlobal::locale()->languageList();
    for(TQStringList::Iterator it = langs.begin(); it != langs.end(); ++it)
      if( !mSummaryMap[ *it ].isEmpty() ) return mSummaryMap[ *it ];
  }
  if ( !mSummaryMap[ TQString() ].isEmpty() ) return mSummaryMap[ TQString() ];
  else return *(mSummaryMap.begin());
}


void Entry::setVersion( const TQString &version )
{
  mVersion = version;
}

TQString Entry::version() const
{
  return mVersion;
}


void Entry::setRelease( int release )
{
  mRelease = release;
}

int Entry::release() const
{
  return mRelease;
}


void Entry::setReleaseDate( const TQDate &d )
{
  mReleaseDate = d;
}

TQDate Entry::releaseDate() const
{
  return mReleaseDate;
}


void Entry::setPayload( const KURL &url, const TQString &lang )
{
  mPayloadMap.insert( lang, url );

  if ( mLangs.find( lang ) == mLangs.end() ) mLangs.append( lang );
}

KURL Entry::payload( const TQString &lang ) const
{
  KURL payload = mPayloadMap[ lang ];
  if ( payload.isEmpty() ) {
    TQStringList langs = KGlobal::locale()->languageList();
    for(TQStringList::Iterator it = langs.begin(); it != langs.end(); ++it)
      if( !mPayloadMap[ *it ].isEmpty() ) return mPayloadMap[ *it ];
  }
  if ( payload.isEmpty() ) payload = mPayloadMap [ TQString() ];
  if ( payload.isEmpty() && !mPayloadMap.isEmpty() ) {
    payload = *(mPayloadMap.begin());
  }
  return payload;
}


void Entry::setPreview( const KURL &url, const TQString &lang )
{
  mPreviewMap.insert( lang, url );
  
  if ( mLangs.find( lang ) == mLangs.end() ) mLangs.append( lang );
}

KURL Entry::preview( const TQString &lang ) const
{
  KURL preview = mPreviewMap[ lang ];
  if ( preview.isEmpty() ) {
    TQStringList langs = KGlobal::locale()->languageList();
    for(TQStringList::Iterator it = langs.begin(); it != langs.end(); ++it)
      if( !mPreviewMap[ *it ].isEmpty() ) return mPreviewMap[ *it ];
  }
  if ( preview.isEmpty() ) preview = mPreviewMap [ TQString() ];
  if ( preview.isEmpty() && !mPreviewMap.isEmpty() ) {
    preview = *(mPreviewMap.begin());
  }
  return preview;
}


void Entry::setRating( int rating )
{
  mRating = rating;
}

int Entry::rating()
{
  return mRating;
}


void Entry::setDownloads( int downloads )
{
  mDownloads = downloads;
}

int Entry::downloads()
{
  return mDownloads;
}

TQString Entry::fullName()
{
  return name() + "-" + version() + "-" + TQString::number( release() );
}

TQStringList Entry::langs()
{
  return mLangs;
}

void Entry::parseDomElement( const TQDomElement &element )
{
  if ( element.tagName() != "stuff" ) return;
  mType = element.attribute("type");

  TQDomNode n;
  for( n = element.firstChild(); !n.isNull(); n = n.nextSibling() ) {
    TQDomElement e = n.toElement();
    if ( e.tagName() == "name" ) setName( e.text().stripWhiteSpace() );
    if ( e.tagName() == "author" ) setAuthor( e.text().stripWhiteSpace() );
    if ( e.tagName() == "licence" ) setLicence( e.text().stripWhiteSpace() );
    if ( e.tagName() == "summary" ) {
      TQString lang = e.attribute( "lang" );
      setSummary( e.text().stripWhiteSpace(), lang );
    }
    if ( e.tagName() == "version" ) setVersion( e.text().stripWhiteSpace() );
    if ( e.tagName() == "release" ) setRelease( e.text().toInt() );
    if ( e.tagName() == "releasedate" ) {
      TQDate date = TQDate::fromString( e.text().stripWhiteSpace(), Qt::ISODate );
      setReleaseDate( date );
    }
    if ( e.tagName() == "preview" ) {
      TQString lang = e.attribute( "lang" );
      setPreview( KURL( e.text().stripWhiteSpace() ), lang );
    }
    if ( e.tagName() == "payload" ) {
      TQString lang = e.attribute( "lang" );
      setPayload( KURL( e.text().stripWhiteSpace() ), lang );
    }
    if ( e.tagName() == "rating" ) setRating( e.text().toInt() );
    if ( e.tagName() == "downloads" ) setDownloads( e.text().toInt() );
  }
}

TQDomElement Entry::createDomElement( TQDomDocument &doc,
                                              TQDomElement &parent )
{
  TQDomElement entry = doc.createElement( "stuff" );
  entry.setAttribute("type", mType);
  parent.appendChild( entry );

  addElement( doc, entry, "name", name() );
  addElement( doc, entry, "author", author() );
  addElement( doc, entry, "licence", license() );
  addElement( doc, entry, "version", version() );
  addElement( doc, entry, "release", TQString::number( release() ) );
  addElement( doc, entry, "rating", TQString::number( rating() ) );
  addElement( doc, entry, "downloads", TQString::number( downloads() ) );

  addElement( doc, entry, "releasedate",
              releaseDate().toString( Qt::ISODate ) );

  TQStringList ls = langs();
  TQStringList::ConstIterator it;
  for( it = ls.begin(); it != ls.end(); ++it ) {
    TQDomElement e = addElement( doc, entry, "summary", summary( *it ) );
    e.setAttribute( "lang", *it );
    e = addElement( doc, entry, "preview", preview( *it ).url() );
    e.setAttribute( "lang", *it );
    e = addElement( doc, entry, "payload", payload( *it ).url() );
    e.setAttribute( "lang", *it );
  }

  return entry;
}

TQDomElement Entry::addElement( TQDomDocument &doc, TQDomElement &parent,
                               const TQString &tag, const TQString &value )
{
  TQDomElement n = doc.createElement( tag );
  n.appendChild( doc.createTextNode( value ) );
  parent.appendChild( n );

  return n;
}
