/*
    This file is part of the scalix resource - based on the kolab resource.

    Copyright (c) 2004 Bo Thorsen <bo@sonofthor.dk>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.

    In addition, as a special exception, the copyright holders give
    permission to link the code of this program with any edition of
    the TQt library by Trolltech AS, Norway (or with modified versions
    of TQt that use the same license as TQt), and distribute linked
    combinations including the two.  You must obey the GNU General
    Public License in all respects for all of the code used other than
    TQt.  If you modify this file, you may extend this exception to
    your version of the file, but you are not obligated to do so.  If
    you do not wish to do so, delete this exception statement from
    your version.
*/

#include "resourcescalixbase.h"
#include "kmailconnection.h"

#include <folderselectdialog.h>

#include <klocale.h>
#include <kstandarddirs.h>
#include <kinputdialog.h>
#include <kurl.h>
#include <ktempfile.h>
#include <kmessagebox.h>
#include <tqtextstream.h>
#include <kdebug.h>

using namespace Scalix;

static unsigned int uniquifier = 0;

ResourceScalixBase::ResourceScalixBase( const TQCString& objId )
  : mSilent( false )
{
  KGlobal::locale()->insertCatalogue( "kres_scalix" );
  KGlobal::locale()->insertCatalogue( "libkcal" );
  TQString uniqueObjId = TQString( objId ) + TQString::number( uniquifier++ );
  mConnection = new KMailConnection( this, uniqueObjId.utf8() );
}

ResourceScalixBase::~ResourceScalixBase()
{
  delete mConnection;
}


bool ResourceScalixBase::kmailSubresources( TQValueList<KMailICalIface::SubResource>& lst,
                                           const TQString& contentsType ) const
{
  return mConnection->kmailSubresources( lst, contentsType );
}

bool ResourceScalixBase::kmailTriggerSync( const TQString& contentsType ) const
{
  return mConnection->kmailTriggerSync( contentsType );
}


bool ResourceScalixBase::kmailIncidencesCount( int &count,
                                              const TQString& mimetype,
                                              const TQString& resource ) const
{
  return mConnection->kmailIncidencesCount( count, mimetype, resource );
}

bool ResourceScalixBase::kmailIncidences( TQMap<TQ_UINT32, TQString>& lst,
                                         const TQString& mimetype,
                                         const TQString& resource,
                                         int startIndex,
                                         int nbMessages ) const
{
  return mConnection->kmailIncidences( lst, mimetype, resource, startIndex, nbMessages );
}

bool ResourceScalixBase::kmailGetAttachment( KURL& url, const TQString& resource,
                                            TQ_UINT32 sernum,
                                            const TQString& filename ) const
{
  return mConnection->kmailGetAttachment( url, resource, sernum, filename );
}

bool ResourceScalixBase::kmailDeleteIncidence( const TQString& resource,
                                              TQ_UINT32 sernum )
{
  return mSilent || mConnection->kmailDeleteIncidence( resource, sernum );
}

bool ResourceScalixBase::kmailUpdate( const TQString& resource,
                                     TQ_UINT32& sernum,
                                     const TQString& xml,
                                     const TQString& mimetype,
                                     const TQString& subject,
                                     const CustomHeaderMap& _customHeaders,
                                     const TQStringList& _attachmentURLs,
                                     const TQStringList& _attachmentMimetypes,
                                     const TQStringList& _attachmentNames,
                                     const TQStringList& deletedAttachments )
{
  if ( mSilent )
    return true;

  TQString subj = subject;
  if ( subj.isEmpty() )
    subj = i18n("Internal kolab data: Do not delete this mail.");

  // ical style, simply put the data inline
  return mConnection->kmailUpdate( resource, sernum, subj, xml, _customHeaders,
    _attachmentURLs, _attachmentMimetypes, _attachmentNames, deletedAttachments );
}

TQString ResourceScalixBase::configFile( const TQString& type ) const
{
  return locateLocal( "config",
                      TQString( "kresources/scalix/%1rc" ).arg( type ) );
}

bool ResourceScalixBase::connectToKMail() const
{
  return mConnection->connectToKMail();
}

TQString ResourceScalixBase::findWritableResource( const ResourceMap& resources )
{
  // I have to use the label (shown in the dialog) as key here. But given how the
  // label is made up, it should be unique. If it's not, well the dialog would suck anyway...
  TQMap<TQString, TQString> possible;
  TQStringList labels;
  ResourceMap::ConstIterator it;
  for ( it = resources.begin(); it != resources.end(); ++it ) {
    if ( it.data().writable() && it.data().active() ) {
      // Writable and active possibility
      possible[ it.data().label() ] = it.key();
    }
  }

  if ( possible.isEmpty() ) { // None found!!
    kdWarning(5650) << "No writable resource found!" << endl;
    KMessageBox::error( 0, i18n( "No writable resource was found, saving will not be possible. Reconfigure KMail first." ) );
    return TQString();
  }
  if ( possible.count() == 1 )
    // Just one found
    return possible.begin().data(); // yes this is the subresource key, i.e. location

  // Several found, ask the user
  TQString chosenLabel = KPIM::FolderSelectDialog::getItem( i18n( "Select Resource Folder" ),
                                                           i18n( "You have more than one writable resource folder. "
                                                                 "Please select the one you want to write to." ),
                                                           possible.keys() );
  if ( chosenLabel.isEmpty() ) // cancelled
    return TQString();
  return possible[chosenLabel];
}

KMailICalIface::StorageFormat ResourceScalixBase::kmailStorageFormat( const TQString &folder ) const
{
  KMailICalIface::StorageFormat format = (KMailICalIface::StorageFormat) 3;
  mConnection->kmailStorageFormat( format, folder );
  return format;
}
