/*
   This file is part of KDE Kontact.

   Copyright (c) 2001 Matthias Hoelzer-Kluepfel <mhk@kde.org>
   Copyright (c) 2002-2003 Daniel Molkentin <molkentin@kde.org>
   Copyright (c) 2003 Cornelius Schumacher <schumacher@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#ifndef KONTACT_PLUGIN_H
#define KONTACT_PLUGIN_H

#include <tqobject.h>
#include <kxmlguiclient.h>
#include <kdepimmacros.h>
#include <tqptrlist.h>

class TQStringList;
class DCOPClient;
class DCOPObject;
class KAboutData;
class KAction;
class KConfig;
class TQWidget;
namespace KParts { class ReadOnlyPart; }

/**
  Increase this version number whenever you make a change
  in the API.
 */
#define KONTACT_PLUGIN_VERSION 6

namespace Kontact
{

class Core;
class Summary;

/**
  Base class for all Plugins in Kontact. Inherit from it
  to get a plugin. It can insert an icon into the sidepane,
  add widgets to the widgetstack and add menu items via XMLGUI.
 */
class KDE_EXPORT Plugin : public TQObject, virtual public KXMLGUIClient
{
  Q_OBJECT
  TQ_OBJECT

  public:
    /**
      Creates a new Plugin, note that name parameter name is required if
      you want your plugin to do dcop via it's own instance of
      DCOPClient by calling dcopClient.
      @note name MUST be the name of the application that
      provides the part! This is the name used for DCOP registration.
      It's ok to have several plugins using the same application name.
    */
    Plugin( Core *core, TQObject *parent, const char *name );

    ~Plugin();

    /**
      Sets the identifier.
    */
    void setIdentifier( const TQString &identifier );

    /**
      Returns the identifier. It is used as argument for several
      methods of Kontacts core.
    */
    TQString identifier() const;

    /**
      Sets the localized title.
     */
    void setTitle( const TQString &title );

    /**
      Returns the localized title.
    */
    TQString title() const;

    /**
      Sets the icon name.
    */
    void setIcon( const TQString &icon );

    /**
      Returns the icon name.
    */
    TQString icon() const;

    /**
      Sets the name of executable (if existant).
    */
    void setExecutableName( const TQString &bin );

    /**
      Returns the name of the binary (if existant).
    */
    TQString executableName() const;

    /**
      Set name of library which contains the KPart used by this plugin.
    */
    void setPartLibraryName( const TQCString & );

    /**
      Create the DCOP interface for the given @p serviceType, if this
      plugin provides it. Return false otherwise.
    */
    virtual bool createDCOPInterface( const TQString& /*serviceType*/ ) { return false; }

    /**
      Reimplement this method and return whether a standalone application is still running
      This is only required if your part is also available as standalone application.
    */
    virtual bool isRunningStandalone() { return false; }

    /**
      Reimplement this method if your application needs a different approach to be brought
      in the foreground. The default behaviour is calling the binary.
      This is only required if your part is also available as standalone application.
    */
    virtual void bringToForeground();

    /**
      Reimplement this method if you want to add your credits to the Kontact
      about dialog.
    */
    virtual const KAboutData *aboutData();

    /**
      You can use this method if you need to access the current part. You can be
      sure that you always get the same pointer as long as the part has not been
      deleted.
    */
    KParts::ReadOnlyPart *part();

     /**
       Reimplement this method and return the a path relative to "data" to the tips file.
     */
    virtual TQString tipFile() const;

    /**
      This function is called when the plugin is selected by the user before the
      widget of the KPart belonging to the plugin is raised.
    */
    virtual void select();

    /**
      This function is called whenever the config dialog has been closed
      successfully.
     */
    virtual void configUpdated();

    /**
      Reimplement this method if you want to add a widget for your application
      to Kontact's summary page.
    */
    virtual Summary *createSummaryWidget( TQWidget * /*parent*/ ) { return 0; }

    /**
      Returns whether the plugin provides a part that should be shown in the sidebar.
    */
    virtual bool showInSideBar() const;

    /**
      Set if the plugin provides a part that should be shown in the sidebar.
    */
    void setShowInSideBar( bool hasPart );

    /**
      Reimplement this method if you want to add checks before closing down the main kontact
      window.  Return true if it's OK to close the window.  If any loaded plugin returns false
      from this method, then the main kontact window will not close.
    */
    virtual bool queryClose() const { return true; }

    /**
      Retrieve the current DCOP Client for the plugin.

      The clients name is taken from the name argument in the constructor.
      @note: The DCOPClient object will only be created when this method is
      called for the first time. Make sure that the part has been loaded
      before calling this method, if it's the one that contains the DCOP
      interface that other parts might use.
    */
    DCOPClient *dcopClient() const;

    /**
      Return the weight of the plugin. The higher the weight the lower it will
      be displayed in the sidebar. The default implementation returns 0.
    */
    virtual int weight() const { return 0; }

    /**
      Insert "New" action.
    */
    void insertNewAction( KAction *action );

    /**
      Insert "Sync" action.
    */
    void insertSyncAction( KAction *action );

    /**
      FIXME: write API doc for Kontact::Plugin::newActions().
    */
    TQPtrList<KAction>* newActions() const;

    /**
      FIXME: write API doc for Kontact::Plugin::syncActions().
    */
    TQPtrList<KAction>* syncActions() const;

    /**
      Returns a list of action name which shall be hidden in the main toolbar.
     */
    virtual TQStringList invisibleToolbarActions() const { return TQStringList(); }

    /**
      Return, if the plugin can handle the drag object of the given mime type.
    */
    virtual bool canDecodeDrag( TQMimeSource * ) { return false; }

    /**
      Process drop event.
    */
    virtual void processDropEvent( TQDropEvent * ) {}

    virtual void loadProfile( const TQString& directoryPath );

    virtual void saveToProfile( const TQString& directoryPath ) const;

    /**
     * Session management: read properties
     */
    virtual void readProperties( KConfig * ) {}

    /**
     * Session management: save properties
     */
    virtual void saveProperties( KConfig * ) {}

    Core *core() const;

    bool disabled() const;
    void setDisabled( bool v );

  public slots:
    /**
      internal usage
     */
    void slotConfigUpdated();

  protected:
    /**
      Reimplement and return the part here. Reimplementing createPart() is
      mandatory!
    */
    virtual KParts::ReadOnlyPart *createPart() = 0;

    KParts::ReadOnlyPart *loadPart();

    virtual void virtual_hook(  int id, void* data );

  private slots:
    void partDestroyed();

  private:
    class Private;
    Private *d;
};

}

#endif
