/*
 *  checkbox.h  -  check box with focus widget and read-only options
 *  Program:  kalarm
 *  Copyright © 2002,2003,2005,2006 by David Jarvie <software@astrojar.org.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#ifndef CHECKBOX_H
#define CHECKBOX_H

#include <tqcheckbox.h>


/**
 *  @short A TQCheckBox with focus widget and read-only options.
 *
 *  The CheckBox class is a TQCheckBox with the ability to transfer focus to another
 *  widget when checked, and with a read-only option.
 *
 *  Another widget may be specified as the focus widget for the check box. Whenever
 *  the user clicks on the check box so as to set its state to checked, focus is
 *  automatically transferred to the focus widget.
 *
 *  The widget may be set as read-only. This has the same effect as disabling it, except
 *  that its appearance is unchanged.
 *
 *  @author David Jarvie <software@astrojar.org.uk>
 */
class CheckBox : public TQCheckBox
{
		Q_OBJECT
  TQ_OBJECT
	public:
		/** Constructor.
		 *  @param parent The parent object of this widget.
		 *  @param name The name of this widget.
		 */
		explicit CheckBox(TQWidget* parent, const char* name = 0);
		/** Constructor.
		 *  @param text Text to display.
		 *  @param parent The parent object of this widget.
		 *  @param name The name of this widget.
		 */
		CheckBox(const TQString& text, TQWidget* parent, const char* name = 0);
		/** Returns true if the widget is read only. */
		bool         isReadOnly() const          { return mReadOnly; }
		/** Sets whether the check box is read-only for the user. If read-only,
		 *  its state cannot be changed by the user.
		 *  @param readOnly True to set the widget read-only, false to set it read-write.
		 */
		virtual void setReadOnly(bool readOnly);
		/** Returns the widget which receives focus when the user selects the check box by clicking on it. */
		TQWidget*     focusWidget() const         { return mFocusWidget; }
		/** Specifies a widget to receive focus when the user selects the check box by clicking on it.
		 *  @param widget Widget to receive focus.
		 *  @param enable If true, @p widget will be enabled before receiving focus. If
		 *                false, the enabled state of @p widget will be left unchanged when
		 *                the check box is clicked.
		 */
		void         setFocusWidget(TQWidget* widget, bool enable = true);
	protected:
		virtual void mousePressEvent(TQMouseEvent*);
		virtual void mouseReleaseEvent(TQMouseEvent*);
		virtual void mouseMoveEvent(TQMouseEvent*);
		virtual void keyPressEvent(TQKeyEvent*);
		virtual void keyReleaseEvent(TQKeyEvent*);
	protected slots:
		void         slotClicked();
	private:
		TQ_FocusPolicy mFocusPolicy;       // default focus policy for the TQCheckBox
		TQWidget*             mFocusWidget;       // widget to receive focus when button is clicked on
		bool                 mFocusWidgetEnable; // enable focus widget before setting focus
		bool                 mReadOnly;          // value cannot be changed
};

#endif // CHECKBOX_H
