/*
 * socks.h - SOCKS5 TCP proxy client/server
 * Copyright (C) 2003  Justin Karneges
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef CS_SOCKS_H
#define CS_SOCKS_H

#include"bytestream.h"

// CS_NAMESPACE_BEGIN

class TQHostAddress;
class SocksClient;
class SocksServer;

class SocksUDP : public TQObject
{
	Q_OBJECT
  TQ_OBJECT
public:
	~SocksUDP();

	void change(const TQString &host, int port);
	void write(const TQByteArray &data);

signals:
	void packetReady(const TQByteArray &data);

private slots:
	void sn_activated(int);

private:
	class Private;
	Private *d;

	friend class SocksClient;
	SocksUDP(SocksClient *sc, const TQString &host, int port, const TQHostAddress &routeAddr, int routePort);
};

class SocksClient : public ByteStream
{
	Q_OBJECT
  TQ_OBJECT
public:
	enum Error { ErrConnectionRefused = ErrCustom, ErrHostNotFound, ErrProxyConnect, ErrProxyNeg, ErrProxyAuth };
	enum Method { AuthNone=0x0001, AuthUsername=0x0002 };
	enum Request { ReqConnect, ReqUDPAssociate };
	SocksClient(TQObject *parent=0);
	SocksClient(int, TQObject *parent=0);
	~SocksClient();

	bool isIncoming() const;

	// outgoing
	void setAuth(const TQString &user, const TQString &pass="");
	void connectToHost(const TQString &proxyHost, int proxyPort, const TQString &host, int port, bool udpMode=false);

	// incoming
	void chooseMethod(int);
	void authGrant(bool);
	void requestDeny();
	void grantConnect();
	void grantUDPAssociate(const TQString &relayHost, int relayPort);

	// from ByteStream
	bool isOpen() const;
	void close();
	void write(const TQByteArray &);
	TQByteArray read(int bytes=0);
	int bytesAvailable() const;
	int bytesToWrite() const;

	// remote address
	TQHostAddress peerAddress() const;
	TQ_UINT16 peerPort() const;

	// udp
	TQString udpAddress() const;
	TQ_UINT16 udpPort() const;
	SocksUDP *createUDP(const TQString &host, int port, const TQHostAddress &routeAddr, int routePort);

signals:
	// outgoing
	void connected();

	// incoming
	void incomingMethods(int);
	void incomingAuth(const TQString &user, const TQString &pass);
	void incomingConnectRequest(const TQString &host, int port);
	void incomingUDPAssociateRequest();

private slots:
	void sock_connected();
	void sock_connectionClosed();
	void sock_delayedCloseFinished();
	void sock_readyRead();
	void sock_bytesWritten(int);
	void sock_error(int);
	void serve();

private:
	class Private;
	Private *d;

	void init();
	void reset(bool clear=false);
	void do_request();
	void processOutgoing(const TQByteArray &);
	void processIncoming(const TQByteArray &);
	void continueIncoming();
	void writeData(const TQByteArray &a);
};

class SocksServer : public TQObject
{
	Q_OBJECT
  TQ_OBJECT
public:
	SocksServer(TQObject *parent=0);
	~SocksServer();

	bool isActive() const;
	bool listen(TQ_UINT16 port, bool udp=false);
	void stop();
	int port() const;
	TQHostAddress address() const;
	SocksClient *takeIncoming();

	void writeUDP(const TQHostAddress &addr, int port, const TQByteArray &data);

signals:
	void incomingReady();
	void incomingUDP(const TQString &host, int port, const TQHostAddress &addr, int sourcePort, const TQByteArray &data);

private slots:
	void connectionReady(int);
	void connectionError();
	void sn_activated(int);

private:
	class Private;
	Private *d;
};

// CS_NAMESPACE_END

#endif
