/*
    ircchannelcontact.h - IRC Channel Contact

    Copyright (c) 2002      by Nick Betcher <nbetcher@kde.org>
    Copyright (c) 2003      by Jason Keirstead <jason@keirstead.org>

    Kopete    (c) 2002      by the Kopete developers <kopete-devel@kde.org>

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/

#ifndef IRCCHANNELCONTACT_H
#define IRCCHANNELCONTACT_H

#include "irccontact.h"

class KActionCollection;
class KAction;
class KActionMenu;
class KCodecAction;
class KToggleAction;

namespace Kopete { class MetaContact; }
namespace Kopete { class ChatSession; }
namespace Kopete { class Message; }
class KopeteView;

class IRCAccount;
class IRCContactManager;

/**
 * @author Jason Keirstead <jason@keirstead.org>
 *
 * This class is the @ref Kopete::Contact object representing IRC Channels, not users.
 * It is derived from IRCContact where much of its functionality is shared with @ref IRCUserContact.
 */
class IRCChannelContact
	: public IRCContact
{
	friend class IRCSignalMapper;

	Q_OBJECT
  TQ_OBJECT

public:
	IRCChannelContact(IRCContactManager *, const TQString &channel, Kopete::MetaContact *metac);
	~IRCChannelContact();

	/**
	 * Returns the current topic for this channel.
	 */
	 const TQString &topic() const { return mTopic; };

	/* Set password for a channel */
	void setPassword(const TQString &password) { mPassword = password; }
	/* Get password for a channel */
	const TQString &password() const { return mPassword; }

	/**
	 * Returns if a mode is enabled for this channel.
	 * @param mode The mode you want to check ( 't', 'n', etc. )
	 * @param value This is a pointer to a TQString which is set to
	 * the value of the mode if it has one. Example, the mode 'l' or
	 * the mode 'k'. If the mode has no such value then the pointer
	 * is always returned null.
	 */
	bool modeEnabled( TQChar mode, TQString *value = 0 );

	// Kopete::Contact stuff
	virtual TQPtrList<KAction> *customContextMenuActions();
	virtual const TQString caption() const;

	//Methods handled by the signal mapper
	void userJoinedChannel(const TQString &user);
	void userPartedChannel(const TQString &user, const TQString &reason);
	void userKicked(const TQString &nick, const TQString &nickKicked, const TQString &reason);
	void channelTopic(const TQString &topic);
	void channelHomePage(const TQString &url);
	void topicChanged(const TQString &nick, const TQString &newtopic);
	void topicUser(const TQString &nick, const TQDateTime &time);
	void namesList(const TQStringList &nicknames);
	void endOfNames();
	void incomingModeChange(const TQString &nick, const TQString &mode);
	void incomingChannelMode(const TQString &mode, const TQString &params );
	void failedChankey();
	void failedChanBanned();
	void failedChanInvite();
	void failedChanFull();
	void newAction(const TQString &from, const TQString &action);

public slots:
	void updateStatus();

	/**
	 * Sets the topic of this channel
	 * @param topic The topic you want set
	 */
	void setTopic( const TQString &topic = TQString() );

	/**
	 * Sets or unsets a mode on this channel
	 * @param mode The full text of the mode change you want performed
	 */
	void setMode( const TQString &mode = TQString() );

	void part();
	void partAction();
	void join();

protected slots:
	void chatSessionDestroyed();

	virtual void privateMessage(IRCContact *from, IRCContact *to, const TQString &message);
	virtual void initConversation();

private slots:
	void slotIncomingUserIsAway( const TQString &nick, const TQString &reason );
	void slotModeChanged();
	void slotAddNicknames();
	void slotConnectedToServer();
	void slotUpdateInfo();
	void slotHomepage();
	void slotChannelListed(const TQString &channel, uint members, const TQString &topic);
	void slotOnlineStatusChanged(Kopete::Contact *c, const Kopete::OnlineStatus &status, const Kopete::OnlineStatus &oldStatus);

private:
	KAction *actionJoin;
	KAction *actionPart;
	KAction *actionTopic;
	KAction *actionHomePage;
	KActionMenu *actionModeMenu;
	KCodecAction *codecAction;

	KToggleAction *actionModeT;    // Only Operators Can Change Topic
	KToggleAction *actionModeN;    // No Outside Messages
	KToggleAction *actionModeS;    // Secret
	KToggleAction *actionModeI;    // Invite Only
	KToggleAction *actionModeM;    // Moderated

	TQString mTopic;
	TQString mPassword;
	TQStringList mJoinedNicks;
	TQMap<TQString, bool> modeMap;
	TQTimer *mInfoTimer;

	void toggleMode( TQChar mode, bool enabled, bool update );
	void toggleOperatorActions( bool enabled );
};

#endif
