
/*
   Copyright (c) 2003,2004,2005 Clarence Dang <dang@kde.org>
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   1. Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.

   THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
   IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
   OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
   IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
   INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
   NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
   THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/


#ifndef __kpviewmanager_h__
#define __kpviewmanager_h__

#include <tqcursor.h>
#include <tqobject.h>
#include <tqptrlist.h>
#include <tqrect.h>


class TQPixmap;
class TQRect;
class TQTimer;

class kpDocument;
class kpView;
class kpMainWindow;
class kpTempPixmap;

class kpViewManager : public TQObject
{
Q_OBJECT
  TQ_OBJECT

public:
    kpViewManager (kpMainWindow *mainWindow);
    ~kpViewManager ();


    //
    // Registering views
    //

    void registerView (kpView *view);
    void unregisterView (kpView *view);
    void unregisterAllViews ();


    //
    // Temp Pixmap
    //

    const kpTempPixmap *tempPixmap () const;
    void setTempPixmap (const kpTempPixmap &tempPixmap);
    void invalidateTempPixmap ();


    //
    // Selections
    //

    bool selectionBorderVisible () const;
    void setSelectionBorderVisible (bool yes = true);

    bool selectionBorderFinished () const;
    void setSelectionBorderFinished (bool yes = true);


    //
    // Text Cursor
    //

    bool textCursorEnabled () const;
    void setTextCursorEnabled (bool yes = true);

    int textCursorRow () const;
    int textCursorCol () const;
    void setTextCursorPosition (int row, int col, bool isUpdateMicroFocusHint = false);

    bool textCursorBlinkState () const;
    void setTextCursorBlinkState (bool on = true);

protected:
    void updateTextCursor ();

    TQTimer *m_textCursorBlinkTimer;
    int m_textCursorRow, m_textCursorCol;
    bool m_textCursorBlinkState;

protected slots:
    void slotTextCursorBlink ();

public:

    //
    // Cursors
    //

    void setCursor (const TQCursor &cursor);
    void unsetCursor ();


    //
    // View
    //

    kpView *viewUnderCursor (bool usingTQt = false) const;

    //
    // TQWidget::hasMouse() is unreliable:
    //
    // "bool TQWidget::hasMouse () const
    //  ... See the "underMouse" property for details.
    //         .
    //         .
    //         .
    //  bool underMouse
    //  ... This value is not updated properly during drag and drop operations."
    //
    // i.e. it's possible that hasMouse() returns false in a mousePressEvent()!
    //
    // This hack needs to be called from kpView so that viewUnderCursor() works
    // as a reasonable replacement (although there is at least one case where
    // it still won't work - just after a fake drag onto the view).
    //
    void setViewUnderCursor (kpView *view);


    // Returns a pointer to the view that has keyboard focus or else, 0
    // TODO: rename to "anActiveView()" or "aViewIsActive()" as more than
    //       1 view can be active at the same time?
    kpView *activeView () const;


    // Specifies whether KolourPaint will queue _all_ paint events
    // (generated by you or the window system), until the
    // corresponding call to restoreQueueUpdates().  Use this
    // before multiple, big, non-interactive changes to the
    // document to eliminate virtually all flicker.
    //
    // This is better than TQWidget::setUpdatesEnabled() because
    // restoreQueueUpdates() automatically restores only the regions
    // of the views that need to be repainted, per view.
    bool queueUpdates () const;
    void setQueueUpdates ();
    void restoreQueueUpdates ();

    // Controls behaviour of updateViews():
    //
    // Slow: Let TQt buffer paint events via TQWidget::update().
    //       Results in less flicker.  Paint events are probably merged
    //       so long-term efficiency is increased at the expense of
    //       reduced responsiveness (default).
    // Fast: Force TQt to redraw immediately.  No paint events
    //       are merged so there is great potential for flicker,
    //       if used inappropriately.  Use this when the redraw
    //       area is small and KolourPaint's responsiveness is
    //       critical. Continual use of this mode can result in
    //       unnecessary redraws and incredibly slugish performance.
    bool fastUpdates () const;
    void setFastUpdates ();
    void restoreFastUpdates ();

private:
    int m_queueUpdatesCounter, m_fastUpdatesCounter;

public slots:
    // updating views
    void updateView (kpView *v);
    void updateView (kpView *v, const TQRect &viewRect);
    void updateView (kpView *v, int x, int y, int w, int h);
    void updateView (kpView *v, const TQRegion &viewRegion);
    void updateViewRectangleEdges (kpView *v, const TQRect &viewRect);

    void updateViews ();
    void updateViews (const TQRect &docRect);
    void updateViews (int x, int y, int w, int h);

    void adjustViewsToEnvironment ();

private:
    // don't use
    kpViewManager (const kpViewManager &);
    bool operator= (const kpViewManager &);

    kpDocument *document () const;

    kpMainWindow *m_mainWindow;
    TQPtrList <kpView> m_views;
    TQCursor m_cursor;

    kpTempPixmap *m_tempPixmap;
    kpView *m_viewUnderCursor;

    bool m_selectionBorderVisible;
    bool m_selectionBorderFinished;

    // There is no need to maintain binary compatibility at this stage.
    // The d-pointer is just so that you can experiment without recompiling
    // the kitchen sink.
    class kpViewManagerPrivate *d;
};

#endif  // __kpviewmanager_h__
