/*
 * googlefetcherdialog.cpp
 *
 * Copyright (C) 2004 Nathan Toone <nathan@toonetown.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include <kapplication.h>
#include <kio/netaccess.h>
#include <klocale.h>
#include <kdebug.h>
#include <kmessagebox.h>
#include <kcombobox.h>

#include <tqhbox.h>
#include <tqimage.h>
#include <tqlabel.h>
#include <tqpushbutton.h>
#include <tqeventloop.h>

#include "googlefetcherdialog.h"

GoogleFetcherDialog::GoogleFetcherDialog(const TQString &name,
                                         const GoogleImageList &imageList,
                                         const TQString &artist,
                                         const TQString &album,
                                         TQWidget *parent) :
    KDialogBase(parent, name.latin1(), true, TQString(),
                Ok | Cancel | User1 , NoDefault, true),
    m_pixmap(TQPixmap()),
    m_imageList(imageList),
    m_takeIt(false),
    m_newSearch(false)
{
    m_artist = artist;
    m_album = album;
    disableResize();

    TQHBox *mainBox = new TQHBox(this);
    m_iconWidget = new KIconView(mainBox);
    m_iconWidget->setResizeMode(TQIconView::Adjust);
    m_iconWidget->setSelectionMode(TQIconView::Extended);
    m_iconWidget->setSpacing(10);
    m_iconWidget->setMode(KIconView::Execute);
    m_iconWidget->setFixedSize(500,550);
    m_iconWidget->arrangeItemsInGrid();
    m_iconWidget->setItemsMovable(FALSE);

    TQHBox *imgSize = new TQHBox(actionButton(User1)->parentWidget());
    //TQLabel *label = new TQLabel(imgSize);
    //label->setText(i18n("Image size:"));

    KComboBox *combo = new KComboBox(imgSize);
    combo->insertItem(i18n("All Sizes"));
    combo->insertItem(i18n("Very Small"));
    combo->insertItem(i18n("Small"));
    combo->insertItem(i18n("Medium"));
    combo->insertItem(i18n("Large"));
    combo->insertItem(i18n("Very Large"));
    combo->setCurrentItem(0);
    connect(combo, TQT_SIGNAL(activated(int)), this, TQT_SLOT(imgSizeChanged(int)));
    connect(m_iconWidget, TQT_SIGNAL( executed(TQIconViewItem*) ), this, TQT_SLOT(slotOk()));

    imgSize->adjustSize();
    setMainWidget(mainBox);
    setButtonText(User1, i18n("New Search"));
}

GoogleFetcherDialog::~GoogleFetcherDialog()
{

}

void GoogleFetcherDialog::setLayout()
{
    setCaption(TQString("%1 - %2 (%3)")
              .arg(m_artist)
              .arg(m_album)
              .arg(m_imageList.size()));

    m_iconWidget->clear();
    for(uint i = 0; i < m_imageList.size(); i++)
        new CoverIconViewItem(m_iconWidget, m_imageList[i]);

    adjustSize();
}

void GoogleFetcherDialog::setImageList(const GoogleImageList &imageList)
{
    m_imageList=imageList;
}

////////////////////////////////////////////////////////////////////////////////
// public slots
////////////////////////////////////////////////////////////////////////////////

void GoogleFetcherDialog::refreshScreen(GoogleImageList &imageList)
{
    setImageList(imageList);
    setLayout();
}

int GoogleFetcherDialog::exec()
{
    setLayout();
    return KDialogBase::exec();
}

void GoogleFetcherDialog::slotOk()
{
    uint selectedIndex = m_iconWidget->index(m_iconWidget->currentItem());
    m_pixmap = pixmapFromURL(m_imageList[selectedIndex].imageURL());

    if(m_pixmap.isNull()) {
        KMessageBox::sorry(this,
                           i18n("The cover you have selected is unavailable. Please select another."),
                           i18n("Cover Unavailable"));
        TQPixmap blankPix;
        blankPix.resize(80, 80);
        blankPix.fill();
        m_iconWidget->currentItem()->setPixmap(blankPix, true, true);
        return;
    }

    m_takeIt = true;
    m_newSearch = false;
    hide();
}

void GoogleFetcherDialog::slotCancel()
{
    m_takeIt = true;
    m_newSearch = false;
    m_pixmap = TQPixmap();
    hide();
}

void GoogleFetcherDialog::slotUser1()
{
    m_takeIt = false;
    m_newSearch = true;
    m_pixmap = TQPixmap();
    hide();
}

void GoogleFetcherDialog::imgSizeChanged(int index)
{
    GoogleFetcher::ImageSize imageSize = GoogleFetcher::All;
    switch (index) {
        case 1:
            imageSize = GoogleFetcher::Icon;
            break;
        case 2:
            imageSize = GoogleFetcher::Small;
            break;
        case 3:
            imageSize = GoogleFetcher::Medium;
            break;
        case 4:
            imageSize=GoogleFetcher::Large;
            break;
        case 5:
            imageSize=GoogleFetcher::XLarge;
            break;
        default:
            break;
    }
    emit sizeChanged(imageSize);
}

TQPixmap GoogleFetcherDialog::fetchedImage(uint index) const
{
    return (index > m_imageList.count()) ? TQPixmap() : pixmapFromURL(m_imageList[index].imageURL());
}

TQPixmap GoogleFetcherDialog::pixmapFromURL(const KURL &url) const
{
    TQString file;

    if(KIO::NetAccess::download(url, file, 0)) {
        TQPixmap pixmap = TQPixmap(file);
        KIO::NetAccess::removeTempFile(file);
        return pixmap;
    }
    KIO::NetAccess::removeTempFile(file);
    return TQPixmap();
}

////////////////////////////////////////////////////////////////////////////////
// CoverIconViewItem
////////////////////////////////////////////////////////////////////////////////

CoverIconViewItem::CoverIconViewItem(TQIconView *parent, const GoogleImage &image) :
    TQObject(parent), KIconViewItem(parent, parent->lastItem(), image.size()), m_job(0)
{
    // Set up the iconViewItem

    TQPixmap mainMap;
    mainMap.resize(80, 80);
    mainMap.fill();
    setPixmap(mainMap, true, true);

    // Start downloading the image.

    m_job = KIO::get(image.thumbURL(), false, false);
    connect(m_job, TQT_SIGNAL(result(KIO::Job *)), this, TQT_SLOT(imageResult(KIO::Job *)));
    connect(m_job, TQT_SIGNAL(data(KIO::Job *, const TQByteArray &)),
            this, TQT_SLOT(imageData(KIO::Job *, const TQByteArray &)));
}

CoverIconViewItem::~CoverIconViewItem()
{
    if(m_job) {
        m_job->kill();

        // Drain results issued by KIO before being deleted,
        // and before deleting the job.
        kapp->eventLoop()->processEvents(TQEventLoop::ExcludeUserInput);

        delete m_job;
    }
}

void CoverIconViewItem::imageData(KIO::Job *, const TQByteArray &data)
{
    int currentSize = m_buffer.size();
    m_buffer.resize(currentSize + data.size(), TQGArray::SpeedOptim);
    memcpy(&(m_buffer.data()[currentSize]), data.data(), data.size());
}

void CoverIconViewItem::imageResult(KIO::Job *job)
{
    if(job->error())
        return;

    TQPixmap iconImage(m_buffer);
    iconImage = TQImage(iconImage.convertToImage()).smoothScale(80, 80);
    setPixmap(iconImage, true, true);
}

#include "googlefetcherdialog.moc"
