/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 * 
 * Date        : 2006-05-15
 * Description : a dialog to see preview ICC color correction 
 *               before to apply color profile.
 * 
 * Copyright (C) 2006-2007 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// TQt includes.

#include <tqlabel.h>
#include <tqwhatsthis.h>
#include <tqlayout.h>
#include <tqframe.h>
#include <tqstring.h>
#include <tqfileinfo.h>
#include <tqpushbutton.h>

// KDE includes.

#include <klocale.h>
#include <kiconloader.h>
#include <kapplication.h>
#include <kseparator.h>

// Local includes.

#include "ddebug.h"
#include "dimg.h"
#include "icctransform.h"
#include "iccprofileinfodlg.h"
#include "colorcorrectiondlg.h"
#include "colorcorrectiondlg.moc"

namespace Digikam
{

ColorCorrectionDlg::ColorCorrectionDlg(TQWidget* parent, DImg *preview, 
                                       IccTransform *iccTrans, const TQString& file)
                  : KDialogBase(parent, "", true, TQString(),
                                Help|Ok|Apply|Cancel, Ok, true)
{
    m_iccTrans = iccTrans;
    m_parent   = parent;
    setHelp("iccprofile.anchor", "digikam");
    setButtonText(Ok,     i18n("Convert"));
    setButtonTip(Ok,      i18n("Apply the default color workspace profile to the image"));
    setButtonText(Cancel, i18n("Do Nothing"));
    setButtonTip(Cancel,  i18n("Do not change the image"));
    setButtonText(Apply,  i18n("Assign"));
    setButtonTip(Apply,   i18n("Only embed the color workspace profile in the image, don't change the image"));

    TQFileInfo fi(file);
    setCaption(fi.fileName());
    
    TQWidget *page     = new TQWidget(this);
    TQGridLayout* grid = new TQGridLayout(page, 3, 2, 0, KDialog::spacingHint());
        
    TQLabel *originalTitle         = new TQLabel(i18n("Original Image:"), page);
    TQLabel *previewOriginal       = new TQLabel(page);
    TQLabel *targetTitle           = new TQLabel(i18n("Corrected Image:"), page);
    TQLabel *previewTarget         = new TQLabel(page);
    TQLabel *logo                  = new TQLabel(page);
    TQLabel *message               = new TQLabel(page);
    TQLabel *currentProfileTitle   = new TQLabel(i18n("Current workspace color profile:"), page);
    TQLabel *currentProfileDesc    = new TQLabel(TQString("<b>%1</b>").arg(m_iccTrans->getOutpoutProfileDescriptor()), page);
    TQPushButton *currentProfInfo  = new TQPushButton(i18n("Info..."), page);
    TQLabel *embeddedProfileTitle  = new TQLabel(i18n("Embedded color profile:"), page);
    TQLabel *embeddedProfileDesc   = new TQLabel(TQString("<b>%1</b>").arg(m_iccTrans->getEmbeddedProfileDescriptor()), page);
    TQPushButton *embeddedProfInfo = new TQPushButton(i18n("Info..."), page);
    KSeparator *line              = new KSeparator(Qt::Horizontal, page);
    
    if (m_iccTrans->embeddedProfile().isEmpty())
    {
        message->setText(i18n("<p>This image has not been assigned a color profile.</p>"
                              "<p>Do you want to convert it to your workspace color profile?</p>"));
                              
        line->hide();
        embeddedProfileTitle->hide();
        embeddedProfileDesc->hide();
        embeddedProfInfo->hide();
    }
    else
    {
        message->setText(i18n("<p>This image has been assigned to a color profile that does not "
                              "match your default workspace color profile.</p>"
                              "<p>Do you want to convert it to your workspace color profile?</p>"));
    }
    
    previewOriginal->setPixmap(preview->convertToPixmap());
    previewTarget->setPixmap(preview->convertToPixmap(m_iccTrans));
    KIconLoader* iconLoader = KApplication::kApplication()->iconLoader();
    logo->setPixmap(iconLoader->loadIcon("digikam", KIcon::NoGroup, 128, KIcon::DefaultState, 0, true));    
    
    grid->addMultiCellWidget(originalTitle, 0, 0, 0, 0);
    grid->addMultiCellWidget(previewOriginal, 1, 1, 0, 0);
    grid->addMultiCellWidget(targetTitle, 2, 2, 0, 0);
    grid->addMultiCellWidget(previewTarget, 3, 3, 0, 0);
    
    TQVBoxLayout *vlay = new TQVBoxLayout( KDialog::spacingHint() );
    vlay->addWidget(logo);
    vlay->addWidget(message);
    
    vlay->addWidget(new KSeparator(Qt::Horizontal, page));
    vlay->addWidget(currentProfileTitle);
    vlay->addWidget(currentProfileDesc);
    
    TQHBoxLayout *hlay1 = new TQHBoxLayout( KDialog::spacingHint() );
    hlay1->addWidget(currentProfInfo);
    hlay1->addStretch();
    vlay->addLayout(hlay1);
    
    vlay->addWidget(line);
    vlay->addWidget(embeddedProfileTitle);
    vlay->addWidget(embeddedProfileDesc);    
    
    TQHBoxLayout *hlay2 = new TQHBoxLayout( KDialog::spacingHint() );
    hlay2->addWidget(embeddedProfInfo);
    hlay2->addStretch();
    vlay->addLayout(hlay2);
    vlay->addStretch();
    
    grid->addMultiCell(new TQSpacerItem(KDialog::spacingHint(), KDialog::spacingHint(), 
                       TQSizePolicy::Minimum, TQSizePolicy::Expanding), 0, 3, 1, 1);
    grid->addMultiCellLayout(vlay, 0, 3, 2, 2);
    
    setMainWidget(page);
    
    // --------------------------------------------------------------------
    
    connect(currentProfInfo, TQT_SIGNAL(clicked()),
            this, TQT_SLOT(slotCurrentProfInfo()) );
    
    connect(embeddedProfInfo, TQT_SIGNAL(clicked()),
            this, TQT_SLOT(slotEmbeddedProfInfo()) );
            
    connect(this, TQT_SIGNAL(applyClicked()), 
            this, TQT_SLOT(slotApplyClicked()));
}

ColorCorrectionDlg::~ColorCorrectionDlg()
{
}

void ColorCorrectionDlg::slotCurrentProfInfo()
{
    if (m_iccTrans->outputProfile().isEmpty())
        return;

    ICCProfileInfoDlg infoDlg(m_parent, TQString(), m_iccTrans->outputProfile());
    infoDlg.exec();
}

void ColorCorrectionDlg::slotEmbeddedProfInfo()
{
    if (m_iccTrans->embeddedProfile().isEmpty())
        return;

    ICCProfileInfoDlg infoDlg(m_parent, TQString(), m_iccTrans->embeddedProfile());
    infoDlg.exec();
}

void ColorCorrectionDlg::slotApplyClicked()
{
    DDebug() << "colorcorrectiondlg: Apply pressed" << endl;
    done(-1);
}

}  // NameSpace Digikam

