/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2006-30-08
 * Description : batch thumbnails generator
 *
 * Copyright (C) 2006-2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// C Ansi includes.

extern "C"
{
#include <unistd.h>
}

// TQt includes.

#include <tqstring.h>
#include <tqtimer.h>
#include <tqdir.h>
#include <tqfileinfo.h>
#include <tqdatetime.h>

// KDE includes.

#include <kmdcodec.h>
#include <klocale.h>
#include <kapplication.h>

// Local includes.

#include "ddebug.h"
#include "album.h"
#include "albumdb.h"
#include "albummanager.h"
#include "albumsettings.h"
#include "thumbnailjob.h"
#include "batchthumbsgenerator.h"
#include "batchthumbsgenerator.moc"

namespace Digikam
{

class BatchThumbsGeneratorPriv
{
public:

    BatchThumbsGeneratorPriv()
    {
        cancel   = false;
        thumbJob = 0;
        duration.start();
    }

    bool                      cancel;
    
    TQTime                     duration;
    
    TQGuardedPtr<ThumbnailJob> thumbJob;
};

BatchThumbsGenerator::BatchThumbsGenerator(TQWidget* parent)
                    : DProgressDlg(parent)
{
    d = new BatchThumbsGeneratorPriv;
    setValue(0);
    setCaption(i18n("Thumbnails processing"));
    setLabel(i18n("<b>Updating thumbnails database. Please wait...</b>"));
    setButtonText(i18n("&Abort"));
    TQTimer::singleShot(500, this, TQT_SLOT(slotRebuildThumbs128()));
    resize(600, 300);
}

BatchThumbsGenerator::~BatchThumbsGenerator()
{
    if (!d->thumbJob.isNull())
    {
        d->thumbJob->kill();
        d->thumbJob = 0;
    }

    delete d;
}

void BatchThumbsGenerator::slotRebuildThumbs128()
{
    setTitle(i18n("Processing small thumbs"));
    rebuildAllThumbs(128);

    connect(this, TQT_SIGNAL(signalRebuildThumbsDone()),
            this, TQT_SLOT(slotRebuildThumbs256()));
}

void BatchThumbsGenerator::slotRebuildThumbs256()
{
    setTitle(i18n("Processing large thumbs"));
    rebuildAllThumbs(256);

    disconnect(this, TQT_SIGNAL(signalRebuildThumbsDone()),
               this, TQT_SLOT(slotRebuildThumbs256()));

    connect(this, TQT_SIGNAL(signalRebuildThumbsDone()),
            this, TQT_SLOT(slotRebuildAllThumbComplete()));
}

void BatchThumbsGenerator::slotRebuildAllThumbComplete()
{
    TQTime t;
    t = t.addMSecs(d->duration.elapsed());
    setLabel(i18n("<b>The thumbnails database has been updated.</b>"));
    setTitle(i18n("Duration: %1").arg(t.toString()));
    setButtonText(i18n("&Close"));
}

void BatchThumbsGenerator::rebuildAllThumbs(int size)
{
    TQStringList allPicturesPath;
    TQString thumbCacheDir = TQDir::homeDirPath() + "/.thumbnails/";
    TQString filesFilter   = AlbumSettings::instance()->getAllFileFilter();
    bool exifRotate       = AlbumSettings::instance()->getExifRotate();
    AlbumDB *db           = AlbumManager::instance()->albumDB();
    AlbumList palbumList  = AlbumManager::instance()->allPAlbums();

    // Get all digiKam albums collection pictures path.
    
    for (AlbumList::Iterator it = palbumList.begin();
         !d->cancel && (it != palbumList.end()); ++it )
    {
        // Don't use the root album
        if ((*it)->isRoot())
            continue;

        db->beginTransaction();
        TQStringList albumItemsPath = db->getItemURLsInAlbum((*it)->id());
        db->commitTransaction();

        TQStringList pathSorted;
        for (TQStringList::iterator it2 = albumItemsPath.begin();
             !d->cancel && (it2 != albumItemsPath.end()); ++it2)
        {
            TQFileInfo fi(*it2);
            if (filesFilter.contains(fi.extension(false)))
                pathSorted.append(*it2);
        }
        
        allPicturesPath += pathSorted;
    }

    setTotalSteps(allPicturesPath.count()*2);

    // Remove all current album item thumbs from disk cache.

    for (TQStringList::iterator it = allPicturesPath.begin(); 
         !d->cancel && (it != allPicturesPath.end()); ++it)
    {
        TQString uri = "file://" + TQDir::cleanDirPath(*it);
        KMD5 md5(TQFile::encodeName(uri).data());
        uri = md5.hexDigest();
    
        TQString smallThumbPath = thumbCacheDir + "normal/" + uri + ".png";
        TQString bigThumbPath   = thumbCacheDir + "large/"  + uri + ".png";

        if (size <= 128)
            ::unlink(TQFile::encodeName(smallThumbPath));
        else
            ::unlink(TQFile::encodeName(bigThumbPath));
    }

    if (!d->thumbJob.isNull())
    {
        d->thumbJob->kill();
        d->thumbJob = 0;
    }
    
    d->thumbJob = new ThumbnailJob(KURL::List(allPicturesPath), size, true, exifRotate);
       
    connect(d->thumbJob, TQT_SIGNAL(signalThumbnail(const KURL&, const TQPixmap&)),
            this, TQT_SLOT(slotRebuildThumbDone(const KURL&, const TQPixmap&)));
    
    connect(d->thumbJob, TQT_SIGNAL(signalFailed(const KURL&)),
            this, TQT_SLOT(slotRebuildThumbDone(const KURL&)));

    connect(d->thumbJob, TQT_SIGNAL(signalCompleted()),
            this, TQT_SIGNAL(signalRebuildThumbsDone()));
}

void BatchThumbsGenerator::slotRebuildThumbDone(const KURL& url, const TQPixmap& pix)
{
    addedAction(pix, url.path());
    advance(1);
}

void BatchThumbsGenerator::slotCancel()
{
    abort();
    done(Cancel);
}

void BatchThumbsGenerator::closeEvent(TQCloseEvent *e)
{
    abort();
    e->accept();
}

void BatchThumbsGenerator::abort()
{
    d->cancel = true;

    if (!d->thumbJob.isNull())
    {
        d->thumbJob->kill();
        d->thumbJob = 0;
    }

    emit signalRebuildAllThumbsDone();
}

}  // namespace Digikam
