/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2007-04-15
 * Description : a zoom bar used in status bar.
 *
 * Copyright (C) 2007 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// TQt includes.

#include <tqtoolbutton.h>
#include <tqtimer.h>
#include <tqslider.h>
#include <tqtooltip.h>
#include <tqevent.h>

// KDE includes.

#include <klocale.h>
#include <kiconloader.h>

// Local includes.

#include "thumbnailsize.h"
#include "dcursortracker.h"
#include "statuszoombar.h"
#include "statuszoombar.moc"

namespace Digikam
{

TQSliderReverseWheel::TQSliderReverseWheel(TQWidget *parent) 
                   : TQSlider(parent)
{
    // empty, we just need to re-implement wheelEvent to reverse the wheel
}

TQSliderReverseWheel::~TQSliderReverseWheel()
{
}

void TQSliderReverseWheel::wheelEvent(TQWheelEvent * e)
{
    if ( e->orientation() != orientation() && !TQT_TQRECT_OBJECT(rect()).contains(e->pos()) )
        return;

    static float offset = 0;
    static TQSlider* offset_owner = 0;
    if (offset_owner != this){
        offset_owner = this;
        offset = 0;
    }
    // note: different sign in front of e->delta vs. original implementation
    offset += e->delta()*TQMAX(pageStep(),lineStep())/120;
    if (TQABS(offset)<1)
        return;
    setValue( value() + int(offset) );
    offset -= int(offset);
    e->accept();
}

// ----------------------------------------------------------------------

class StatusZoomBarPriv
{

public:

    StatusZoomBarPriv()
    {
        zoomTracker     = 0;
        zoomMinusButton = 0;
        zoomPlusButton  = 0;
        zoomSlider      = 0;
        zoomTimer       = 0;
    }

    TQToolButton *zoomPlusButton;
    TQToolButton *zoomMinusButton;

    TQTimer      *zoomTimer;

    TQSlider     *zoomSlider;

    DTipTracker *zoomTracker;
};

StatusZoomBar::StatusZoomBar(TQWidget *parent)
             : TQHBox(parent, 0, TQt::WDestructiveClose)
{
    d = new StatusZoomBarPriv;
    setFocusPolicy(TQ_NoFocus);

    d->zoomMinusButton = new TQToolButton(this);
    d->zoomMinusButton->setAutoRaise(true);
    d->zoomMinusButton->setFocusPolicy(TQ_NoFocus);
    d->zoomMinusButton->setIconSet(SmallIconSet("viewmag-"));
    TQToolTip::add(d->zoomMinusButton, i18n("Zoom Out"));

    d->zoomSlider = new TQSliderReverseWheel(this);
    d->zoomSlider->setMinValue(ThumbnailSize::Small);
    d->zoomSlider->setMaxValue(ThumbnailSize::Huge);
    d->zoomSlider->setPageStep(ThumbnailSize::Step);
    d->zoomSlider->setValue(ThumbnailSize::Medium);
    d->zoomSlider->setOrientation(Qt::Horizontal);
    d->zoomSlider->setLineStep(ThumbnailSize::Step);
    d->zoomSlider->setMaximumHeight(fontMetrics().height()+2);    
    d->zoomSlider->setFixedWidth(120);
    d->zoomSlider->setFocusPolicy(TQ_NoFocus);

    d->zoomPlusButton = new TQToolButton(this);
    d->zoomPlusButton->setAutoRaise(true);
    d->zoomPlusButton->setIconSet(SmallIconSet("viewmag+"));
    d->zoomPlusButton->setFocusPolicy(TQ_NoFocus);
    TQToolTip::add(d->zoomPlusButton, i18n("Zoom In"));

    d->zoomTracker = new DTipTracker("", d->zoomSlider);

    // -------------------------------------------------------------

    connect(d->zoomMinusButton, TQT_SIGNAL(clicked()),
            this, TQT_SIGNAL(signalZoomMinusClicked()));

    connect(d->zoomPlusButton, TQT_SIGNAL(clicked()),
            this, TQT_SIGNAL(signalZoomPlusClicked()));

    connect(d->zoomSlider, TQT_SIGNAL(valueChanged(int)),
            this, TQT_SIGNAL(signalZoomSliderChanged(int)));

    connect(d->zoomSlider, TQT_SIGNAL(valueChanged(int)),
            this, TQT_SLOT(slotZoomSliderChanged(int)));

    connect(d->zoomSlider, TQT_SIGNAL(sliderReleased()),
            this, TQT_SLOT(slotZoomSliderReleased()));
}

StatusZoomBar::~StatusZoomBar()
{
    if (d->zoomTimer)
        delete d->zoomTimer;

    delete d->zoomTracker;
    delete d;
}

void StatusZoomBar::slotZoomSliderChanged(int)
{
    if (d->zoomTimer)
    {
        d->zoomTimer->stop();
        delete d->zoomTimer;
    }

    d->zoomTimer = new TQTimer( this );
    connect(d->zoomTimer, TQT_SIGNAL(timeout()),
            this, TQT_SLOT(slotDelayedZoomSliderChanged()) );
    d->zoomTimer->start(300, true);    
}

void StatusZoomBar::slotDelayedZoomSliderChanged()
{
    emit signalDelayedZoomSliderChanged(d->zoomSlider->value());
}

void StatusZoomBar::slotZoomSliderReleased()
{
    emit signalZoomSliderReleased(d->zoomSlider->value());
}

void StatusZoomBar::setZoomSliderValue(int v)
{
    d->zoomSlider->blockSignals(true);
    d->zoomSlider->setValue(v);
    d->zoomSlider->blockSignals(false);
}

void StatusZoomBar::setZoomTrackerText(const TQString& text)
{
    d->zoomTracker->setText(text);
}

void StatusZoomBar::setEnableZoomPlus(bool e)
{
    d->zoomPlusButton->setEnabled(e);
}

void StatusZoomBar::setEnableZoomMinus(bool e)
{
    d->zoomMinusButton->setEnabled(e);
}

}  // namespace Digikam

