/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2005-05-19
 * Description : a dialog to perform simple search in albums
 *
 * Copyright (C) 2005 by Renchi Raju <renchi@pooh.tam.uiuc.edu>
 * Copyright (C) 2006-2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

/** @file searchquickdialog.cpp */

// TQt includes.

#include <tqtimer.h>
#include <tqlayout.h>
#include <tqstringlist.h>
#include <tqdatetime.h>
#include <tqlabel.h>
#include <tqwhatsthis.h>

// KDE includes.

#include <klineedit.h>
#include <klocale.h>
#include <kurl.h>

// Local includes.

#include "ddebug.h"
#include "searchtextbar.h"
#include "searchresultsview.h"
#include "searchquickdialog.h"
#include "searchquickdialog.moc"

namespace Digikam
{

class SearchQuickDialogPriv
{
public:

    SearchQuickDialogPriv()
    {
        timer       = 0;
        searchEdit  = 0;
        nameEdit    = 0;
        resultsView = 0;
    }

    TQTimer            *timer;

    KLineEdit         *nameEdit;

    SearchTextBar     *searchEdit;

    SearchResultsView *resultsView;
};

SearchQuickDialog::SearchQuickDialog(TQWidget* parent, KURL& url)
                 : KDialogBase(Plain, i18n("Quick Search"), Help|Ok|Cancel, Ok, 
                               parent, 0, true, true), m_url(url)
{
    d = new SearchQuickDialogPriv;
    d->timer = new TQTimer(this);
    setHelp("quicksearchtool.anchor", "digikam");
    
    TQGridLayout* grid = new TQGridLayout(plainPage(), 2, 2, 0, spacingHint());
    
    TQLabel *label1 = new TQLabel("<b>" + i18n("Search:") + "</b>", plainPage());
    d->searchEdit  = new SearchTextBar(plainPage(), "SearchQuickDialogSearchEdit", i18n("Enter here your search criteria"));
    TQWhatsThis::add( d->searchEdit, i18n("<p>Enter your search criteria to find items in the album library"));
    
    d->resultsView = new SearchResultsView(plainPage());
    d->resultsView->setMinimumSize(320, 200);
    TQWhatsThis::add( d->resultsView, i18n("<p>Here you can see the items found in album library, "
                                          "using the current search criteria"));
    
    TQLabel *label2 = new TQLabel(i18n("Save search as:"), plainPage());
    d->nameEdit    = new KLineEdit(plainPage());
    d->nameEdit->setText(i18n("Last Search"));
    TQWhatsThis::add( d->nameEdit, i18n("<p>Enter the name of the current search to save in the "
                                       "\"My Searches\" view"));

    grid->addMultiCellWidget(label1, 0, 0, 0, 0);
    grid->addMultiCellWidget(d->searchEdit, 0, 0, 1, 2);
    grid->addMultiCellWidget(d->resultsView, 1, 1, 0, 2);
    grid->addMultiCellWidget(label2, 2, 2, 0, 1);
    grid->addMultiCellWidget(d->nameEdit, 2, 2, 2, 2);
    
    connect(d->searchEdit, TQT_SIGNAL(signalTextChanged(const TQString&)),
            this, TQT_SLOT(slotSearchChanged(const TQString&)));

    connect(d->resultsView, TQT_SIGNAL(signalSearchResultsMatch(bool)),
            d->searchEdit, TQT_SLOT(slotSearchResult(bool)));

    connect(d->timer, TQT_SIGNAL(timeout()),
            this, TQT_SLOT(slotTimeOut()));

    enableButtonOK(false);
    resize(configDialogSize("QuickSearch Dialog"));

    // check if we are being passed a valid url
    if (m_url.isValid())
    {
        int count = m_url.queryItem("count").toInt();
        if (count > 0)
        {
            TQStringList strList;

            for (int i=1; i<=count; i++)
            {
                TQString val = m_url.queryItem(TQString::number(i) + ".val");
                if (!strList.contains(val))
                {
                    strList.append(val);
                }
            }
            
            d->searchEdit->setText(strList.join(" "));
            d->nameEdit->setText(url.queryItem("name"));
            d->timer->start(0, true);
        }
    }
}

SearchQuickDialog::~SearchQuickDialog()
{
    saveDialogSize(("QuickSearch Dialog"));
    delete d->timer;    
    delete d;
}

void SearchQuickDialog::slotTimeOut()
{
    if (d->searchEdit->text().isEmpty())
    {
        d->resultsView->clear();
        enableButtonOK(false);
        return;
    }

    enableButtonOK(true);
    
    KURL url;
    url.setProtocol("digikamsearch");

    TQString path, num;
    int     count = 0;
    
    TQStringList textList = TQStringList::split(' ', d->searchEdit->text());
    for (TQStringList::iterator it = textList.begin(); it != textList.end(); ++it)
    {
        if (count != 0)
            path += " AND ";

        path += TQString(" %1 ").arg(count + 1);

        num = TQString::number(++count);
        url.addQueryItem(num + ".key", "keyword");
        url.addQueryItem(num + ".op", "like");
        url.addQueryItem(num + ".val", *it);
    }

    url.setPath(path);
    url.addQueryItem("name", "Live Search");
    url.addQueryItem("count", num);

    m_url = url;
    d->resultsView->openURL(url);
}

void SearchQuickDialog::slotSearchChanged(const TQString&)
{
    d->timer->start(500, true);    
}

void SearchQuickDialog::hideEvent(TQHideEvent* e)
{
    m_url.removeQueryItem("name");
    m_url.addQueryItem("name", d->nameEdit->text().isEmpty() ?
                       i18n("Last Search") : d->nameEdit->text());
    KDialogBase::hideEvent(e);
}

}  // namespace Digikam
