/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2003-04-25
 * Description : implementation to render album icon item.
 * 
 * Copyright (C) 2003-2005 by Renchi Raju <renchi@pooh.tam.uiuc.edu>
 * Copyright (C) 2003-2007 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// TQt includes.

#include <tqpainter.h>
#include <tqpixmap.h>
#include <tqpalette.h>
#include <tqstring.h>
#include <tqpen.h>
#include <tqfontmetrics.h>
#include <tqfont.h>
#include <tqdatetime.h>
#include <tqstringlist.h>

// KDE includes.

#include <kurl.h>
#include <kglobal.h>
#include <klocale.h>
#include <kio/global.h>

// Local includes.

#include "themeengine.h"
#include "thumbnailsize.h"
#include "imageinfo.h"
#include "albumsettings.h"
#include "icongroupitem.h"
#include "pixmapmanager.h"
#include "albumiconview.h"
#include "albumiconitem.h"

namespace Digikam
{

class AlbumIconItemPriv
{
public:

    AlbumIconItemPriv()
    {
        dirty = true;
        info  = 0;
        view  = 0;
    }

    bool           dirty;

    TQRect          tightPixmapRect;

    ImageInfo     *info;

    AlbumIconView *view;
};

static void dateToString(const TQDateTime& datetime, TQString& str)
{
    str = KGlobal::locale()->formatDateTime(datetime, true, false);
}

AlbumIconItem::AlbumIconItem(IconGroupItem* parent, ImageInfo* info)
             : IconItem(parent)
{
    d = new AlbumIconItemPriv;
    d->view = (AlbumIconView*) parent->iconView();
    d->info = info;
}

AlbumIconItem::~AlbumIconItem()
{
    delete d;
}

TQString AlbumIconItem::squeezedText(TQPainter* p, int width, const TQString& text)
{
    TQString fullText(text);
    fullText.replace("\n"," ");
    TQFontMetrics fm(p->fontMetrics());
    int textWidth = fm.width(fullText);
    
    if (textWidth > width) 
    {
        // start with the dots only
        TQString squeezedText = "...";
        int squeezedWidth = fm.width(squeezedText);

        // estimate how many letters we can add to the dots on both sides
        int letters = fullText.length() * (width - squeezedWidth) / textWidth;
        if (width < squeezedWidth) letters=1;
        squeezedText = fullText.left(letters) + "...";
        squeezedWidth = fm.width(squeezedText);

        if (squeezedWidth < width) 
        {
            // we estimated too short
            // add letters while text < label
            do 
            {
                letters++;
                squeezedText = fullText.left(letters) + "..."; 
                squeezedWidth = fm.width(squeezedText);
            }
            while (squeezedWidth < width);

            letters--;
            squeezedText = fullText.left(letters) + "..."; 
        }
        else if (squeezedWidth > width) 
        {
            // we estimated too long
            // remove letters while text > label
            do 
            {
                letters--;
                squeezedText = fullText.left(letters) + "...";
                squeezedWidth = fm.width(squeezedText);
            }
            while (letters && squeezedWidth > width);
        }

        if (letters >= 5) 
        {
            return squeezedText;
        }
    }
    
    return fullText;   
}

bool AlbumIconItem::isDirty()
{
    return d->dirty;
}

ImageInfo* AlbumIconItem::imageInfo() const
{
    return d->info;
}

int AlbumIconItem::compare(IconItem *item)
{
    const AlbumSettings *settings = d->view->settings();
    AlbumIconItem *iconItem = static_cast<AlbumIconItem*>(item);
    
    switch (settings->getImageSortOrder())
    {
        case(AlbumSettings::ByIName):
        {
            return d->info->name().localeAwareCompare(iconItem->d->info->name());
        }
        case(AlbumSettings::ByIPath):
        {
            return d->info->kurl().path().compare(iconItem->d->info->kurl().path());
        }
        case(AlbumSettings::ByIDate):
        {
            if (d->info->dateTime() < iconItem->d->info->dateTime())
                return -1;
            else if (d->info->dateTime() > iconItem->d->info->dateTime())
                return 1;
            else
                return 0;
        }
        case(AlbumSettings::ByISize):
        {
            int mysize(d->info->fileSize());
            int hissize(iconItem->d->info->fileSize());
            if (mysize < hissize)
                return -1;
            else if (mysize > hissize)
                return 1;
            else
                return 0;
        }
        case(AlbumSettings::ByIRating):
        {
            int myrating(d->info->rating());
            int hisrating(iconItem->d->info->rating());
            if (myrating < hisrating)
                return 1;
            else if (myrating > hisrating)
                return -1;
            else
                return 0;
        }
    }

    return 0;
}

TQRect AlbumIconItem::clickToOpenRect()
{
    if (d->tightPixmapRect.isNull())
        return rect();
    
    TQRect pixmapRect = d->tightPixmapRect;
    TQRect r          = rect();

    pixmapRect.moveBy(r.x(), r.y());
    return pixmapRect;
}

void AlbumIconItem::paintItem()
{
    TQPixmap pix;
    TQRect   r;
    const AlbumSettings *settings = d->view->settings();
    
    if (isSelected())
        pix = *(d->view->itemBaseSelPixmap());
    else
        pix = *(d->view->itemBaseRegPixmap());

    ThemeEngine* te = ThemeEngine::instance();
    
    TQPainter p(&pix);
    p.setPen(isSelected() ? te->textSelColor() : te->textRegColor());


    d->dirty = true;
    
    TQPixmap *thumbnail = d->view->pixmapManager()->find(d->info->kurl());
    if (thumbnail)
    {
        r = d->view->itemPixmapRect();
        p.drawPixmap(r.x() + (r.width()-thumbnail->width())/2,
                     r.y() + (r.height()-thumbnail->height())/2,
                     *thumbnail);
        d->tightPixmapRect.setRect(r.x() + (r.width()-thumbnail->width())/2,
                                 r.y() + (r.height()-thumbnail->height())/2,
                                 thumbnail->width(), thumbnail->height());
        d->dirty = false;
    }

    if (settings->getIconShowRating())
    {
        r = d->view->itemRatingRect();
        TQPixmap ratingPixmap = d->view->ratingPixmap();

        int rating = d->info->rating();
        
        int x, w;
        x = r.x() + (r.width() - rating * ratingPixmap.width())/2;
        w = rating * ratingPixmap.width();
        
        p.drawTiledPixmap(x, r.y(), w, r.height(), ratingPixmap);
    }    
    
    if (settings->getIconShowName())
    {
        r = d->view->itemNameRect();
        p.setFont(d->view->itemFontReg());
        p.drawText(r, TQt::AlignCenter, squeezedText(&p, r.width(),
                                                    d->info->name()));
    }

    p.setFont(d->view->itemFontCom());
    
    if (settings->getIconShowComments())
    {
        TQString comments = d->info->caption();
        
        r = d->view->itemCommentsRect();
        p.drawText(r, TQt::AlignCenter, squeezedText(&p, r.width(), comments));
    }

    p.setFont(d->view->itemFontXtra());

    if (settings->getIconShowDate())
    {
        TQDateTime date(d->info->dateTime());

        r = d->view->itemDateRect();    
        p.setFont(d->view->itemFontXtra());
        TQString str;
        dateToString(date, str);
        str = i18n("created : %1").arg(str);
        p.drawText(r, TQt::AlignCenter, squeezedText(&p, r.width(), str));
    }
    
    if (settings->getIconShowModDate())
    {
        TQDateTime date(d->info->modDateTime());

        r = d->view->itemModDateRect();    
        p.setFont(d->view->itemFontXtra());
        TQString str;
        dateToString(date, str);
        str = i18n("modified : %1").arg(str);
        p.drawText(r, TQt::AlignCenter, squeezedText(&p, r.width(), str));
    }
    
    if (settings->getIconShowResolution())
    {
        TQSize dims = d->info->dimensions();
        if (dims.isValid())
        {
            TQString mpixels, resolution;
            mpixels.setNum(dims.width()*dims.height()/1000000.0, 'f', 2);
            resolution = (!dims.isValid()) ? i18n("Unknown") : i18n("%1x%2 (%3Mpx)")
                         .arg(dims.width()).arg(dims.height()).arg(mpixels);
            r = d->view->itemResolutionRect();    
            p.drawText(r, TQt::AlignCenter, squeezedText(&p, r.width(), resolution));
        }
    }

    if (settings->getIconShowSize())
    {
        r = d->view->itemSizeRect();    
        p.drawText(r, TQt::AlignCenter,
                   squeezedText(&p, r.width(),
                                KIO::convertSize(d->info->fileSize())));
    }

    p.setFont(d->view->itemFontCom());
    p.setPen(isSelected() ? te->textSpecialSelColor() : te->textSpecialRegColor());

    if (settings->getIconShowTags())
    {
        TQString tags = d->info->tagNames().join(", ");
        
        r = d->view->itemTagRect();    
        p.drawText(r, TQt::AlignCenter, 
                   squeezedText(&p, r.width(), tags));
    }

    if (this == d->view->currentItem())
    {
        p.setPen(TQPen(isSelected() ? te->textSelColor() : te->textRegColor(),
                      0, TQt::DotLine));
        p.drawRect(1, 1, pix.width()-2, pix.height()-2);
    }
    
    p.end();
    
    r = rect();
    r = TQRect(d->view->contentsToViewport(TQPoint(r.x(), r.y())),
              TQSize(r.width(), r.height()));

    bitBlt(d->view->viewport(), r.x(), r.y(), &pix,
           0, 0, r.width(), r.height());
}

TQRect AlbumIconItem::thumbnailRect() const
{
    TQRect pixmapRect = d->view->itemPixmapRect();
    TQRect r          = rect();

    pixmapRect.moveBy(r.x(), r.y());
    return pixmapRect;
}

}  // namespace Digikam
