/***************************************************************************
 *   Copyright (C) 2005 by David Saxton                                    *
 *   david@bluehaze.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include "recentfilesaction.h"

#include <tdeconfig.h>
#include <tdepopupmenu.h>
#include <tdestandarddirs.h>
#include <kurl.h>

RecentFilesAction::RecentFilesAction( const TQString & configGroupName, const TQString& text, const TQObject* receiver, const char* slot, TQObject* parent, const char* name )
	: TDESelectAction( text, 0/*pix*/, parent, name )
{
	m_configGroupName = configGroupName;
	m_maxItems = 10;
	
	m_popup = new TDEPopupMenu;
	connect(m_popup, TQ_SIGNAL(aboutToShow()), this, TQ_SLOT(menuAboutToShow()));
	connect(m_popup, TQ_SIGNAL(activated(int)), this, TQ_SLOT(menuItemActivated(int)));
	connect( this, TQ_SIGNAL( activated( const TQString& ) ),
			 this, TQ_SLOT( itemSelected( const TQString& ) ) );

	setMenuAccelsEnabled( false );
	
	if ( receiver )
		connect( this, TQ_SIGNAL(urlSelected(const KURL &)), receiver, slot );
}


RecentFilesAction::~RecentFilesAction()
{
	delete m_popup;
}

void RecentFilesAction::addURL( const KURL& url )
{
	if ( url.isLocalFile() && !TDEGlobal::dirs()->relativeLocation("tmp", url.path()).startsWith("/"))
		return;
	
	TQString file;
	if ( url.isLocalFile() && url.ref().isNull() && url.query().isNull() )
		file = url.path();
	else
		file = url.prettyURL();
	
	TQStringList lst = items();

    // remove file if already in list
	lst.remove( file );

    // remove last item if already maxitems in list
	if( lst.count() == m_maxItems )
	{
        // remove last item
		lst.remove( lst.last() );
	}

    // add file to list
	lst.prepend( file );
	setItems( lst );
	
	saveEntries();
}


void RecentFilesAction::loadEntries()
{
	TDEConfig * config = TDEGlobal::config();
	
	TQString     key;
	TQString     value;
	TQString     oldGroup;
	TQStringList lst;

	oldGroup = config->group();

	config->setGroup( m_configGroupName );

    // read file list
	for( unsigned int i = 1 ; i <= m_maxItems ; i++ )
	{
		key = TQString( "File%1" ).arg( i );
		value = config->readPathEntry( key );

		if (!value.isNull())
			lst.append( value );
	}

    // set file
	setItems( lst );

	config->setGroup( oldGroup );
}

void RecentFilesAction::saveEntries()
{
	TDEConfig * config = TDEGlobal::config();
	
	TQString     key;
	TQString     value;
	TQString     oldGroup;
	TQStringList lst = items();

	oldGroup = config->group();

	config->deleteGroup( m_configGroupName, true );
	config->setGroup( m_configGroupName );

    // write file list
	for( unsigned int i = 1 ; i <= lst.count() ; i++ )
	{
		key = TQString( "File%1" ).arg( i );
		value = lst[ i - 1 ];
		config->writePathEntry( key, value );
	}

	config->setGroup( oldGroup );
	
	config->sync();
}

void RecentFilesAction::itemSelected( const TQString& text )
{
	emit urlSelected( KURL( text ) );
}

void RecentFilesAction::menuItemActivated( int id )
{
	emit urlSelected( KURL(m_popup->text(id)) );
}

void RecentFilesAction::menuAboutToShow()
{
	TDEPopupMenu *menu = m_popup;
	menu->clear();
	TQStringList list = items();
	TQStringList::iterator end = list.end();
	for ( TQStringList::Iterator it = list.begin(); it != end; ++it )
		menu->insertItem(*it);
}

void RecentFilesAction::slotClicked()
{
	TDEAction::slotActivated();
}

void RecentFilesAction::slotActivated(const TQString& text)
{
	TDESelectAction::slotActivated(text);
}


void RecentFilesAction::slotActivated(int id)
{
	TDESelectAction::slotActivated(id);
}


void RecentFilesAction::slotActivated()
{
	emit activated( currentItem() );
	emit activated( currentText() );
}


#include "recentfilesaction.moc"
