/****************************************************************************
**
** Copyright (C) 2012 Timothy Pearson.  All rights reserved.
**
** This file is part of the TDE Qt4 style interface
**
** This file may be used under the terms of the GNU General
** Public License versions 2.0 or 3.0 as published by the Free
** Software Foundation and appearing in the files LICENSE.GPL2
** and LICENSE.GPL3 included in the packaging of this file.
**
** This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
** INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE. Trolltech reserves all rights not granted
** herein.
**
**********************************************************************/

#include "tdeqt4converter.h"

TQt::Orientation convertQt4ToTQt3Orientation(Qt::Orientation qt4orient)
{
	TQt::Orientation tqt3orient;

	switch ( qt4orient ) {
		case Qt::Horizontal:
			tqt3orient = TQt::Horizontal;
			break;
		case Qt::Vertical:
			tqt3orient = TQt::Vertical;
			break;
	}

	return tqt3orient;
}

TQSlider::TickSetting convertQt4ToTQt3TickSetting(QSlider::TickPosition qt4ts)
{
	TQSlider::TickSetting tqt3ts;

	switch ( qt4ts ) {
		case QSlider::NoTicks:
			tqt3ts = TQSlider::NoMarks;
			break;
		case QSlider::TicksBothSides:
			tqt3ts = TQSlider::Both;
			break;
		case QSlider::TicksAbove:
			tqt3ts = TQSlider::Above;
			break;
		case QSlider::TicksBelow:
			tqt3ts = TQSlider::Below;
			break;
// 		case QSlider::TicksLeft:
// 			tqt3ts = TQSlider::Left;
// 			break;
// 		case QSlider::TicksRight:
// 			tqt3ts = TQSlider::Right;
// 			break;
	}

	return tqt3ts;
}

QColor convertTQt3ToQt4Color(TQColor tqt3color)
{
	return QColor(tqt3color.red(), tqt3color.green(), tqt3color.blue());
}

TQColor convertQt4ToTQt3Color(QColor qt4color)
{
	return TQColor(qt4color.red(), qt4color.green(), qt4color.blue());
}

QString convertTQt3ToQt4String(TQString tqt3string)
{
	return QString::fromUtf8(tqt3string.utf8().data());
}

TQString convertQt4ToTQt3String(QString qt4string)
{
	return TQString::fromUtf8(qt4string.toUtf8().data());
}

QStringList convertTQt3ToQt4StringList(TQStringList tqt3stringlist)
{
	QStringList qt4stringlist;

	for (TQStringList::Iterator it = tqt3stringlist.begin(); it != tqt3stringlist.end(); ++it) {
		qt4stringlist.append(convertTQt3ToQt4String(*it));
	}

	return qt4stringlist;
}

QSize convertTQt3ToQt4Size(TQSize tqt3size)
{
	return QSize(tqt3size.width(), tqt3size.height());
}

TQSize convertQt4ToTQt3Size(QSize qt4size)
{
	return TQSize(qt4size.width(), qt4size.height());
}

TQString generateTQt3CacheKey(QIcon qt4icon, int iconsize, bool transparent, TQColor* bgcolor)
{
	if (bgcolor) {
		return TQString("%1 %2 %3 %4 %5 %6").arg(qt4icon.cacheKey()).arg(iconsize).arg(transparent).arg(bgcolor->red()).arg(bgcolor->green()).arg(bgcolor->blue());
	}
	else {
		return TQString("%1 %2 %3 0 0 0").arg(qt4icon.cacheKey()).arg(iconsize).arg(transparent);
	}
}

TQPixmap convertQt4ToTQt3Pixmap(QPixmap qt4pixmap, bool copyTransparency, TQColor* bgcolor)
{
	TQPixmap tqtPM(qt4pixmap.width(), qt4pixmap.height(), (copyTransparency)?qt4pixmap.depth():((qt4pixmap.depth()>24)?24:qt4pixmap.depth()));
	QPixmap qtPM = QPixmap::fromX11Pixmap(tqtPM.handle(), QPixmap::ExplicitlyShared);
	qtPM.detach();

	if (copyTransparency) {
		qtPM.fill(Qt::transparent);
	}
	else {
		tqtPM.fill(*bgcolor);
	}
	QPainter qt4painter(&qtPM);
	qt4painter.drawPixmap(0, 0, qt4pixmap);
	qt4painter.end();

	return tqtPM;
}

TQRegion convertQt4ToTQt3Region(QRegion qt4region)
{
	TQRegion tqt3region;

	QRect qt4br = qt4region.boundingRect();
	// SLOW
	if (!qt4br.isNull()) {
		QBitmap regionMask(qt4br.x()+qt4br.width(), qt4br.y()+qt4br.height());
		regionMask.fill(Qt::color0);
		QPainter qt4brpainter(&regionMask);
		qt4brpainter.setClipRegion(qt4region);
		qt4brpainter.fillRect(0,0,regionMask.width(),regionMask.height(),Qt::color1);
		qt4brpainter.end();
		TQBitmap tqt3RegionMask;
		tqt3RegionMask = convertQt4ToTQt3Pixmap(regionMask);
		tqt3region = TQRegion(tqt3RegionMask);
	}

	return tqt3region;
}

TQPixmap convertQt4IconToTQt3Pixmap(QIcon qt4icon, int iconsize, TQPixmapCache* pmCache, bool copyTransparency, TQColor* bgcolor)
{
	TQString cacheKey;
	if (pmCache) {
		// See if the pixmap for this icon and size is already in the TQt3 cache
		cacheKey = generateTQt3CacheKey(qt4icon, iconsize, copyTransparency, bgcolor);
		TQPixmap tqtCachedPM;
		if (pmCache->find(cacheKey, tqtCachedPM) == true) {
			return tqtCachedPM;
		}
	}

	QSize qt4size = QSize(iconsize, iconsize);

	QPixmap qt4iconpm = qt4icon.pixmap(qt4size);
	TQPixmap tqtPM = convertQt4ToTQt3Pixmap(qt4iconpm);

	if (pmCache) {
		// Insert this pixmap into the cache
		pmCache->insert(cacheKey, tqtPM);
	}

	return tqtPM;
}

QIcon convertTQt3PixmapToQt4Icon(TQPixmap tqt3pixmap)
{
	QPixmap qt4pm = QPixmap::fromX11Pixmap(tqt3pixmap.handle(), QPixmap::ImplicitlyShared);
	qt4pm.detach();
	return QIcon(qt4pm);
}

TQIconSet convertQt4ToTQt3IconSet(QIcon qt4icon, int smallsize, int largesize, TQPixmapCache* pmCache)
{
	int i=0;
	int j=0;
	int k=0;
	QIcon::Mode qt4iconmode;
	TQIconSet::Mode tqt3iconmode;
	QIcon::State qt4iconstate;
	TQIconSet::State tqt3iconstate;
	TQIconSet::Size tqt3iconsize;
	TQSize tqt3iconsized;

	TQIconSet tqt3iconset;

	// Generate all required sizes of icon from the TQt3 settings
	TQSize smallIconSize(smallsize, smallsize);
	TQSize largeIconSize(largesize, largesize);

	for (i=0;i<3;i++) {
		for (j=0;j<2;j++) {
			for (k=0;k<2;k++) {
				if (i==0) {
					qt4iconmode = QIcon::Normal;
					tqt3iconmode = TQIconSet::Normal;
				}
				if (i==1) {
					qt4iconmode = QIcon::Disabled;
					tqt3iconmode = TQIconSet::Disabled;
				}
				if (i==2) {
					qt4iconmode = QIcon::Active;
					tqt3iconmode = TQIconSet::Active;
				}
				if (j==0) {
					qt4iconstate = QIcon::On;
					tqt3iconstate = TQIconSet::On;
				}
				if (j==1) {
					qt4iconstate = QIcon::Off;
					tqt3iconstate = TQIconSet::Off;
				}
				if (k==0) {
					tqt3iconsize = TQIconSet::Small;
					tqt3iconsized = smallIconSize;
				}
				if (k==1) {
					tqt3iconsize = TQIconSet::Large;
					tqt3iconsized = largeIconSize;
				}

				tqt3iconset.setPixmap(convertQt4IconToTQt3Pixmap(qt4icon, tqt3iconsized.width(), pmCache), tqt3iconsize, tqt3iconmode, tqt3iconstate);
			}
		}
	}

	return tqt3iconset;
}

QRect convertTQt3ToQt4Rect(TQRect tqt3rect)
{
	return QRect(tqt3rect.x(), tqt3rect.y(), tqt3rect.width(), tqt3rect.height());
}

QBrush convertTQt3ToQt4Brush(TQBrush tqt3brush)
{
	QBrush qt4brush;
	TQt::BrushStyle tqt3brushstyle = tqt3brush.style();
	Qt::BrushStyle qt4brushstyle;

	switch ( tqt3brushstyle ) {
		case TQt::NoBrush:
			qt4brushstyle = Qt::NoBrush;
			break;
		case TQt::SolidPattern:
			qt4brushstyle = Qt::SolidPattern;
			break;
		case TQt::Dense1Pattern:
			qt4brushstyle = Qt::Dense1Pattern;
			break;
		case TQt::Dense2Pattern:
			qt4brushstyle = Qt::Dense2Pattern;
			break;
		case TQt::Dense3Pattern:
			qt4brushstyle = Qt::Dense3Pattern;
			break;
		case TQt::Dense4Pattern:
			qt4brushstyle = Qt::Dense4Pattern;
			break;
		case TQt::Dense5Pattern:
			qt4brushstyle = Qt::Dense5Pattern;
			break;
		case TQt::Dense6Pattern:
			qt4brushstyle = Qt::Dense6Pattern;
			break;
		case TQt::Dense7Pattern:
			qt4brushstyle = Qt::Dense7Pattern;
			break;
		case TQt::HorPattern:
			qt4brushstyle = Qt::HorPattern;
			break;
		case TQt::VerPattern:
			qt4brushstyle = Qt::VerPattern;
			break;
		case TQt::CrossPattern:
			qt4brushstyle = Qt::CrossPattern;
			break;
		case TQt::BDiagPattern:
			qt4brushstyle = Qt::BDiagPattern;
			break;
		case TQt::FDiagPattern:
			qt4brushstyle = Qt::FDiagPattern;
			break;
		case TQt::DiagCrossPattern:
			qt4brushstyle = Qt::DiagCrossPattern;
			break;
		case TQt::CustomPattern:
			qt4brushstyle = Qt::TexturePattern;
			break;
		default:
			qt4brushstyle = Qt::NoBrush;
#if defined(QT_CHECK_RANGE)
			tqWarning( "convertTQt3ToQt4Brush: Invalid brush style %d", tqt3brushstyle );
#endif
	}
	qt4brush.setStyle(qt4brushstyle);
	TQColor qt3color = tqt3brush.color();
	qt4brush.setColor(QColor(qt3color.red(), qt3color.green(), qt3color.blue(), 255));
	if (qt4brushstyle == Qt::TexturePattern) {
		TQPixmap tqtPM = *tqt3brush.pixmap();
		QPixmap qtPM = QPixmap::fromX11Pixmap(tqtPM.handle(), QPixmap::ImplicitlyShared);
		qtPM.detach();
		qt4brush.setTexture(qtPM);
	}

	return qt4brush;
}

TQBrush convertQt4ToTQt3Brush(QBrush qt4brush)
{
	TQBrush tqt3brush;
	Qt::BrushStyle qt4brushstyle = qt4brush.style();
	TQt::BrushStyle tqt3brushstyle;

	switch ( qt4brushstyle ) {
		case Qt::NoBrush:
			tqt3brushstyle = TQt::NoBrush;
			break;
		case Qt::SolidPattern:
			tqt3brushstyle = TQt::SolidPattern;
			break;
		case Qt::Dense1Pattern:
			tqt3brushstyle = TQt::Dense1Pattern;
			break;
		case Qt::Dense2Pattern:
			tqt3brushstyle = TQt::Dense2Pattern;
			break;
		case Qt::Dense3Pattern:
			tqt3brushstyle = TQt::Dense3Pattern;
			break;
		case Qt::Dense4Pattern:
			tqt3brushstyle = TQt::Dense4Pattern;
			break;
		case Qt::Dense5Pattern:
			tqt3brushstyle = TQt::Dense5Pattern;
			break;
		case Qt::Dense6Pattern:
			tqt3brushstyle = TQt::Dense6Pattern;
			break;
		case Qt::Dense7Pattern:
			tqt3brushstyle = TQt::Dense7Pattern;
			break;
		case Qt::HorPattern:
			tqt3brushstyle = TQt::HorPattern;
			break;
		case Qt::VerPattern:
			tqt3brushstyle = TQt::VerPattern;
			break;
		case Qt::CrossPattern:
			tqt3brushstyle = TQt::CrossPattern;
			break;
		case Qt::BDiagPattern:
			tqt3brushstyle = TQt::BDiagPattern;
			break;
		case Qt::FDiagPattern:
			tqt3brushstyle = TQt::FDiagPattern;
			break;
		case Qt::DiagCrossPattern:
			tqt3brushstyle = TQt::DiagCrossPattern;
			break;
		case Qt::TexturePattern:
			tqt3brushstyle = TQt::CustomPattern;
			break;
		default:
			tqt3brushstyle = TQt::NoBrush;
#if defined(QT_CHECK_RANGE)
			tqWarning( "convertQt4ToTQt3Brush: Invalid brush style %d", tqt3brushstyle );
#endif
	}
	tqt3brush.setStyle(tqt3brushstyle);
	QColor qt4color = qt4brush.color();
	tqt3brush.setColor(TQColor(qt4color.red(), qt4color.green(), qt4color.blue()));
	if (tqt3brushstyle == TQt::CustomPattern) {
		QPixmap qt4texture = qt4brush.texture();
		TQPixmap tqtPM(qt4texture.width(), qt4texture.height(), qt4texture.depth());
		QPixmap qtPM = QPixmap::fromX11Pixmap(tqtPM.handle(), QPixmap::ExplicitlyShared);
		QPainter qt4painter(&qtPM);
		qt4painter.drawPixmap(0, 0, qt4texture);
		qt4painter.end();
		tqt3brush.setPixmap(tqtPM);
	}

	return tqt3brush;
}

QPalette convertTQt3ToQt4Palette(TQPalette tqt3palette)
{
	int i;
	QPalette qt4palette;

	// Color groups
	TQPalette::ColorGroup tqt3cgroup;
	QPalette::ColorGroup qt4cgroup;
	TQColor tqt3color;
	for (i=0;i<3;i++) {
		if (i == 0) {
			tqt3cgroup = TQPalette::Disabled;
			qt4cgroup = QPalette::Disabled;
		}
		if (i == 1) {
			tqt3cgroup = TQPalette::Active;
			qt4cgroup = QPalette::Active;
		}
		if (i == 2) {
			tqt3cgroup = TQPalette::Inactive;
			qt4cgroup = QPalette::Inactive;
		}

		// Brushes
		qt4palette.setBrush(qt4cgroup, QPalette::Background, convertTQt3ToQt4Brush(tqt3palette.brush(tqt3cgroup, TQColorGroup::Background)));
		qt4palette.setBrush(qt4cgroup, QPalette::Foreground, convertTQt3ToQt4Brush(tqt3palette.brush(tqt3cgroup, TQColorGroup::Foreground)));
		qt4palette.setBrush(qt4cgroup, QPalette::Base, convertTQt3ToQt4Brush(tqt3palette.brush(tqt3cgroup, TQColorGroup::Base)));
		qt4palette.setBrush(qt4cgroup, QPalette::Text, convertTQt3ToQt4Brush(tqt3palette.brush(tqt3cgroup, TQColorGroup::Text)));
		qt4palette.setBrush(qt4cgroup, QPalette::Button, convertTQt3ToQt4Brush(tqt3palette.brush(tqt3cgroup, TQColorGroup::Button)));
		qt4palette.setBrush(qt4cgroup, QPalette::ButtonText, convertTQt3ToQt4Brush(tqt3palette.brush(tqt3cgroup, TQColorGroup::ButtonText)));
		qt4palette.setBrush(qt4cgroup, QPalette::Light, convertTQt3ToQt4Brush(tqt3palette.brush(tqt3cgroup, TQColorGroup::Light)));
		qt4palette.setBrush(qt4cgroup, QPalette::Midlight, convertTQt3ToQt4Brush(tqt3palette.brush(tqt3cgroup, TQColorGroup::Midlight)));
		qt4palette.setBrush(qt4cgroup, QPalette::Dark, convertTQt3ToQt4Brush(tqt3palette.brush(tqt3cgroup, TQColorGroup::Dark)));
		qt4palette.setBrush(qt4cgroup, QPalette::Mid, convertTQt3ToQt4Brush(tqt3palette.brush(tqt3cgroup, TQColorGroup::Mid)));
		qt4palette.setBrush(qt4cgroup, QPalette::Shadow, convertTQt3ToQt4Brush(tqt3palette.brush(tqt3cgroup, TQColorGroup::Shadow)));
		qt4palette.setBrush(qt4cgroup, QPalette::Highlight, convertTQt3ToQt4Brush(tqt3palette.brush(tqt3cgroup, TQColorGroup::Highlight)));
		qt4palette.setBrush(qt4cgroup, QPalette::HighlightedText, convertTQt3ToQt4Brush(tqt3palette.brush(tqt3cgroup, TQColorGroup::HighlightedText)));
		qt4palette.setBrush(qt4cgroup, QPalette::BrightText, convertTQt3ToQt4Brush(tqt3palette.brush(tqt3cgroup, TQColorGroup::BrightText)));
		qt4palette.setBrush(qt4cgroup, QPalette::Link, convertTQt3ToQt4Brush(tqt3palette.brush(tqt3cgroup, TQColorGroup::Link)));
		qt4palette.setBrush(qt4cgroup, QPalette::LinkVisited, convertTQt3ToQt4Brush(tqt3palette.brush(tqt3cgroup, TQColorGroup::LinkVisited)));

		// Colors
		qt4palette.setColor(qt4cgroup, QPalette::Background, convertTQt3ToQt4Color(tqt3palette.color(tqt3cgroup, TQColorGroup::Background)));
		qt4palette.setColor(qt4cgroup, QPalette::Foreground, convertTQt3ToQt4Color(tqt3palette.color(tqt3cgroup, TQColorGroup::Foreground)));
		qt4palette.setColor(qt4cgroup, QPalette::Base, convertTQt3ToQt4Color(tqt3palette.color(tqt3cgroup, TQColorGroup::Base)));
		qt4palette.setColor(qt4cgroup, QPalette::Text, convertTQt3ToQt4Color(tqt3palette.color(tqt3cgroup, TQColorGroup::Text)));
		qt4palette.setColor(qt4cgroup, QPalette::Button, convertTQt3ToQt4Color(tqt3palette.color(tqt3cgroup, TQColorGroup::Button)));
		qt4palette.setColor(qt4cgroup, QPalette::ButtonText, convertTQt3ToQt4Color(tqt3palette.color(tqt3cgroup, TQColorGroup::ButtonText)));
		qt4palette.setColor(qt4cgroup, QPalette::Light, convertTQt3ToQt4Color(tqt3palette.color(tqt3cgroup, TQColorGroup::Light)));
		qt4palette.setColor(qt4cgroup, QPalette::Midlight, convertTQt3ToQt4Color(tqt3palette.color(tqt3cgroup, TQColorGroup::Midlight)));
		qt4palette.setColor(qt4cgroup, QPalette::Dark, convertTQt3ToQt4Color(tqt3palette.color(tqt3cgroup, TQColorGroup::Dark)));
		qt4palette.setColor(qt4cgroup, QPalette::Mid, convertTQt3ToQt4Color(tqt3palette.color(tqt3cgroup, TQColorGroup::Mid)));
		qt4palette.setColor(qt4cgroup, QPalette::Shadow, convertTQt3ToQt4Color(tqt3palette.color(tqt3cgroup, TQColorGroup::Shadow)));
		qt4palette.setColor(qt4cgroup, QPalette::Highlight, convertTQt3ToQt4Color(tqt3palette.color(tqt3cgroup, TQColorGroup::Highlight)));
		qt4palette.setColor(qt4cgroup, QPalette::HighlightedText, convertTQt3ToQt4Color(tqt3palette.color(tqt3cgroup, TQColorGroup::HighlightedText)));
		qt4palette.setColor(qt4cgroup, QPalette::BrightText, convertTQt3ToQt4Color(tqt3palette.color(tqt3cgroup, TQColorGroup::BrightText)));
		qt4palette.setColor(qt4cgroup, QPalette::Link, convertTQt3ToQt4Color(tqt3palette.color(tqt3cgroup, TQColorGroup::Link)));
		qt4palette.setColor(qt4cgroup, QPalette::LinkVisited, convertTQt3ToQt4Color(tqt3palette.color(tqt3cgroup, TQColorGroup::LinkVisited)));
	}

	return qt4palette;
}

TQPalette convertQt4ToTQt3Palette(QPalette qt4palette)
{
	int i;
	TQPalette tqt3palette;

	// Color groups
	TQPalette::ColorGroup tqt3cgroup;
	QPalette::ColorGroup qt4cgroup;
	TQColor tqt3color;
	for (i=0;i<3;i++) {
		if (i == 0) {
			tqt3cgroup = TQPalette::Disabled;
			qt4cgroup = QPalette::Disabled;
		}
		if (i == 1) {
			tqt3cgroup = TQPalette::Active;
			qt4cgroup = QPalette::Active;
		}
		if (i == 2) {
			tqt3cgroup = TQPalette::Inactive;
			qt4cgroup = QPalette::Inactive;
		}

		// Brushes
		tqt3palette.setBrush(tqt3cgroup, TQColorGroup::Background, convertQt4ToTQt3Brush(qt4palette.brush(qt4cgroup, QPalette::Background)));
		tqt3palette.setBrush(tqt3cgroup, TQColorGroup::Foreground, convertQt4ToTQt3Brush(qt4palette.brush(qt4cgroup, QPalette::Foreground)));
		tqt3palette.setBrush(tqt3cgroup, TQColorGroup::Base, convertQt4ToTQt3Brush(qt4palette.brush(qt4cgroup, QPalette::Base)));
		tqt3palette.setBrush(tqt3cgroup, TQColorGroup::Text, convertQt4ToTQt3Brush(qt4palette.brush(qt4cgroup, QPalette::Text)));
		tqt3palette.setBrush(tqt3cgroup, TQColorGroup::Button, convertQt4ToTQt3Brush(qt4palette.brush(qt4cgroup, QPalette::Button)));
		tqt3palette.setBrush(tqt3cgroup, TQColorGroup::ButtonText, convertQt4ToTQt3Brush(qt4palette.brush(qt4cgroup, QPalette::ButtonText)));
		tqt3palette.setBrush(tqt3cgroup, TQColorGroup::Light, convertQt4ToTQt3Brush(qt4palette.brush(qt4cgroup, QPalette::Light)));
		tqt3palette.setBrush(tqt3cgroup, TQColorGroup::Midlight, convertQt4ToTQt3Brush(qt4palette.brush(qt4cgroup, QPalette::Midlight)));
		tqt3palette.setBrush(tqt3cgroup, TQColorGroup::Dark, convertQt4ToTQt3Brush(qt4palette.brush(qt4cgroup, QPalette::Dark)));
		tqt3palette.setBrush(tqt3cgroup, TQColorGroup::Mid, convertQt4ToTQt3Brush(qt4palette.brush(qt4cgroup, QPalette::Mid)));
		tqt3palette.setBrush(tqt3cgroup, TQColorGroup::Shadow, convertQt4ToTQt3Brush(qt4palette.brush(qt4cgroup, QPalette::Shadow)));
		tqt3palette.setBrush(tqt3cgroup, TQColorGroup::Highlight, convertQt4ToTQt3Brush(qt4palette.brush(qt4cgroup, QPalette::Highlight)));
		tqt3palette.setBrush(tqt3cgroup, TQColorGroup::HighlightedText, convertQt4ToTQt3Brush(qt4palette.brush(qt4cgroup, QPalette::HighlightedText)));
		tqt3palette.setBrush(tqt3cgroup, TQColorGroup::BrightText, convertQt4ToTQt3Brush(qt4palette.brush(qt4cgroup, QPalette::BrightText)));
		tqt3palette.setBrush(tqt3cgroup, TQColorGroup::Link, convertQt4ToTQt3Brush(qt4palette.brush(qt4cgroup, QPalette::Link)));
		tqt3palette.setBrush(tqt3cgroup, TQColorGroup::LinkVisited, convertQt4ToTQt3Brush(qt4palette.brush(qt4cgroup, QPalette::LinkVisited)));

		// Colors
		tqt3palette.setColor(tqt3cgroup, TQColorGroup::Background, convertQt4ToTQt3Color(qt4palette.color(qt4cgroup, QPalette::Background)));
		tqt3palette.setColor(tqt3cgroup, TQColorGroup::Foreground, convertQt4ToTQt3Color(qt4palette.color(qt4cgroup, QPalette::Foreground)));
		tqt3palette.setColor(tqt3cgroup, TQColorGroup::Base, convertQt4ToTQt3Color(qt4palette.color(qt4cgroup, QPalette::Base)));
		tqt3palette.setColor(tqt3cgroup, TQColorGroup::Text, convertQt4ToTQt3Color(qt4palette.color(qt4cgroup, QPalette::Text)));
		tqt3palette.setColor(tqt3cgroup, TQColorGroup::Button, convertQt4ToTQt3Color(qt4palette.color(qt4cgroup, QPalette::Button)));
		tqt3palette.setColor(tqt3cgroup, TQColorGroup::ButtonText, convertQt4ToTQt3Color(qt4palette.color(qt4cgroup, QPalette::ButtonText)));
		tqt3palette.setColor(tqt3cgroup, TQColorGroup::Light, convertQt4ToTQt3Color(qt4palette.color(qt4cgroup, QPalette::Light)));
		tqt3palette.setColor(tqt3cgroup, TQColorGroup::Midlight, convertQt4ToTQt3Color(qt4palette.color(qt4cgroup, QPalette::Midlight)));
		tqt3palette.setColor(tqt3cgroup, TQColorGroup::Dark, convertQt4ToTQt3Color(qt4palette.color(qt4cgroup, QPalette::Dark)));
		tqt3palette.setColor(tqt3cgroup, TQColorGroup::Mid, convertQt4ToTQt3Color(qt4palette.color(qt4cgroup, QPalette::Mid)));
		tqt3palette.setColor(tqt3cgroup, TQColorGroup::Shadow, convertQt4ToTQt3Color(qt4palette.color(qt4cgroup, QPalette::Shadow)));
		tqt3palette.setColor(tqt3cgroup, TQColorGroup::Highlight, convertQt4ToTQt3Color(qt4palette.color(qt4cgroup, QPalette::Highlight)));
		tqt3palette.setColor(tqt3cgroup, TQColorGroup::HighlightedText, convertQt4ToTQt3Color(qt4palette.color(qt4cgroup, QPalette::HighlightedText)));
		tqt3palette.setColor(tqt3cgroup, TQColorGroup::BrightText, convertQt4ToTQt3Color(qt4palette.color(qt4cgroup, QPalette::BrightText)));
		tqt3palette.setColor(tqt3cgroup, TQColorGroup::Link, convertQt4ToTQt3Color(qt4palette.color(qt4cgroup, QPalette::Link)));
		tqt3palette.setColor(tqt3cgroup, TQColorGroup::LinkVisited, convertQt4ToTQt3Color(qt4palette.color(qt4cgroup, QPalette::LinkVisited)));
	}

	return tqt3palette;
}

QFont convertTQt3ToQt4Font(TQFont qt3font)
{
	QFont qt4font;

	qt4font = QFont(QString::fromUtf8(qt3font.family().utf8().data()), qt3font.pointSize(), qt3font.weight(), qt3font.italic());

	qt4font.setBold(qt3font.bold());
// 	qt4font.setCapitalization ( Capitalization caps )
	qt4font.setFamily(QString::fromUtf8(qt3font.family().utf8().data()));
	qt4font.setFixedPitch(qt3font.fixedPitch());
	qt4font.setItalic(qt3font.italic());
// 	qt4font.setKerning ( bool enable )
// 	qt4font.setLetterSpacing ( SpacingType type, qreal spacing )
	qt4font.setOverline(qt3font.overline());
	int pixelSize = qt3font.pixelSize();
	if (pixelSize < 0) {
		float pointSizeFloat = qt3font.pointSizeFloat();
		if (pointSizeFloat < 0) {
			qt4font.setPointSize(qt3font.pointSize());
		}
		else {
			qt4font.setPointSizeF(pointSizeFloat);
		}
	}
	else {
		qt4font.setPixelSize(pixelSize);
	}
// 	qt4font.setRawMode ( bool enable )
// 	qt4font.setRawName(QString::fromUtf8(qt3font.rawName().utf8().data()));
	qt4font.setStretch(qt3font.stretch());
	qt4font.setStrikeOut(qt3font.strikeOut());
// 	qt4font.setStyle ( Style style )

	TQFont::StyleHint qt3fontstylehint = qt3font.styleHint();
	QFont::StyleHint qt4fontstylehint = QFont::AnyStyle;
	if (qt3fontstylehint == TQFont::Helvetica)       qt4fontstylehint = QFont::Helvetica;
	else if (qt3fontstylehint == TQFont::SansSerif)  qt4fontstylehint = QFont::SansSerif;
	else if (qt3fontstylehint == TQFont::Times)      qt4fontstylehint = QFont::Times;
	else if (qt3fontstylehint == TQFont::Serif)      qt4fontstylehint = QFont::Serif;
	else if (qt3fontstylehint == TQFont::Courier)    qt4fontstylehint = QFont::Courier;
	else if (qt3fontstylehint == TQFont::TypeWriter) qt4fontstylehint = QFont::TypeWriter;
	else if (qt3fontstylehint == TQFont::OldEnglish) qt4fontstylehint = QFont::OldEnglish;
	else if (qt3fontstylehint == TQFont::Decorative) qt4fontstylehint = QFont::Decorative;
	else if (qt3fontstylehint == TQFont::System)     qt4fontstylehint = QFont::System;
	else if (qt3fontstylehint == TQFont::AnyStyle)   qt4fontstylehint = QFont::AnyStyle;

	TQFont::StyleStrategy qt3fontstrategy = qt3font.styleStrategy();
	QFont::StyleStrategy qt4fontstrategy = (QFont::StyleStrategy)0;
	if (qt3fontstrategy & TQFont::PreferDefault)    qt4fontstrategy = (QFont::StyleStrategy)(qt4fontstrategy | QFont::PreferDefault);
	if (qt3fontstrategy & TQFont::PreferBitmap)     qt4fontstrategy = (QFont::StyleStrategy)(qt4fontstrategy | QFont::PreferBitmap);
	if (qt3fontstrategy & TQFont::PreferDevice)     qt4fontstrategy = (QFont::StyleStrategy)(qt4fontstrategy | QFont::PreferDevice);
	if (qt3fontstrategy & TQFont::PreferOutline)    qt4fontstrategy = (QFont::StyleStrategy)(qt4fontstrategy | QFont::PreferOutline);
	if (qt3fontstrategy & TQFont::ForceOutline)     qt4fontstrategy = (QFont::StyleStrategy)(qt4fontstrategy | QFont::ForceOutline);
	if (qt3fontstrategy & TQFont::PreferMatch)      qt4fontstrategy = (QFont::StyleStrategy)(qt4fontstrategy | QFont::PreferMatch);
	if (qt3fontstrategy & TQFont::PreferQuality)    qt4fontstrategy = (QFont::StyleStrategy)(qt4fontstrategy | QFont::PreferQuality);
	if (qt3fontstrategy & TQFont::PreferAntialias)  qt4fontstrategy = (QFont::StyleStrategy)(qt4fontstrategy | QFont::PreferAntialias);
	if (qt3fontstrategy & TQFont::NoAntialias)      qt4fontstrategy = (QFont::StyleStrategy)(qt4fontstrategy | QFont::NoAntialias);
	if (qt3fontstrategy & TQFont::OpenGLCompatible) qt4fontstrategy = (QFont::StyleStrategy)(qt4fontstrategy | QFont::OpenGLCompatible);

	qt4font.setStyleHint (qt4fontstylehint, qt4fontstrategy);
// 	qt4font.setStyleStrategy ( StyleStrategy s )
	qt4font.setUnderline(qt3font.underline());
	qt4font.setWeight(qt3font.weight());
// 	qt4font.setWordSpacing ( qreal spacing )

	return qt4font;
}

TQStyle::SFlags convertQt4ToTQt3SFlags(QStyle::State qt4stateflags, TQt3WidgetType wt)
{
	// Convert sflags
	TQStyle::SFlags sflags = 0;

	if (qt4stateflags & QStyle::State_Active) {
		if ((wt != TQT3WT_TQPopupMenu) && (wt != TQT3WT_TQMenuBar)) {
			sflags = sflags | TQStyle::Style_Active;
		}
	}
	if (qt4stateflags & QStyle::State_AutoRaise) {
		sflags = sflags | TQStyle::Style_AutoRaise;
	}
// 	if (qt4stateflags & QStyle::State_Children) {
// 		sflags = sflags | TQStyle::;
// 	}
	if (qt4stateflags & QStyle::State_DownArrow) {
		sflags = sflags | TQStyle::Style_Down;
	}
// 	if (qt4stateflags & QStyle::State_Editing) {
// 		sflags = sflags | TQStyle::;
// 	}
	if (qt4stateflags & QStyle::State_Enabled) {
		sflags = sflags | TQStyle::Style_Enabled;
	}
// 	if (qt4stateflags & QStyle::State_HasEditFocus) {
// 		sflags = sflags | TQStyle::Style_HasFocus;
// 	}
	if (qt4stateflags & QStyle::State_HasFocus) {
		sflags = sflags | TQStyle::Style_HasFocus;
	}
	if (qt4stateflags & QStyle::State_Horizontal) {
		sflags = sflags | TQStyle::Style_Horizontal;
	}
// 	if (qt4stateflags & QStyle::State_KeyboardFocusChange) {
// 		sflags = sflags | TQStyle::;
// 	}
	if (qt4stateflags & QStyle::State_MouseOver) {
		sflags = sflags | TQStyle::Style_MouseOver;
		if ((wt == TQT3WT_TQToolButton) && (qt4stateflags & QStyle::State_Enabled)) {
			sflags = sflags | TQStyle::Style_Raised;
		}
	}
	if (qt4stateflags & QStyle::State_NoChange) {
		sflags = sflags | TQStyle::Style_NoChange;
	}
	if (qt4stateflags & QStyle::State_Off) {
		sflags = sflags | TQStyle::Style_Off;
	}
	if (qt4stateflags & QStyle::State_On) {
		sflags = sflags | TQStyle::Style_On;
	}
	if (qt4stateflags & QStyle::State_Raised) {
		if (wt != TQT3WT_TQToolButton) {
			sflags = sflags | TQStyle::Style_Raised;
		}
	}
// 	if (qt4stateflags & QStyle::State_ReadOnly) {
// 		sflags = sflags | TQStyle::;
// 	}
	if (qt4stateflags & QStyle::State_Selected) {
		if ((wt == TQT3WT_TQPopupMenu) || (wt == TQT3WT_TQMenuBar)) {
			sflags = sflags | TQStyle::Style_Active;
		}
		sflags = sflags | TQStyle::Style_Selected;
	}
// 	if (qt4stateflags & QStyle::State_Item) {
// 		sflags = sflags | TQStyle::;
// 	}
// 	if (qt4stateflags & QStyle::State_Open) {
// 		sflags = sflags | TQStyle::;
// 	}
// 	if (qt4stateflags & QStyle::State_Sibling) {
// 		sflags = sflags | TQStyle::;
// 	}
	if (qt4stateflags & QStyle::State_Sunken) {
		if ((wt == TQT3WT_TQPopupMenu) || (wt == TQT3WT_TQMenuBar) || (wt == TQT3WT_TQToolButton) || (wt == TQT3WT_TQRadioButton) || (wt == TQT3WT_TQCheckBox) || (wt == TQT3WT_TQPushButton)) {
			sflags = sflags | TQStyle::Style_Down;
		}
		sflags = sflags | TQStyle::Style_Sunken;
	}
	if (qt4stateflags & QStyle::State_UpArrow) {
		sflags = sflags | TQStyle::Style_Up;
	}
// 	if (qt4stateflags & QStyle::State_Mini) {
// 		sflags = sflags | TQStyle::;
// 	}
// 	if (qt4stateflags & QStyle::State_Small) {
// 		sflags = sflags | TQStyle::;
// 	}

	return sflags;
}
