// This module implements the QextScintillaLexerLua class.
//
// Copyright (c) 2006
// 	Riverbank Computing Limited <info@riverbankcomputing.co.uk>
// 
// This file is part of TQScintilla.
// 
// This copy of TQScintilla is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option) any
// later version.
// 
// TQScintilla is supplied in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
// details.
// 
// You should have received a copy of the GNU General Public License along with
// TQScintilla; see the file LICENSE.  If not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.


#include <tqcolor.h>
#include <tqfont.h>
#include <tqsettings.h>

#include "qextscintillalexerlua.h"


// The ctor.
QextScintillaLexerLua::QextScintillaLexerLua(TQObject *parent, const char *name)
	: QextScintillaLexer(parent,name), fold_compact(TRUE)
{
}


// The dtor.
QextScintillaLexerLua::~QextScintillaLexerLua()
{
}


// Returns the language name.
const char *QextScintillaLexerLua::language() const
{
	return "Lua";
}


// Returns the lexer name.
const char *QextScintillaLexerLua::lexer() const
{
	return "lua";
}


// Return the list of characters that can start a block.
const char *QextScintillaLexerLua::blockStart(int *style) const
{
	if (style)
		*style = Operator;

	return "";
}


// Return the style used for braces.
int QextScintillaLexerLua::braceStyle() const
{
	return Operator;
}


// Returns the foreground colour of the text for a style.
TQColor QextScintillaLexerLua::color(int style) const
{
	switch (style)
	{
	case Default:
		return TQColor(0x00,0x00,0x00);

	case Comment:
	case LineComment:
		return TQColor(0x00,0x7f,0x00);

	case Number:
		return TQColor(0x00,0x7f,0x7f);

	case Keyword:
	case BasicFunctions:
	case StringTableMathsFunctions:
	case CoroutinesIOSystemFacilities:
		return TQColor(0x00,0x00,0x7f);

	case String:
	case Character:
	case LiteralString:
		return TQColor(0x7f,0x00,0x7f);

	case Preprocessor:
		return TQColor(0x7f,0x7f,0x00);

	case Operator:
	case Identifier:
		break;
	}

	return QextScintillaLexer::color(style);
}


// Returns the end-of-line fill for a style.
bool QextScintillaLexerLua::eolFill(int style) const
{
	return (style == Comment || style == UnclosedString);
}


// Returns the font of the text for a style.
TQFont QextScintillaLexerLua::font(int style) const
{
	TQFont f;

	switch (style)
	{
	case Comment:
	case LineComment:
	case LiteralString:
#if defined(Q_OS_WIN)
		f = TQFont("Comic Sans MS",9);
#else
		f = TQFont("Bitstream Vera Serif",9);
#endif
		break;

	default:
		f = QextScintillaLexer::font(style);
	}

	return f;
}


// Returns the set of keywords.
const char *QextScintillaLexerLua::keywords(int set) const
{
	if (set == 1)
		// Keywords.
		return
			"and break do else elseif end false for function if "
			"in local nil not or repeat return then true until "
			"while";

	if (set == 2)
		// Basic functions.
		return
			"_ALERT _ERRORMESSAGE _INPUT _PROMPT _OUTPUT _STDERR "
			"_STDIN _STDOUT call dostring foreach foreachi getn "
			"globals newtype rawget rawset require sort tinsert "
			"tremove "

			"G getfenv getmetatable ipairs loadlib next pairs "
			"pcall rawegal rawget rawset require setfenv "
			"setmetatable xpcall string table math coroutine io "
			"os debug";

	if (set == 3)
		// String, table and maths functions.
		return
			"abs acos asin atan atan2 ceil cos deg exp floor "
			"format frexp gsub ldexp log log10 max min mod rad "
			"random randomseed sin sqrt strbyte strchar strfind "
			"strlen strlower strrep strsub strupper tan "

			"string.byte string.char string.dump string.find "
			"string.len string.lower string.rep string.sub "
			"string.upper string.format string.gfind string.gsub "
			"table.concat table.foreach table.foreachi table.getn "
			"table.sort table.insert table.remove table.setn "
			"math.abs math.acos math.asin math.atan math.atan2 "
			"math.ceil math.cos math.deg math.exp math.floor "
			"math.frexp math.ldexp math.log math.log10 math.max "
			"math.min math.mod math.pi math.rad math.random "
			"math.randomseed math.sin math.sqrt math.tan";

	if (set == 4)
		// Coroutine, I/O and system facilities.
		return
			"openfile closefile readfrom writeto appendto remove "
			"rename flush seek tmpfile tmpname read write clock "
			"date difftime execute exit getenv setlocale time "

			"coroutine.create coroutine.resume coroutine.status "
			"coroutine.wrap coroutine.yield io.close io.flush "
			"io.input io.lines io.open io.output io.read "
			"io.tmpfile io.type io.write io.stdin io.stdout "
			"io.stderr os.clock os.date os.difftime os.execute "
			"os.exit os.getenv os.remove os.rename os.setlocale "
			"os.time os.tmpname";

	return 0;
}


// Returns the user name of a style.
TQString QextScintillaLexerLua::description(int style) const
{
	switch (style)
	{
	case Default:
		return tr("Default");

	case Comment:
		return tr("Comment");

	case LineComment:
		return tr("Line comment");

	case Number:
		return tr("Number");

	case Keyword:
		return tr("Keyword");

	case String:
		return tr("String");

	case Character:
		return tr("Character");

	case LiteralString:
		return tr("Literal string");

	case Preprocessor:
		return tr("Preprocessor");

	case Operator:
		return tr("Operator");

	case Identifier:
		return tr("Identifier");

	case UnclosedString:
		return tr("Unclosed string");

	case BasicFunctions:
		return tr("Basic functions");

	case StringTableMathsFunctions:
		return tr("String, table and maths functions");

	case CoroutinesIOSystemFacilities:
		return tr("Coroutines, i/o and system facilities");
	}

	return TQString();
}


// Returns the background colour of the text for a style.
TQColor QextScintillaLexerLua::paper(int style) const
{
	switch (style)
	{
	case Comment:
		return TQColor(0xd0,0xf0,0xf0);

	case LiteralString:
		return TQColor(0xe0,0xff,0xff);

	case UnclosedString:
		return TQColor(0xe0,0xc0,0xe0);

	case BasicFunctions:
		return TQColor(0xd0,0xff,0xd0);

	case StringTableMathsFunctions:
		return TQColor(0xd0,0xd0,0xff);

	case CoroutinesIOSystemFacilities:
		return TQColor(0xff,0xd0,0xd0);
	}

	return QextScintillaLexer::paper(style);
}


// Refresh all properties.
void QextScintillaLexerLua::refreshProperties()
{
	setCompactProp();
}


// Read properties from the settings.
bool QextScintillaLexerLua::readProperties(TQSettings &qs,const TQString &prefix)
{
	int rc = TRUE;
	bool ok, flag;

	// Read the fold compact flag.
	flag = qs.readBoolEntry(prefix + "foldcompact",TRUE,&ok);

	if (ok)
		fold_compact = flag;
	else
		rc = FALSE;

	return rc;
}


// Write properties to the settings.
bool QextScintillaLexerLua::writeProperties(TQSettings &qs,const TQString &prefix) const
{
	int rc = TRUE;

	// Write the fold compact flag.
	if (!qs.writeEntry(prefix + "foldcompact",fold_compact))
		rc = FALSE;

	return rc;
}


// Return TRUE if folds are compact.
bool QextScintillaLexerLua::foldCompact() const
{
	return fold_compact;
}


// Set if folds are compact.
void QextScintillaLexerLua::setFoldCompact(bool fold)
{
	fold_compact = fold;

	setCompactProp();
}


// Set the "fold.compact" property.
void QextScintillaLexerLua::setCompactProp()
{
	emit propertyChanged("fold.compact",(fold_compact ? "1" : "0"));
}
