/*
 * qcaprovider.h - TQCA Plugin API
 * Copyright (C) 2003  Justin Karneges
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef TQCAPROVIDER_H
#define TQCAPROVIDER_H

#include<tqglobal.h>
#include<tqstring.h>
#include<tqdatetime.h>
#include<tqobject.h>
#include<tqhostaddress.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_TQCA
#include <tqca.h>
#else
#include "qca.h"
#endif

#define TQCA_PLUGIN_VERSION 1

class TQCAProvider
{
public:
	TQCAProvider() {}
	virtual ~TQCAProvider() {}

	virtual void init()=0;
	virtual int qcaVersion() const=0;
	virtual int capabilities() const=0;
	virtual void *context(int cap)=0;
};

class TQCA_HashContext
{
public:
	virtual ~TQCA_HashContext() {}

	virtual TQCA_HashContext *clone()=0;
	virtual void reset()=0;
	virtual void update(const char *in, unsigned int len)=0;
	virtual void final(TQByteArray *out)=0;
};

class TQCA_CipherContext
{
public:
	virtual ~TQCA_CipherContext() {}

	virtual TQCA_CipherContext *clone()=0;
	virtual int keySize()=0;
	virtual int blockSize()=0;
	virtual bool generateKey(char *out, int keysize=-1)=0;
	virtual bool generateIV(char *out)=0;

	virtual bool setup(int dir, int mode, const char *key, int keysize, const char *iv, bool pad)=0;
	virtual bool update(const char *in, unsigned int len)=0;
	virtual bool final(TQByteArray *out)=0;
};

class TQCA_RSAKeyContext
{
public:
	virtual ~TQCA_RSAKeyContext() {}

	virtual TQCA_RSAKeyContext *clone() const=0;
	virtual bool isNull() const=0;
	virtual bool havePublic() const=0;
	virtual bool havePrivate() const=0;
	virtual bool createFromDER(const char *in, unsigned int len)=0;
	virtual bool createFromPEM(const char *in, unsigned int len)=0;
	virtual bool createFromNative(void *in)=0;
	virtual bool generate(unsigned int bits)=0;
	virtual bool toDER(TQByteArray *out, bool publicOnly)=0;
	virtual bool toPEM(TQByteArray *out, bool publicOnly)=0;

	virtual bool encrypt(const TQByteArray &in, TQByteArray *out, bool oaep)=0;
	virtual bool decrypt(const TQByteArray &in, TQByteArray *out, bool oaep)=0;
};

struct TQCA_CertProperty
{
	TQString var;
	TQString val;
};

class TQCA_CertContext
{
public:
	virtual ~TQCA_CertContext() {}

	virtual TQCA_CertContext *clone() const=0;
	virtual bool isNull() const=0;
	virtual bool createFromDER(const char *in, unsigned int len)=0;
	virtual bool createFromPEM(const char *in, unsigned int len)=0;
	virtual bool toDER(TQByteArray *out)=0;
	virtual bool toPEM(TQByteArray *out)=0;

	virtual TQString serialNumber() const=0;
	virtual TQString subjectString() const=0;
	virtual TQString issuerString() const=0;
	virtual TQValueList<TQCA_CertProperty> subject() const=0;
	virtual TQValueList<TQCA_CertProperty> issuer() const=0;
	virtual TQDateTime notBefore() const=0;
	virtual TQDateTime notAfter() const=0;
	virtual bool matchesAddress(const TQString &realHost) const=0;
};

class TQCA_TLSContext
{
public:
	enum Result { Success, Error, Continue };
	virtual ~TQCA_TLSContext() {}

	virtual void reset()=0;
	virtual bool startClient(const TQPtrList<TQCA_CertContext> &store, const TQCA_CertContext &cert, const TQCA_RSAKeyContext &key)=0;
	virtual bool startServer(const TQPtrList<TQCA_CertContext> &store, const TQCA_CertContext &cert, const TQCA_RSAKeyContext &key)=0;

	virtual int handshake(const TQByteArray &in, TQByteArray *out)=0;
	virtual int shutdown(const TQByteArray &in, TQByteArray *out)=0;
	virtual bool encode(const TQByteArray &plain, TQByteArray *to_net, int *encoded)=0;
	virtual bool decode(const TQByteArray &from_net, TQByteArray *plain, TQByteArray *to_net)=0;
	virtual bool eof() const=0;
	virtual TQByteArray unprocessed()=0;

	virtual TQCA_CertContext *peerCertificate() const=0;
	virtual int validityResult() const=0;
};

struct TQCA_SASLHostPort
{
	TQHostAddress addr;
	TQ_UINT16 port;
};

struct TQCA_SASLNeedParams
{
	bool user, authzid, pass, realm;
};

class TQCA_SASLContext
{
public:
	enum Result { Success, Error, NeedParams, AuthCheck, Continue };
	virtual ~TQCA_SASLContext() {}

	// common
	virtual void reset()=0;
	virtual void setCoreProps(const TQString &service, const TQString &host, TQCA_SASLHostPort *local, TQCA_SASLHostPort *remote)=0;
	virtual void setSecurityProps(bool noPlain, bool noActive, bool noDict, bool noAnon, bool reqForward, bool reqCreds, bool reqMutual, int ssfMin, int ssfMax, const TQString &_ext_authid, int _ext_ssf)=0;
	virtual int security() const=0;
	virtual int errorCond() const=0;

	// init / first step
	virtual bool clientStart(const TQStringList &mechlist)=0;
	virtual int clientFirstStep(bool allowClientSendFirst)=0;
	virtual bool serverStart(const TQString &realm, TQStringList *mechlist, const TQString &name)=0;
	virtual int serverFirstStep(const TQString &mech, const TQByteArray *in)=0;

	// get / set params
	virtual TQCA_SASLNeedParams clientParamsNeeded() const=0;
	virtual void setClientParams(const TQString *user, const TQString *authzid, const TQString *pass, const TQString *realm)=0;
	virtual TQString username() const=0;
	virtual TQString authzid() const=0;

	// continue steps
	virtual int nextStep(const TQByteArray &in)=0;
	virtual int tryAgain()=0;

	// results
	virtual TQString mech() const=0;
	virtual const TQByteArray *clientInit() const=0;
	virtual TQByteArray result() const=0;

	// security layer
	virtual bool encode(const TQByteArray &in, TQByteArray *out)=0;
	virtual bool decode(const TQByteArray &in, TQByteArray *out)=0;
};

#endif
