/***************************************************************************
 *   Copyright (C) 2004 by Alexander Dymo                                  *
 *   cloudtemple@mksat.net                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/
#include "docqtplugin.h"

#include <unistd.h>

#include <tqfile.h>
#include <tqfileinfo.h>
#include <tqdialog.h>

#include <kurl.h>
#include <tdeaboutdata.h>
#include <tdeconfig.h>
#include <tdelocale.h>
#include <tdelistview.h>

#include <urlutil.h>
#include <kdevgenericfactory.h>
#include <kdevplugininfo.h>

#include "../../../../config.h"

class TQtDocumentationCatalogItem: public DocumentationCatalogItem
{
public:
    TQtDocumentationCatalogItem(const TQString &dcfFile, DocumentationPlugin* plugin,
        TDEListView *parent, TDEListViewItem *after, const TQString &name)
        :DocumentationCatalogItem(plugin, parent, after, name), m_dcfFile(dcfFile)
    {
    }
    TQtDocumentationCatalogItem(const TQString &dcfFile, DocumentationPlugin* plugin,
        DocumentationItem *parent, const TQString &name)
        :DocumentationCatalogItem(plugin, parent, name), m_dcfFile(dcfFile)
    {
    }
    TQString dcfFile() const { return m_dcfFile; }

private:
    TQString m_dcfFile;
};

static const KDevPluginInfo data("docqtplugin");
typedef KDevGenericFactory<DocQtPlugin> DocQtPluginFactory;
K_EXPORT_COMPONENT_FACTORY( libdocqtplugin, DocQtPluginFactory(data) )

DocQtPlugin::DocQtPlugin(TQObject* parent, const char* name, const TQStringList)
    :DocumentationPlugin(DocQtPluginFactory::instance()->config(), parent, name)
{
    setCapabilities(Index | FullTextSearch);
    autoSetup();
}

DocQtPlugin::~DocQtPlugin()
{
}

void DocQtPlugin::createTOC(DocumentationCatalogItem *item)
{
    TQtDocumentationCatalogItem *qtItem = dynamic_cast<TQtDocumentationCatalogItem *>(item);
    if (!qtItem)
        return;

    TQFileInfo fi(qtItem->dcfFile());

    TQFile f(qtItem->dcfFile());
    if (!f.open(IO_ReadOnly))
    {
        kdDebug(9002) << "Could not read" << qtItem->dcfFile() << endl;
        return;
    }
    TQDomDocument doc;
    if (!doc.setContent(&f) || doc.doctype().name() != "DCF")
    {
        kdDebug(9002) << "Not a valid DCF file: " << qtItem->dcfFile() << endl;
        return;
    }
    f.close();

    TQDomElement docEl = doc.documentElement();
    TQDomElement titleEl = docEl.namedItem("DCF").toElement();

    TQDomElement childEl = docEl.lastChild().toElement();
    while (!childEl.isNull())
    {
        if (childEl.tagName() == "section")
        {
            TQString ref = childEl.attribute("ref");
            TQString title = childEl.attribute("title");

            DocumentationItem *sectionItem = new DocumentationItem(DocumentationItem::Book, item, title);
            KURL sectionUrl(fi.dirPath(true) + "/" + ref);
            sectionItem->setURL(sectionUrl);

            TQDomElement grandChild = childEl.lastChild().toElement();
            while(!grandChild.isNull())
            {
                if (grandChild.tagName() == "keyword")
                {
                    TQString keyRef = grandChild.attribute("ref");
                    TQString keyTitle = grandChild.text();

                    DocumentationItem *keyItem = new DocumentationItem(DocumentationItem::Document, sectionItem, keyTitle);
                    KURL keyUrl(fi.dirPath(true) + "/" + keyRef);
                    keyItem->setURL(keyUrl);
                }
                if (grandChild.tagName() == "section")
                {
                    TQString keyRef = grandChild.attribute("ref");
                    TQString keyTitle = grandChild.attribute("title");
                    DocumentationItem *keyItem = new DocumentationItem(DocumentationItem::Book, sectionItem, keyTitle);
                    KURL keyUrl(fi.dirPath(true) + "/" + keyRef);
                    keyItem->setURL(keyUrl);
                }
                grandChild = grandChild.previousSibling().toElement();
            }
        }
        childEl = childEl.previousSibling().toElement();
    }
}

void DocQtPlugin::autoSetupPlugin()
{
    // Clear groups, to allow for re-autosetup calls
    config->deleteGroup("Index");
    config->deleteGroup("Index Settings");
    config->deleteGroup("Locations");
    config->deleteGroup("Search Settings");
    config->deleteGroup("TOC Settings");
    config->sync();
    
    TQString qtDocDir(QT_DOCDIR);
    qtDocDir = URLUtil::envExpand(qtDocDir);
    if (qtDocDir.isEmpty())
    {
        qtDocDir = getenv("QTDIR");
    }
    if (!qtDocDir.isEmpty())
    {        
        config->setGroup("Search Settings");
        config->writeEntry("TQt Reference Documentation", true);
        config->setGroup("Index Settings");
        config->writeEntry("TQt Reference Documentation", true);
        config->setGroup("Locations");
        config->writePathEntry("TQt Reference Documentation", qtDocDir + TQString("/qt.dcf"));
        config->writePathEntry("TQt Assistant Manual", qtDocDir + TQString("/assistant.dcf"));
        config->writePathEntry("TQt Designer Manual", qtDocDir + TQString("/designer.dcf"));
        config->writePathEntry("Guide to the TQt Translation Tools", qtDocDir + TQString("/linguist.dcf"));
        config->writePathEntry("qmake User Guide", qtDocDir + TQString("/qmake.dcf"));
    }
    
    config->sync();
}

void DocQtPlugin::setCatalogURL(DocumentationCatalogItem *item)
{
    TQtDocumentationCatalogItem *qtItem = dynamic_cast<TQtDocumentationCatalogItem *>(item);
    if (!qtItem)
        return;

    TQFileInfo fi(qtItem->dcfFile());

    TQFile f(qtItem->dcfFile());
    if (!f.open(IO_ReadOnly))
    {
        kdDebug(9002) << "Could not read" << qtItem->dcfFile() << endl;
        return;
    }
    TQDomDocument doc;
    if (!doc.setContent(&f) || doc.doctype().name() != "DCF")
    {
        kdDebug(9002) << "Not a valid DCF file: " << qtItem->dcfFile() << endl;
        return;
    }
    f.close();

    TQDomElement docEl = doc.documentElement();
    TQDomElement titleEl = docEl.namedItem("DCF").toElement();

    if (item->url().isEmpty())
    {
        KURL url(fi.dirPath(true) + "/" + docEl.attribute("ref", TQString()));
        item->setURL(url);
    }
}

TQString DocQtPlugin::pluginName() const
{
    return i18n("TQt Documentation Collection");
}

bool DocQtPlugin::needRefreshIndex(DocumentationCatalogItem *item)
{
    TQtDocumentationCatalogItem *qtItem = dynamic_cast<TQtDocumentationCatalogItem *>(item);
    if (!qtItem)
        return false;

    TQFileInfo fi(qtItem->dcfFile());
    config->setGroup("Index");
    if (fi.lastModified() > config->readDateTimeEntry(qtItem->text(0), new TQDateTime()))
    {
        kdDebug() << "need rebuild index for " << qtItem->text(0) << endl;
        config->writeEntry(item->text(0), fi.lastModified());
        return true;
    }
    else
        return false;
}

void DocQtPlugin::createIndex(IndexBox *index, DocumentationCatalogItem *item)
{
    TQtDocumentationCatalogItem *qtItem = dynamic_cast<TQtDocumentationCatalogItem *>(item);
    if (!qtItem)
        return;

    TQFileInfo fi(qtItem->dcfFile());

    TQFile f(qtItem->dcfFile());
    if (!f.open(IO_ReadOnly))
    {
        kdDebug(9002) << "Could not read" << qtItem->dcfFile() << endl;
        return;
    }
    TQDomDocument doc;
    if (!doc.setContent(&f) || doc.doctype().name() != "DCF")
    {
        kdDebug(9002) << "Not a valid DCF file: " << qtItem->dcfFile() << endl;
        return;
    }
    f.close();

    TQDomElement docEl = doc.documentElement();
    TQDomElement titleEl = docEl.namedItem("DCF").toElement();

    TQDomElement childEl = docEl.firstChild().toElement();
    while (!childEl.isNull())
    {
        if (childEl.tagName() == "section")
        {
            createSectionIndex(fi, index, item, childEl);
        }
        childEl = childEl.nextSibling().toElement();
    }
}

void DocQtPlugin::createSectionIndex(TQFileInfo &fi, IndexBox *index, DocumentationCatalogItem *item,
                                     TQDomElement section)
{
    //adymo: do not load section to index for TQt reference documentation
    TQString title = section.attribute("title");
    if (fi.fileName() != "qt.dcf")
    {
        TQString ref = section.attribute("ref");

        IndexItemProto *ii = new IndexItemProto(this, item, index, title, item->text(0));
        ii->addURL(KURL(fi.dirPath(true) + "/" + ref));
    }

    TQDomElement grandChild = section.firstChild().toElement();
    while(!grandChild.isNull())
    {
        if (grandChild.tagName() == "keyword")
        {
            TQString keyRef = grandChild.attribute("ref");
            TQString keyTitle = grandChild.text();

                    //adymo: a little hack to avoid unwanted elements
            if (keyRef != "qdir-example.html")
            {
                IndexItemProto *ii = new IndexItemProto(this, item, index, keyTitle, title);
                ii->addURL(KURL(fi.dirPath(true) + "/" + keyRef));
            }
        }
        if (grandChild.tagName() == "section")
        {
            createSectionIndex(fi, index, item, grandChild);
        }
        grandChild = grandChild.nextSibling().toElement();
    }
}

TQStringList DocQtPlugin::fullTextSearchLocations()
{
    TQStringList locs;

    TQMap<TQString, TQString> entryMap = config->entryMap("Locations");

    for (TQMap<TQString, TQString>::const_iterator it = entryMap.begin();
        it != entryMap.end(); ++it)
    {
        config->setGroup("Search Settings");
        if (config->readBoolEntry(it.key(), false))
        {
            config->setGroup("Locations");
            TQFileInfo fi(config->readPathEntry(it.key()));
            locs << fi.dirPath(true);
        }
    }

    return locs;
}

TQPair<KFile::Mode, TQString> DocQtPlugin::catalogLocatorProps()
{
    return TQPair<KFile::Mode, TQString>(KFile::File, "*.xml *.dcf");
}

TQString DocQtPlugin::catalogTitle(const TQString &url)
{
    TQFileInfo fi(url);
    if (!fi.exists())
        return TQString();

    TQFile f(url);
    if (!f.open(IO_ReadOnly))
        return TQString();

    TQDomDocument doc;
    if (!doc.setContent(&f) || (doc.doctype().name() != "DCF"))
        return TQString();
    f.close();

    TQDomElement docEl = doc.documentElement();

    return docEl.attribute("title", TQString());
}

DocumentationCatalogItem *DocQtPlugin::createCatalog(TDEListView *contents, TDEListViewItem *after, const TQString &title, const TQString &url)
{
    return new TQtDocumentationCatalogItem(url, this, contents, after, title);
}

#include "docqtplugin.moc"
