/* This file is part of the KDE project
   Copyright (C) 2002 Sandy Meier <smeier@kdevelop.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "filetemplate.h"

#include <tqdatetime.h>
#include <tqfile.h>
#include <tqfileinfo.h>
#include <tqregexp.h>
#include <tqtextstream.h>

#include <kstandarddirs.h>

#include "kdevplugin.h"
#include "kdevproject.h"
#include "domutil.h"


bool FileTemplate::exists(KDevPlugin *part, const TQString &name, Policy p)
{
    //TQString fileName = (p == Default) ?
    //   (part->project()->projectDirectory() + "/templates/" + name) : name;

    return TQFile::exists( fullPathForName(part,name,p) );
}

TQString FileTemplate::read(KDevPlugin *part, const TQString &name, Policy p)
{
    
  //KDevProject *project = part->project();
    //TQString fileName = (p == Default) ? (project->projectDirectory() +
//					 "/templates/" + name) : name;

    return readFile(part, fullPathForName(part, name, p) );
}

TQString FileTemplate::readFile(KDevPlugin *part, const TQString &fileName)
{
    TQDomDocument &dom = *part->projectDom();

    TQFile f(fileName);
    if (!f.open(IO_ReadOnly))
        return TQString();
    TQTextStream stream(&f);
    TQString str = stream.read();

    return makeSubstitutions( dom, str );
}

TQString FileTemplate::makeSubstitutions( TQDomDocument & dom, const TQString & text )
{
    TQString author = DomUtil::readEntry(dom, "/general/author");
    TQString email = DomUtil::readEntry(dom, "/general/email");
    TQString version = DomUtil::readEntry(dom, "/general/version");
    TQString appname = DomUtil::readEntry(dom, "/general/projectname");
    TQString date = TQDate::currentDate().toString();
    TQString year = TQString::number(TQDate::currentDate().year());

    TQString str = text;
    str.replace(TQRegExp("\\$EMAIL\\$"),email);
    str.replace(TQRegExp("\\$AUTHOR\\$"),author);
    str.replace(TQRegExp("\\$VERSION\\$"),version);
    str.replace(TQRegExp("\\$DATE\\$"),date);
    str.replace(TQRegExp("\\$YEAR\\$"),year);
    str.replace(TQRegExp("\\$APPNAME\\$"),appname);
    str.replace(TQRegExp("\\$APPNAME\\$"),appname);
    str.replace(TQRegExp("\\$APPNAMEUC\\$"),appname.upper());
    str.replace(TQRegExp("\\$APPNAMELC\\$"),appname.lower());

    return str;
}


bool FileTemplate::copy(KDevPlugin *part, const TQString &name,
                        const TQString &dest, Policy p)
{
    TQString text = read(part, name, p);

    TQFile f(dest);
    if (!f.open(IO_WriteOnly))
        return false;

    TQFileInfo fi(f);
    TQString module = fi.baseName();
    TQString basefilename = fi.baseName(true);
    text.replace(TQRegExp("\\$MODULE\\$"),module);
    text.replace(TQRegExp("\\$FILENAME\\$"),basefilename);

    TQTextStream stream(&f);
    stream << text;
    f.close();

    return true;
}

TQString FileTemplate::fullPathForName(KDevPlugin *part, const TQString &name,
                                      Policy p) {
    // if Policy is not default, full path is just the name
    if (p!=Default) return name;

    TQString fileName;
    // first try project-specific
    if (part->project())
    {
        fileName = (part->project()->projectDirectory() + "/templates/" + name);
        if (TQFile::exists(fileName)) return fileName;
    }

    // next try global
    TQString globalName = ::locate("data", "kdevfilecreate/file-templates/" + name);
    return globalName.isNull() ? fileName : globalName;
}
