/*
  Copyright (C) 2002 by Roberto Raggi <roberto@kdevelop.org>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  version 2, License as published by the Free Software Foundation.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public License
  along with this library; see the file COPYING.LIB.  If not, write to
  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
  Boston, MA 02110-1301, USA.
*/

#ifndef PROBLEMSREPORTER_H
#define PROBLEMSREPORTER_H

#include <tdelistview.h>
#include <klineedit.h>
#include <tqguardedptr.h>
#include <tqdatetime.h>
#include <map>
#include <ext/hash_map>
#include "hashedstring.h"

class CppSupportPart;
class TQTimer;
class TQTabBar;
class TQWidgetStack;
class TQGridLayout;
class KDialogBase;
class Problem;
class KURL;

class EfficientTDEListView {
public:
    typedef __gnu_cxx::hash_multimap<HashedString, TQListViewItem*> Map;
    typedef std::pair< Map::const_iterator, Map::const_iterator > Range;
    EfficientTDEListView( TDEListView* list = 0 ) : m_list( list ), m_insertionNumber( 0 ) {
    }
    
    EfficientTDEListView& operator = ( TDEListView* list ) {
        m_list = list;
        return *this;
    }
    
    operator TDEListView* () {
        return m_list;
    }
    
    operator const TDEListView* () const {
        return m_list;
    }
    
    TDEListView* operator -> () {
        return m_list;
    }
    
    const TDEListView* operator -> () const {
        return m_list;
    }
    
    void addItem( const TQString& str, TQListViewItem* item ) {
        HashedString h( str );
        m_insertionNumbers[h] = ++m_insertionNumber;
        m_map.insert( std::make_pair( h, item ) );
    }
    
    Range getRange( const TQString& str ) const {
        return m_map.equal_range( HashedString(str) );
    }

    ///If the list has more then size items, the first items are removed until the size fits.
    void limitSize( int size );
  
    void removeAllItems( const TQString& str );
    
    bool hasItem( const TQString& str ) const {
        Map::const_iterator it = m_map.find( HashedString(str) );
        return it != m_map.end();
    }
private:
    int m_insertionNumber;
    Map m_map;
    typedef __gnu_cxx::hash_map<HashedString, int> InsertionMap;
  InsertionMap m_insertionNumbers; //This is used to count which file was inserted first(higher insertion-number -> inserted later)
    TDEListView* m_list;
};

namespace KParts
{
	class Part;
}

namespace KTextEditor
{
	class MarkInterface;
	class Document;
}

class ProblemReporter: public TQWidget
{
	Q_OBJECT
	
public:
	ProblemReporter( CppSupportPart* part, TQWidget* parent = 0, const char* name = 0 );
	virtual ~ProblemReporter();

	void removeAllProblems( const TQString& filename );
	void reportProblem( const TQString& fileName, const Problem& p );
	bool hasErrors(const TQString& file);

public slots:
	void configWidget( KDialogBase* );

private slots:
	void slotPartAdded( KParts::Part* );
	void slotActivePartChanged( KParts::Part* );
	void slotSelected( TQListViewItem* );
	void slotTabSelected( int tabindex );
	void slotFilter();
	void initCurrentList();

private:
	TQString levelToString( int level ) const;
	int levelToMarkType( int level ) const;
	void InitListView( TDEListView* listview );
	void filterList( TDEListView* listview, const TQString& level );
    void updateCurrentWith( EfficientTDEListView& listview, const TQString& level, const TQString& filename );

private:
	TQGridLayout* m_gridLayout;
	TQTabBar* m_tabBar;
	TQWidgetStack* m_widgetStack;
	TDEListView* m_currentList;
    TQTimer* m_initCurrentTimer;
	EfficientTDEListView m_errorList;
	EfficientTDEListView m_fixmeList;
	EfficientTDEListView m_todoList;
    EfficientTDEListView m_warningList;
	TDEListView* m_filteredList;
	KLineEdit* m_filterEdit;

	CppSupportPart* m_cppSupport;
	KTextEditor::MarkInterface* m_markIface;
	TQString m_fileName;
	
	///@todo move these to cppsupportpart
	int m_active;
	int m_delay;
};

#endif 
// kate: indent-mode csands; tab-width 4;
