#include "toolsconfig.h"

#include <tqapplication.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqlistbox.h>
#include <tqpushbutton.h>
#include <tqheader.h>

#include <tdeapplication.h>
#include <kdesktopfile.h>
#include <kdialog.h>
#include <kiconloader.h>
#include <tdelocale.h>

#include "tools_part.h"
#include "kapplicationtree.h"


ToolsConfig::ToolsConfig(TQWidget *parent, const char *name)
    : TQWidget(parent, name), _tree(0)
{
  _entries.setAutoDelete(true);
}


void ToolsConfig::showEvent(TQShowEvent *e)
{
  TQWidget::showEvent(e);

  if (!_tree)
    {
      TQApplication::setOverrideCursor(TQt::waitCursor);

      TQHBoxLayout *hbox = new TQHBoxLayout(this, KDialog::marginHint(), KDialog::spacingHint());

      TQVBoxLayout *vbox = new TQVBoxLayout(hbox);
      _tree = new KDevApplicationTree(this);
      _tree->header()->hide();
      TQLabel *l = new TQLabel(_tree, i18n("&Applications:"), this);
      l->show();
      _tree->show();

      vbox->addWidget(l);
      vbox->addWidget(_tree);

      vbox = new TQVBoxLayout(hbox);

      _toList = new TQPushButton(TQApplication::reverseLayout() ? "<<" : ">>", this);
      _toList->show();
      vbox->addWidget(_toList);

      connect(_toList, TQT_SIGNAL(clicked()), this, TQT_SLOT(toList()));

      _toTree = new TQPushButton(TQApplication::reverseLayout() ? ">>" : "<<", this);
      _toTree->show();
      vbox->addWidget(_toTree);

      connect(_toTree, TQT_SIGNAL(clicked()), this, TQT_SLOT(toTree()));

      vbox = new TQVBoxLayout(hbox);
      _list = new TQListBox(this);
      l = new TQLabel(_list, i18n("&Tools menu:"), this);
      l->show();
      _list->show();
      vbox->addWidget(l);
      vbox->addWidget(_list);

      TQApplication::restoreOverrideCursor();
    }

  fill();
  checkButtons();

  connect(_tree, TQT_SIGNAL(selectionChanged()), this, TQT_SLOT(checkButtons()));
  connect(_list, TQT_SIGNAL(selectionChanged()), this, TQT_SLOT(checkButtons()));
}


void ToolsConfig::checkButtons()
{
  _toList->setEnabled(_tree->selectedItem() && !_tree->selectedItem()->firstChild());
  _toTree->setEnabled(_list->currentItem() >= 0 && _list->currentItem() < (int)_list->count());
}


void ToolsConfig::fill()
{
  _entries.clear();

  TDEConfig *config = ToolsFactory::instance()->config();
  config->setGroup("Tools");

  TQStringList list = config->readListEntry("Tools");

  for (TQStringList::Iterator it = list.begin(); it != list.end(); ++it)
	add(*it);
}


void ToolsConfig::add(const TQString &desktopFile)
{
  KDesktopFile df(desktopFile, true);
  if (df.readName().isEmpty())
    return;

  Entry *entry = new Entry;

  if (!df.readIcon().isEmpty())
    entry->icon = BarIcon(df.readIcon());
  entry->name = df.readName();
  entry->desktopFile = desktopFile;

  _entries.append(entry);

  updateList();

  checkButtons();
}


void ToolsConfig::toList()
{
  KDevAppTreeListItem *item = dynamic_cast<KDevAppTreeListItem*>(_tree->selectedItem());
  if (item && !item->desktopEntryPath().isEmpty())
    add(item->desktopEntryPath());
  checkButtons();
}


void ToolsConfig::toTree()
{
  _entries.remove(_list->currentItem());
  updateList();
  checkButtons();
}


void ToolsConfig::accept()
{
  TDEConfig *config = ToolsFactory::instance()->config();
  config->setGroup("Tools");

  TQStringList l;
  TQPtrListIterator<Entry> it(_entries);
    for ( ; it.current(); ++it)
	  l.append(it.current()->desktopFile);

  config->writeEntry("Tools", l);
  config->sync();
}


void ToolsConfig::updateList()
{
  _list->setUpdatesEnabled(false);

  _list->clear();

  TQPtrListIterator<Entry> it(_entries);
  for ( ; it.current(); ++it)
	_list->insertItem(it.current()->icon, it.current()->name);

  _list->setUpdatesEnabled(true);
  _list->repaint();
}


#include "toolsconfig.moc"
