/*
    This file is part of libtdepim.

    Copyright (c) 2003 Tobias Koenig <tokoe@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#ifndef KPIM_ADDRESSEEVIEW_H
#define KPIM_ADDRESSEEVIEW_H

#include <tqcstring.h>

#include <tdeabc/addressee.h>
#include <ktextbrowser.h>
#include <tdeimproxy.h>
#include <tdepimmacros.h>

namespace TDEIO {
class Job;
}
class TDEToggleAction;

class TQPopupMenu;


namespace KPIM {


class KDE_EXPORT AddresseeView : public KTextBrowser
{
  Q_OBJECT
  
  public:
    /**
      Constructor.
 
      @param config The config object where the settings are stored
                    which fields will be shown.
     */
    AddresseeView( TQWidget *parent = 0, const char *name = 0,
                   TDEConfig *config = 0 );

    ~AddresseeView();

    /**
      Sets the addressee object. The addressee is displayed immediately.

      @param addr The addressee object.
     */
    void setAddressee( const TDEABC::Addressee& addr );

    /**
      Returns the current addressee object.
     */
    TDEABC::Addressee addressee() const;


    /**
      This enums are used by enableLinks to set which kind of links shall
      be enabled.
     */
    enum LinkMask {
      NoLinks = 0,
      AddressLinks = 1,
      EmailLinks = 2,
      PhoneLinks = 4,
      URLLinks = 8,
      IMLinks = 16,
      DefaultLinks = AddressLinks | EmailLinks | PhoneLinks | URLLinks | IMLinks
    };

    /**
      Sets which parts of the contact shall be presented as links.
      The mask can be OR'ed LinkMask. By default all links are enabled.
     */
    void enableLinks( int linkMask );

    /**
      This enums are used by vCardAsHTML to decide which fields shall be
      shown.
     */
    enum FieldMask {
      NoFields = 0,
      BirthdayFields = 1,
      AddressFields = 2,
      EmailFields = 4,
      PhoneFields = 8,
      URLFields = 16,
      IMFields = 32,
      CustomFields = 64,
      DefaultFields = AddressFields | EmailFields | PhoneFields | URLFields
    };

    /**
      Returns the HTML representation of a contact.
      The HTML code looks like
        <div>
        <table>
        ...
        </table>
        </div>

      @param addr The addressee object.
      @param linkMask The mask for which parts of the contact will
                      be displayed as links.
                      The links looks like this:
                        "addr://<addr id>" for addresses
                        "mailto:<email address>" for emails
                        "phone://<phone number>" for phone numbers
                        "http://<url>" for urls
                        "im:<im addrss>" for instant messaging addresses
                        "sms://<phone number>" for sending a sms
      @param internalLoading If true, the loading of internal pictures is done automatically.
      @param fieldMask The mask for which fields of the contact will
                       be displayed.
     */
    static TQString vCardAsHTML( const TDEABC::Addressee& addr, ::KIMProxy *proxy, LinkMask linkMask = DefaultLinks,
                                bool internalLoading = true, FieldMask fieldMask = DefaultFields );

    /**
     * Encodes a TQPixmap as a PNG into a data: URL (rfc2397), readable by the data tdeio protocol
     * @param pixmap the pixmap to encode
     * @return a data: URL
     */
    static TQString pixmapAsDataUrl( const TQPixmap& pixmap );

  signals:
    void urlHighlighted( const TQString &url );
    void emailHighlighted( const TQString &email );
    void phoneNumberHighlighted( const TQString &number );
    void faxNumberHighlighted( const TQString &number );

    void highlightedMessage( const TQString &message );

    void addressClicked( const TQString &uid );

  protected:
    virtual void urlClicked( const TQString &url );
    virtual void emailClicked( const TQString &mail );
    virtual void phoneNumberClicked( const TQString &number );
    virtual void smsTextClicked( const TQString &number );
    virtual void sendSMS( const TQString &number, const TQString &msg );
    virtual void faxNumberClicked( const TQString &number );
    virtual void imAddressClicked();

    virtual TQPopupMenu *createPopupMenu( const TQPoint& );

  private slots:
    void slotMailClicked( const TQString&, const TQString& );
    void slotUrlClicked( const TQString& );
    void slotHighlighted( const TQString& );
    void slotPresenceChanged( const TQString & );
    void slotPresenceInfoExpired();
    void configChanged();

    void data( TDEIO::Job*, const TQByteArray& );
    void result( TDEIO::Job* );

  private:
    void load();
    void save();

    void updateView();

    TQString strippedNumber( const TQString &number );

    TDEConfig *mConfig;
    bool mDefaultConfig;

    TQByteArray mImageData;
    TDEIO::Job *mImageJob;

    TDEToggleAction *mActionShowBirthday;
    TDEToggleAction *mActionShowAddresses;
    TDEToggleAction *mActionShowEmails;
    TDEToggleAction *mActionShowPhones;
    TDEToggleAction *mActionShowURLs;
    TDEToggleAction *mActionShowIMAddresses;
    TDEToggleAction *mActionShowCustomFields;

    TDEABC::Addressee mAddressee;
    int mLinkMask;

    class AddresseeViewPrivate;
    AddresseeViewPrivate *d;
    ::KIMProxy *mKIMProxy;
};

}

#endif
