/*
    This file is part of libkcal.

    Copyright (c) 2003,2004 Cornelius Schumacher <schumacher@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include <typeinfo>
#include <stdlib.h>

#include <tqdatetime.h>
#include <tqstring.h>
#include <tqptrlist.h>

#include <kdebug.h>
#include <kurl.h>
#include <tdeio/job.h>
#include <tdelocale.h>
#include <tdemessagebox.h>
#include <kstandarddirs.h>

#include <libkcal/icalformat.h>
#include <libkcal/exceptions.h>
#include <libkcal/incidence.h>
#include <libkcal/event.h>
#include <libkcal/todo.h>
#include <libkcal/journal.h>
#include <libkcal/filestorage.h>

#include <tdeabc/lock.h>

#include <tderesources/configwidget.h>

#include "resourceremoteconfig.h"

#include "resourceremote.h"

using namespace KCal;

ResourceRemote::ResourceRemote( const TDEConfig *config )
  : ResourceCached( config ), mUseProgressManager( true ), mUseCacheFile( true )
{
  if ( config ) {
    readConfig( config );
  } else {
    setResourceName( i18n( "Remote Calendar" ) );
  }

  init();
}

ResourceRemote::ResourceRemote( const KURL &downloadUrl, const KURL &uploadUrl )
  : ResourceCached( 0 ), mUseProgressManager( false ), mUseCacheFile( false )
{
  mDownloadUrl = downloadUrl;

  if ( uploadUrl.isEmpty() ) {
    mUploadUrl = mDownloadUrl;
  } else {
    mUploadUrl = uploadUrl;
  }

  init();
}

ResourceRemote::~ResourceRemote()
{
  close();

  if ( mDownloadJob ) mDownloadJob->kill();
  if ( mUploadJob ) mUploadJob->kill();

  delete mLock;
}

void ResourceRemote::init()
{
  mDownloadJob = 0;
  mUploadJob = 0;

  mProgress = 0;

  setType( "remote" );

  mLock = new TDEABC::Lock( cacheFile() );

  enableChangeNotification();
}

void ResourceRemote::readConfig( const TDEConfig *config )
{
  TQString url = config->readEntry( "DownloadUrl" );
  mDownloadUrl = KURL( url );

  url = config->readEntry( "UploadUrl" );
  mUploadUrl = KURL( url );

  ResourceCached::readConfig( config );
}

void ResourceRemote::writeConfig( TDEConfig *config )
{
  kdDebug(5800) << "ResourceRemote::writeConfig()" << endl;

  ResourceCalendar::writeConfig( config );

  config->writeEntry( "DownloadUrl", mDownloadUrl.url() );
  config->writeEntry( "UploadUrl", mUploadUrl.url() );

  ResourceCached::writeConfig( config );
}

void ResourceRemote::setDownloadUrl( const KURL &url )
{
  mDownloadUrl = url;
}

KURL ResourceRemote::downloadUrl() const
{
  return mDownloadUrl;
}

void ResourceRemote::setUploadUrl( const KURL &url )
{
  mUploadUrl = url;
}

KURL ResourceRemote::uploadUrl() const
{
  return mUploadUrl;
}

void ResourceRemote::setUseProgressManager( bool useProgressManager )
{
  mUseProgressManager = useProgressManager;
}

bool ResourceRemote::useProgressManager() const
{
  return mUseProgressManager;
}

void ResourceRemote::setUseCacheFile( bool useCacheFile )
{
  mUseCacheFile = useCacheFile;
}

bool ResourceRemote::useCacheFile() const
{
  return mUseCacheFile;
}

bool ResourceRemote::doLoad()
{
  kdDebug(5800) << "ResourceRemote::load()" << endl;

  if ( mDownloadJob ) {
    kdWarning() << "ResourceRemote::load(): download still in progress."
                << endl;
    return true;
  }
  if ( mUploadJob ) {
    kdWarning() << "ResourceRemote::load(): upload still in progress."
                << endl;
    return false;
  }

  mCalendar.close();

  if ( mUseCacheFile ) {
    disableChangeNotification();
    loadCache();
    enableChangeNotification();
  }

  clearChanges();

  emit resourceChanged( this );

  if ( mLock->lock() )
  {
    kdDebug() << "Download from: " << mDownloadUrl << endl;

    mDownloadJob = TDEIO::file_copy( mDownloadUrl, KURL( cacheFile() ), -1, true,
                                    false, !mUseProgressManager );
    connect( mDownloadJob, TQT_SIGNAL( result( TDEIO::Job * ) ),
            TQT_SLOT( slotLoadJobResult( TDEIO::Job * ) ) );
    if ( mUseProgressManager ) {
        connect( mDownloadJob, TQT_SIGNAL( percent( TDEIO::Job *, unsigned long ) ),
                TQT_SLOT( slotPercent( TDEIO::Job *, unsigned long ) ) );
        mProgress = KPIM::ProgressManager::createProgressItem(
            KPIM::ProgressManager::getUniqueID(), i18n("Downloading Calendar") );

        mProgress->setProgress( 0 );
    }
  }
  else
    kdDebug() << "ResourceRemote::load(): cache file is locked - something else must be loading the file" << endl;
  return true;
}

void ResourceRemote::slotPercent( TDEIO::Job *, unsigned long percent )
{
  kdDebug() << "ResourceRemote::slotPercent(): " << percent << endl;

  mProgress->setProgress( percent );
}

void ResourceRemote::slotLoadJobResult( TDEIO::Job *job )
{
  if ( job->error() ) {
    // TODO: Should detect 404, 401 etc. vs host not found and prompt to create new resource only when 404 is returned
    TQString warningString = TQString("<b>") + i18n("Remote data access failure") + TQString("</b><p>") + i18n("Detailed information") \
           + TQString(":<br>") + job->errorString() + TQString("<p>") + i18n("What would you like to do?");
    int rc = KMessageBox::warningContinueCancel(0,
	   warningString, i18n("Remote Data Access Failure"), i18n("&Continue with cached resource"));
    if ( rc == KMessageBox::Continue ) {
        mCalendar.close();
        disableChangeNotification();
        loadCache();
        enableChangeNotification();

        emit resourceChanged( this );
    }
    //job->showErrorDialog( 0 );
  } else {
    kdDebug(5800) << "ResourceRemote::slotLoadJobResult() success" << endl;

    mCalendar.close();
    disableChangeNotification();
    loadCache();
    enableChangeNotification();

    emit resourceChanged( this );
  }

  mDownloadJob = 0;
  if ( mProgress ) {
    mProgress->setComplete();
    mProgress = 0;
  }

  mLock->unlock();
  emit resourceLoaded( this );
}

bool ResourceRemote::doSave()
{
  kdDebug(5800) << "ResourceRemote::save()" << endl;

  if ( readOnly() || !hasChanges() ) {
    emit resourceSaved( this );
    return true;
  }

  if ( mDownloadJob ) {
    kdWarning() << "ResourceRemote::save(): download still in progress."
                << endl;
    return false;
  }
  if ( mUploadJob ) {
    kdWarning() << "ResourceRemote::save(): upload still in progress."
                << endl;
    return false;
  }

  mChangedIncidences = allChanges();

  saveCache();

  mUploadJob = TDEIO::file_copy( KURL( cacheFile() ), mUploadUrl, -1, true );
  connect( mUploadJob, TQT_SIGNAL( result( TDEIO::Job * ) ),
           TQT_SLOT( slotSaveJobResult( TDEIO::Job * ) ) );

  return true;
}

bool ResourceRemote::isSaving()
{
  return mUploadJob;
}

void ResourceRemote::slotSaveJobResult( TDEIO::Job *job )
{
  if ( job->error() ) {
    job->showErrorDialog( 0 );
  } else {
    kdDebug(5800) << "ResourceRemote::slotSaveJobResult() success" << endl;

    Incidence::List::ConstIterator it;
    for( it = mChangedIncidences.begin(); it != mChangedIncidences.end();
         ++it ) {
      clearChange( *it );
    }
    mChangedIncidences.clear();
  }

  mUploadJob = 0;

  emit resourceSaved( this );
}

TDEABC::Lock *ResourceRemote::lock()
{
  return mLock;
}

void ResourceRemote::dump() const
{
  ResourceCalendar::dump();
  kdDebug(5800) << "  DownloadUrl: " << mDownloadUrl.url() << endl;
  kdDebug(5800) << "  UploadUrl: " << mUploadUrl.url() << endl;
  kdDebug(5800) << "  ReloadPolicy: " << reloadPolicy() << endl;
}

void ResourceRemote::addInfoText( TQString &txt ) const
{
  txt += "<br>";
  txt += i18n("URL: %1").arg( mDownloadUrl.prettyURL() );
}

bool ResourceRemote::setValue( const TQString &key, const TQString &value )
{
  if ( key == "URL" ) {
    setUploadUrl( KURL( value ) );
    setDownloadUrl( KURL( value ) );
    return true;
  } else  if ( key == "DownloadURL" ) {
    setDownloadUrl( KURL( value ) );
    return true;
  } else if ( key == "UploadURL" ) {
    setUploadUrl( KURL( value ) );
    return true;
  } else
    return ResourceCached::setValue( key, value );
}

#include "resourceremote.moc"
