/*
 * Copyright (C) 2004, Mart Kelder (mart.kde@hccnet.nl)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "tdeio_single_subject.h"

#include "mailsubject.h"
#include "tdeio_proto.h"
#include "stringid.h"

#include <kdebug.h>
#include <tdelocale.h>
#include <kurl.h>
#include <tdeio/global.h>
#include <tdeio/scheduler.h>
#include <tdeio/job.h>
#include <tdeio/jobclasses.h>
#include <krfcdate.h>

#include <tqregexp.h>
#include <tqcstring.h>
#include <tqstring.h>

TDEIO_Single_Subject::TDEIO_Single_Subject( TQObject * parent, const char * name,
		    KURL &kurl, TDEIO::MetaData &metadata, const TDEIO_Protocol * protocol, TDEIO::Slave *& slave,
		    const TQString &url, const long size ) 
		: TQObject( parent, name )
{
	_kurl = new KURL( kurl );
	_metadata = new TDEIO::MetaData( metadata );
	_protocol = protocol;
	_name = new TQString( url );
	_size = size;
	_message = new TQString;
	
	init( slave );
}

TDEIO_Single_Subject::~TDEIO_Single_Subject( )
{
	if( _job )
		TDEIO::Scheduler::cancelJob( _job );
	delete _kurl;
	delete _metadata;
	delete _name;
	delete _message;
}

void TDEIO_Single_Subject::init( TDEIO::Slave *& slave)
{
	_job = TDEIO::get( *_kurl, false, false );
	_job->addMetaData( *_metadata );
	
	connect( _job, TQT_SIGNAL( result( TDEIO::Job* ) ), this, TQT_SLOT( slotResult( TDEIO::Job* ) ) );
	connect( _job, TQT_SIGNAL( data         (TDEIO::Job *, const TQByteArray &) ),
	         this, TQT_SLOT( slotData(TDEIO::Job *, const TQByteArray &) ) );
	
	if( _protocol->connectionBased( ) && slave )
		TDEIO::Scheduler::assignJobToSlave( slave , _job );
	else
		TDEIO::Scheduler::scheduleJob( _job );
		 
}

void TDEIO_Single_Subject::parseMail( TQString * message, KornMailSubject *subject, bool fullMessage )
{
	TQTextStream stream( message, IO_ReadOnly );
	TQString line;
	TQRegExp rx_sender( "^[fF]rom: " ); //Ex: From: ...
	TQRegExp rx_sender_has_name1( "^[fF]rom:\\s*(\\w+[\\w\\s]*)\\<" ); //Ex: From: A name<email@domein.country>
	TQRegExp rx_sender_has_name2( "^[fF]rom:\\s*\\\"\\s*(\\w+[\\w\\s]*)\\\""); //Ex: From: "A name"<a@invalid>
	TQRegExp rx_subject( "^[sS]ubject: " ); //Ex: Subject: ...
	TQRegExp rx_date  ( "^[dD]ate: ");
	bool inheader = true;
	int fieldnumber = 0;

	while ( ! stream.atEnd() )
	{
		line = stream.readLine();
		if( line.isEmpty() && fieldnumber >= 2 )
			inheader = false;
		
		if( inheader )
		{
			if( rx_sender.search( line ) == 0 )
			{
				if( rx_sender_has_name1.search( line ) == 0 )
					subject->setSender( rx_sender_has_name1.cap( 1 )  );
				else if(rx_sender_has_name2.search( line ) == 0)
					subject->setSender( rx_sender_has_name2.cap( 1 ) );
				else
					subject->setSender( line.remove( rx_sender ) );
				++fieldnumber;
			}
			else if( rx_subject.search( line ) == 0 )
			{
				subject->setSubject( line.remove( rx_subject ) );
				++fieldnumber;
			}
			else if( rx_date.search( line ) == 0 )
			{
				subject->setDate( KRFCDate::parseDate( line.right( line.length() - 6 ) ) );
				++fieldnumber;
			}
		}
	}

	subject->setHeader( *message, fullMessage );
}

void TDEIO_Single_Subject::slotData( TDEIO::Job* job, const TQByteArray& data )
{
	if( job != _job )
		kdWarning() << i18n( "Got invalid job; something strange happened?" ) << endl;
	if( !data.isEmpty() )
		_message->append( data );
}

//TDEIO::Scheduler::disconnectSlave missing  if connection stops
void TDEIO_Single_Subject::slotResult( TDEIO::Job *job )
{
	if( job != _job )
		kdWarning() << i18n( "Got invalid job; something strange happened?" ) << endl;
		
	if( job->error() )
	{
		kdWarning() << i18n( "Error when fetching %1: %2" ).arg( *_name ).arg( job->errorString() ) << endl;
	} else {
		KornMailSubject * mailSubject = new KornMailSubject( new KornStringId( *_name ), 0 );
		parseMail( _message, mailSubject, _protocol->fullMessage() );
		mailSubject->decodeHeaders();
		mailSubject->setSize( _size );
		emit readSubject( mailSubject );
	}
	
	_job = 0;
	
	emit finished( this );
}

#include "tdeio_single_subject.moc"
