/***************************************************************************
 *
 * tdenetman-wireless_network.cpp - A NetworkManager frontend for TDE 
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Helmut Schaa <hschaa@suse.de>, <Helmut.Schaa@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

// TDENM includes
#include "tdenetman.h"
#include "tdenetman-wireless_network.h"

// TDE includes
#include <kdebug.h>

// TQt includes
#include <tqhostaddress.h>
#include <tqvaluelist.h>

class WirelessNetworkPrivate
{
	public:
	WirelessNetworkPrivate()
	  : active(false)
	{
	}

	~WirelessNetworkPrivate()
	{
	}

	bool       active;

	TQ_UINT32   match;
	TQMap<TDEMACAddress,TQString> aps;
};

WirelessNetwork::WirelessNetwork(const WirelessNetwork& other)
{
	hwdevices = TDEGlobal::hardwareDevices();
	d = new WirelessNetworkPrivate(*other.d);
}

WirelessNetwork::WirelessNetwork(TQ_UINT32 match)
{
	hwdevices = TDEGlobal::hardwareDevices();
	d = new WirelessNetworkPrivate();

	// which attributes have to match
	d->match = match;
}

WirelessNetwork::~WirelessNetwork()
{
	delete d;
}

WirelessNetwork& WirelessNetwork::operator= (const WirelessNetwork& other)
{
	if (d) {
		delete d;
	}
	d = new WirelessNetworkPrivate(*other.d);
	return *this;
}

bool WirelessNetwork::contains(const TDEMACAddress ap)
{
	TDEGlobalNetworkManager* nm = TDEGlobal::networkManager();

	TDENetworkDevice* dev = NULL;
	TDENetworkWiFiAPInfo* apInfo = NULL;
	if (d->aps.contains(ap)) {
		dev = dynamic_cast<TDENetworkDevice*>(hwdevices->findByUniqueID(d->aps[ap]));
		if (dev) {
			TDENetworkConnectionManager* deviceConnMan = dev->connectionManager();
			apInfo = deviceConnMan->findAccessPointByBSSID(ap);
		}
		else if (d->aps[ap] == "") {
			apInfo = nm->findAccessPointByBSSID(ap);
		}
	}
	else {
		apInfo = nm->findAccessPointByBSSID(ap);
	}

	if (apInfo) {
		if ( (d->match & MATCH_SSID) && (getSsid() != apInfo->SSID)) {
			return false;
		}
	}
	else {
		// AP was not found in list
		return false;
	}

	// AP matches
	return true;
}

bool WirelessNetwork::addAP(const TDEMACAddress ap, const TQString dev)
{
	if ( this->contains(ap) || d->aps.isEmpty()) {
		d->aps[ap] = dev;
		return true;
	}
	return false;
}

const int WirelessNetwork::getAPCount() const
{
	return d->aps.count();
}

const TQByteArray WirelessNetwork::getSsid() const
{
	TDEGlobalNetworkManager* nm = TDEGlobal::networkManager();

	if ( !d->aps.isEmpty() ) {
		TDENetworkDevice* dev = dynamic_cast<TDENetworkDevice*>(hwdevices->findByUniqueID(d->aps.begin().data()));
		TDENetworkWiFiAPInfo* apInfo = NULL;
		if (dev) {
			TDENetworkConnectionManager* deviceConnMan = dev->connectionManager();
			apInfo = deviceConnMan->findAccessPointByBSSID(d->aps.begin().key());
		}
		else if (d->aps.begin().data() == "") {
			apInfo = nm->findAccessPointByBSSID(d->aps.begin().key());
		}

		if (apInfo) {
			return apInfo->SSID;
		}
		else {
			return TQByteArray();
		}
	}
	else {
		return TQByteArray();
	}
}

TQString WirelessNetwork::getDisplaySsid() const
{
	TDEGlobalNetworkManager* nm = TDEGlobal::networkManager();

	if (!d->aps.isEmpty()) {
		TDENetworkDevice* dev = dynamic_cast<TDENetworkDevice*>(hwdevices->findByUniqueID(d->aps.begin().data()));
		TDENetworkWiFiAPInfo* apInfo = NULL;
		if (dev) {
			TDENetworkConnectionManager* deviceConnMan = dev->connectionManager();
			apInfo = deviceConnMan->findAccessPointByBSSID(d->aps.begin().key());
		}
		else if (d->aps.begin().data() == "") {
			apInfo = nm->findAccessPointByBSSID(d->aps.begin().key());
		}

		if (apInfo) {
			return apInfo->friendlySSID();
		}
		else {
			return TQString();
		}
	}
	else {
		return TQString();
	}
}

TDENetworkWiFiAPFlags::TDENetworkWiFiAPFlags WirelessNetwork::getFlags() const
{
	TDEGlobalNetworkManager* nm = TDEGlobal::networkManager();

	TDENetworkWiFiAPFlags::TDENetworkWiFiAPFlags flags = TDENetworkWiFiAPFlags::None;
	for (TQMap<TDEMACAddress,TQString>::Iterator it = d->aps.begin(); it != d->aps.end(); ++it) {
		TDENetworkDevice* dev = dynamic_cast<TDENetworkDevice*>(hwdevices->findByUniqueID(it.data()));
		TDENetworkWiFiAPInfo* apInfo = NULL;
		if (dev) {
			TDENetworkConnectionManager* deviceConnMan = dev->connectionManager();
			apInfo = deviceConnMan->findAccessPointByBSSID(it.key());
		}
		else if (it.data() == "") {
			apInfo = nm->findAccessPointByBSSID(it.key());
		}

		if (apInfo) {
			flags |= (apInfo->wpaFlags & TDENetworkWiFiAPFlags::GeneralFlagsMask);
			flags |= (apInfo->rsnFlags & TDENetworkWiFiAPFlags::GeneralFlagsMask);
		}
	}
	return flags;
}

TDENetworkWiFiAPFlags::TDENetworkWiFiAPFlags WirelessNetwork::getWpaFlags() const
{
	TDEGlobalNetworkManager* nm = TDEGlobal::networkManager();

	TDENetworkWiFiAPFlags::TDENetworkWiFiAPFlags flags = TDENetworkWiFiAPFlags::None;
	for (TQMap<TDEMACAddress,TQString>::Iterator it = d->aps.begin(); it != d->aps.end(); ++it)
	{
		TDENetworkDevice* dev = dynamic_cast<TDENetworkDevice*>(hwdevices->findByUniqueID(it.data()));
		TDENetworkWiFiAPInfo* apInfo = NULL;
		if (dev) {
			TDENetworkConnectionManager* deviceConnMan = dev->connectionManager();
			apInfo = deviceConnMan->findAccessPointByBSSID(it.key());
		}
		else if (it.data() == "") {
			apInfo = nm->findAccessPointByBSSID(it.key());
		}

		if (apInfo) {
			flags |= (apInfo->wpaFlags & TDENetworkWiFiAPFlags::EncryptionFlagsMask);
		}
	}
	return flags;
}

TDENetworkWiFiAPFlags::TDENetworkWiFiAPFlags WirelessNetwork::getRsnFlags() const
{
	TDEGlobalNetworkManager* nm = TDEGlobal::networkManager();

	TDENetworkWiFiAPFlags::TDENetworkWiFiAPFlags flags = TDENetworkWiFiAPFlags::None;
	for (TQMap<TDEMACAddress,TQString>::Iterator it = d->aps.begin(); it != d->aps.end(); ++it)
	{
		TDENetworkDevice* dev = dynamic_cast<TDENetworkDevice*>(hwdevices->findByUniqueID(it.data()));
		TDENetworkWiFiAPInfo* apInfo = NULL;
		if (dev) {
			TDENetworkConnectionManager* deviceConnMan = dev->connectionManager();
			apInfo = deviceConnMan->findAccessPointByBSSID(it.key());
		}
		else if (it.data() == "") {
			apInfo = nm->findAccessPointByBSSID(it.key());
		}

		if (apInfo) {
			flags |= (apInfo->rsnFlags & TDENetworkWiFiAPFlags::EncryptionFlagsMask);
		}
	}
	return flags;
}

bool WirelessNetwork::isEncrypted() const
{
	return (getFlags() & TDENetworkWiFiAPFlags::PrivacySupport);
}

TQ_UINT8 WirelessNetwork::getStrength() const
{
	TDEGlobalNetworkManager* nm = TDEGlobal::networkManager();

	TQ_UINT8 strength = 0;
	for (TQMap<TDEMACAddress,TQString>::Iterator it = d->aps.begin(); it != d->aps.end(); ++it) {
		TDENetworkDevice* dev = dynamic_cast<TDENetworkDevice*>(hwdevices->findByUniqueID(it.data()));
		TDENetworkWiFiAPInfo* apInfo = NULL;
		if (dev) {
			TDENetworkConnectionManager* deviceConnMan = dev->connectionManager();
			apInfo = deviceConnMan->findAccessPointByBSSID(it.key());
		}
		else if (it.data() == "") {
			apInfo = nm->findAccessPointByBSSID(it.key());
		}

		if (apInfo) {
			if ((apInfo->signalQuality*100.0) > strength) {
				strength = (apInfo->signalQuality*100.0);
			}
		}
	}
	return strength;
}

void WirelessNetwork::setActive(bool a)
{
	d->active=a;
}

bool WirelessNetwork::getActive() const
{
	return d->active;
}

