// -*- c++ -*-

/*
 *  Copyright (C) 2001-2004, Richard J. Moore <rich@kde.org>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#ifndef KJSEMBEDFACTORY_H
#define KJSEMBEDFACTORY_H
#include "global.h"
#include <tqmap.h>
#include <tqdict.h>
#include <tqstringlist.h>

#include <kjs/interpreter.h>
#include <kjs/object.h>

class TQEvent;
class TQObject;
class TQWidget;
class TQTextStream;

namespace KParts {
    class ReadOnlyPart;
    class ReadWritePart;
}

/**
 * Namespace containing the KJSEmbed library.
 */
namespace KJSEmbed {

namespace Bindings {
    class JSFactoryImp;
    class JSBindingPlugin;
    class JSBindingBase;
}

class KJSEmbedPart;
class JSEventMapper;
class JSOpaqueProxy;
class JSObjectProxy;

/**
 * Factory class for KJSEmbed.
 *
 * @author Richard Moore, rich@kde.org
 */
class KJSEMBED_EXPORT JSFactory
{
public:
    /**
     * The types of proxy that can be associated with a class name. These are
     * used when decoding the return values of slots.
     */
    enum ProxyTypes {
	TypeInvalid=0,
        TypeTQObject=1,
        TypeValue=2,
	TypeOpaque=4,
	TypePlugin=8,
	TypeTQObjectPlugin=9 /*TypeObject|TypePlugin*/
    };

    /** Creates a factory object for the specified part. */
    JSFactory( KJSEmbedPart *part );

    /** Cleans up. */
    virtual ~JSFactory();

    /**
     * Returns the mapper that translates between event types and the names of
     * the handler methods.
     */
    JSEventMapper *eventMapper() const { return evmapper; }

    /** Returns the part that the factory is attached to. */
    KJSEmbedPart *part() const { return jspart; }

    //
    // Methods for creating objects and wrappers.
    //

    /** Creates an object of the specified class, then returns a proxy. */
    KJS::Object create( KJS::ExecState *exec, const TQString &classname, const KJS::List &args );

    /** Creates a proxy object for the specified target with the specified context. */
    KJS::Object createProxy( KJS::ExecState *exec, TQObject *target,
			     const JSObjectProxy *context=0 ) const;

    /** Creates a proxy object for the specified target with the specified context. */
    KJS::Object createProxy( KJS::ExecState *exec, TQTextStream *target,
			     const JSObjectProxy *context=0 ) const;

    /** Creates a proxy object for the specified target with the specified context. */
    KJS::Object createProxy( KJS::ExecState *exec, TQEvent *target,
			     const JSObjectProxy *context ) const;
    
    
    /**
    * Registers an opaque proxy factory with the with the factory.
    * To add an opaque type to the system you need to provide a factory based off of @ref KJSEmbed::Bindings::JSBindingBase
     */
    void registerOpaqueType( const TQString &className, KJSEmbed::Bindings::JSBindingBase *bindingFactory);

    /**
    * Remove an opaque type from the system.
     */
    void unregisterOpaqueType( const TQString &className );

    /**
     * When passed an object for a type handled by an Opaque proxy this method
     * will add the bindings for that opaque object to the proxy.  If the object
     * is not supported it is unmodified.
     */
    void extendOpaqueProxy( KJS::ExecState *exec, KJS::Object &proxy) const;

    /**
    * Registers a TQObject proxy factory with the with the factory.
    * To add an opaque type to the system you need to provide a factory based off of @ref KJSEmbed::Bindings::JSBindingBase
     */
    void registerObjectType( const TQString &className, KJSEmbed::Bindings::JSBindingBase *bindingFactory);

    /**
    * Remove an opaque type from the system.
     */
    void unregisterObjectType( const TQString &className );

    /**
     * When passed an object for a type handled by an Opaque proxy this method
     * will add the bindings for that opaque object to the proxy.  If the object
     * is not supported it is unmodified.
     */
    void extendObjectProxy( KJS::ExecState *exec, KJS::Object &proxy) const;

    
    //
    // Methods for creating custom objects.
    //

    /**
     * Creates a ReadOnlyPart that views the specified service type. The
     * service type will often be a MIME type like 'text/html'. The part
     * created is the first offer returned by the trader, and will have the
     * specified parent and name.
     */
    KParts::ReadOnlyPart *createROPart( const TQString &svc, TQObject *parent=0, const char *name=0 );

    /**
     * Creates a ReadOnlyPart that views the specified service type and matches
     * the specified constraint.
     */
    KParts::ReadOnlyPart *createROPart( const TQString &svc, const TQString &constraint,
					TQObject *parent=0, const char *name=0 );

    /**
     * Creates a ReadOnlyPart that views the specified service type and matches
     * the specified constraint.
     */
    KParts::ReadOnlyPart *createROPart( const TQString &svc, const TQString &constraint,
					TQObject *parent, const char *name,
					const TQStringList &args );

    /**
     * Creates a ReadWritePart that edits the specified service type. The
     * service type will often be a MIME type like 'text/plain'. The part
     * created is the first offer returned by the trader, and will have the
     * specified parent and name.
     */
    KParts::ReadWritePart *createRWPart( const TQString &svc, TQObject *parent=0, const char *name=0 );

    /**
     * Creates a ReadWritePart that edits the specified service type and matches
     * the specified constraint.
     */
    KParts::ReadWritePart *createRWPart( const TQString &svc, const TQString &constraint,
					 TQObject *parent=0, const char *name=0 );

    /**
     * Creates a ReadWritePart that edits the specified service type and matches
     * the specified constraint.
     */
    KParts::ReadWritePart *createRWPart( const TQString &svc, const TQString &constraint,
					 TQObject *parent, const char *name,
					 const TQStringList &args );

    /**
     * Loads the widget defined in the specified .ui file. If the widget
     * cannot be created then 0 is returned.
     */
    TQWidget *loadUI( const TQString &uiFile, TQObject *connector=0, TQWidget *parent=0, const char *name=0 );

    /**
    * Queries TDETrader for a plugin that provides the asked for object binding.
    *If the binding was added the object is returned, otherwise a KJS::Null is.
    */
    TQStringList listBindingPlugins( KJS::ExecState *exec, KJS::Object &self);
    //
    // Methods that tell the factory how to handle different classes.
    //
    /**
    * Adds a binding plugin type to the list of available types the factory can create.
    */
    void addBindingPluginTypes(KJS::ExecState *exec, KJS::Object &parent);
    bool isBindingPlugin(const TQString &classname) const;
    /**
    * Creates the actual object from the binding plugin.
    */
    KJS::Object createBindingPlugin(KJS::ExecState *exec, const TQString &classname, const KJS::List &args );
    /**
     * Returns true iff the factory knows the type of proxy to use for the
     * class with the name specified.
     */
    bool isSupported( const TQString &clazz ) const;

    /**
     * Returns true iff the class with the specified name is handled with the
     * TQObject proxy type.
     */
    bool isTQObject( const TQString &clazz ) const;

    /**
     * Returns true iff the class with the specified name is handled with the
     * value proxy type.
     */
    bool isValue( const TQString &clazz ) const;

    /**
     * Returns true iff the class with the specified name is handled with the
     * opaque proxy type.
     */
    bool isOpaque( const TQString &clazz ) const;

    /**
     * Returns the ProxyType of the class with the specified name. If the
     * named class is not known to the interpreter then TypeInvalid is
     * returned.
     */
    uint proxyType( const TQString &clazz ) const;


    /**
     *Allows adding of an already loaded binding plugin for a certain class type
     *
     */
    void addTQObjectPlugin(const TQString &classname, KJSEmbed::Bindings::JSBindingPlugin* plugin);

    void addBindingsPlugin(KJS::ExecState *exec, KJS::Object &target) const;

    /**
     * Tells the factory the specified type of proxy to use for the named
     * class. Note that you can remove support for a type by using this method
     * with TypeInvalid.
     */
    void addType( const TQString &clazz, uint proxytype=JSFactory::TypeTQObject );

    /** Adds the types defined by the factory to the specified parent. */
    void addTypes( KJS::ExecState *exec, KJS::Object &parent );

    TQStringList types() const;

protected:
    /** Adds custom bindings to the specified proxy object. */
    KJS::Object extendProxy( KJS::ExecState *exec, KJS::Object &target ) const;

    /** Creates an instance of the named class and returns it in a JSValueProxy. */
    KJS::Object createValue( KJS::ExecState *exec, const TQString &cname, const KJS::List &args );

    /**
     * Creates an instance of a TQObject subclass. If the instance cannot be
     * created then 0 is returned.
     */
    TQObject *create( const TQString &classname, TQObject *parent=0, const char *name=0  );

    /** Creates an instance of the named class and returns it in a JSOpaqueProxy. */
    KJS::Object createOpaque( KJS::ExecState *exec, const TQString &cname, const KJS::List &args );


    /** Creates an instance of the named binding TQObject. */
    TQObject *createBinding( const TQString &cname, TQObject *parent, const char *name );

    /**
     * Creates an instance of the named TQObject. This method is only used for
     * non-widget objects.
     */
    TQObject *createObject( const TQString &cname, TQObject *parent, const char *name );

    /**
     * Creates an instance of the named TQWidget. Note that this method is only
     * used to create widgets that are not supported by TQWidgetFactory.
     */
    TQWidget *createWidget( const TQString &cname, TQWidget *parent, const char *name );

private:
    
    /** Adds the types defined by TQWidgetFactory to the specified parent. */
    void addWidgetFactoryTypes( KJS::ExecState *exec, KJS::Object &parent );

    /** Adds support for custom TQObject types to the specified parent. */
    void addCustomTypes( KJS::ExecState *exec, KJS::Object &parent );

    /** Adds support for TQObject binding types to the specified parent. */
    void addBindingTypes( KJS::ExecState *exec, KJS::Object &parent );

    /**
     * Adds support for any TQObject types that are known about, but have no
     * custom support (and no constructor).
     */
    void addObjectTypes( KJS::ExecState *exec, KJS::Object &parent );

    void addOpaqueTypes( KJS::ExecState *exec, KJS::Object &parent );

    void addValueTypes( KJS::ExecState *exec, KJS::Object &parent );

private:
    KJSEmbedPart *jspart;
    JSEventMapper *evmapper;
    TQMap<TQString,uint> objtypes;
    class JSFactoryPrivate *d;
};

} // namespace KJSEmbed

#endif // KJSEMBEDFACTORY_H

// Local Variables:
// c-basic-offset: 4
// End:
