
/* -*- c-basic-offset: 4 indent-tabs-mode: nil -*- vi:set ts=8 sts=4 sw=4: */

/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.

    This program is Copyright 2000-2008
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <richard.bown@ferventsoftware.com>

    The moral rights of Guillaume Laurent, Chris Cannam, and Richard
    Bown to claim authorship of this work have been asserted.

    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/

#ifndef _RG_BANKEDITORDIALOG_H_
#define _RG_BANKEDITORDIALOG_H_

#include "base/Device.h"
#include "base/MidiProgram.h"
#include <map>
#include <string>
#include <tdemainwindow.h>
#include <utility>


class TQWidget;
class TQString;
class TQPushButton;
class TQListViewItem;
class TQGroupBox;
class TQCloseEvent;
class TQCheckBox;
class TDEListView;
class KCommand;
class KComboBox;


namespace Rosegarden
{

class Studio;
class RosegardenGUIDoc;
class MultiViewCommandHistory;
class MidiProgramsEditor;
class MidiKeyMappingEditor;
class MidiDeviceListViewItem;
class MidiDevice;


class BankEditorDialog : public TDEMainWindow
{
    Q_OBJECT
  

public:
    BankEditorDialog(TQWidget *parent,
                     RosegardenGUIDoc *doc,
                     DeviceId defaultDevice =
                     Device::NO_DEVICE);

    ~BankEditorDialog();

    // Initialise the devices/banks and programs - the whole lot
    //
    void initDialog();

    std::pair<int, int> getFirstFreeBank(TQListViewItem*);

    void addCommandToHistory(KCommand *command);
    MultiViewCommandHistory* getCommandHistory();

    void setCurrentDevice(DeviceId device);

    // Get a MidiDevice from an index number
    //
    MidiDevice* getMidiDevice(DeviceId);
    MidiDevice* getMidiDevice(TQListViewItem*);
    MidiDevice* getCurrentMidiDevice();
    BankList&   getBankList()     { return m_bankList; }
    ProgramList&getProgramList()  { return m_programList; }

    Studio *getStudio() { return m_studio; }

    void setModified(bool value);

    void checkModified();

    // Set the listview to select a certain device - used after adding
    // or deleting banks.
    //
    void selectDeviceItem(MidiDevice *device);

    // Select a device/bank combination
    //
    void selectDeviceBankItem(DeviceId device, int bank);

public slots:
    void slotPopulateDevice(TQListViewItem*);

    void slotApply();
    void slotReset();

    void slotUpdate();

    void slotAddBank();
    void slotAddKeyMapping();
    void slotDelete();
    void slotDeleteAll();

    void slotImport();
    void slotExport();

    void slotModifyDeviceOrBankName(TQListViewItem*, const TQString&,int);

    void slotFileClose();

    void slotEditCopy();
    void slotEditPaste();

    void slotVariationToggled();
    void slotVariationChanged(int);

signals:
    void closing();
    void deviceNamesChanged();

protected:
    virtual void closeEvent(TQCloseEvent*);

    void resetProgramList();
    void setProgramList(MidiDevice *device);

    void updateDialog();

    void populateDeviceItem(TQListViewItem* deviceItem,
                            MidiDevice* midiDevice);

    void updateDeviceItem(MidiDeviceListViewItem* deviceItem);

    bool deviceItemHasBank(MidiDeviceListViewItem* deviceItem, int bankNb);

    void clearItemChildren(TQListViewItem* deviceItem);

    MidiDeviceListViewItem* getParentDeviceItem(TQListViewItem*);
    void keepBankListForNextPopulate() { m_keepBankList = true; }

    void populateDevice(TQListViewItem*);

    void setupActions();

    //--------------- Data members ---------------------------------
    Studio      *m_studio;
    RosegardenGUIDoc        *m_doc;

    MidiProgramsEditor      *m_programEditor;
    MidiKeyMappingEditor    *m_keyMappingEditor;
    TDEListView               *m_listView;

    TQGroupBox               *m_optionBox;
    TQCheckBox               *m_variationToggle;
    KComboBox               *m_variationCombo;

    TQPushButton             *m_closeButton;
    TQPushButton             *m_resetButton;
    TQPushButton             *m_applyButton;

    TQPushButton             *m_addBank;
    TQPushButton             *m_addKeyMapping;
    TQPushButton             *m_delete;
    TQPushButton             *m_deleteAll;

    TQPushButton             *m_importBanks;
    TQPushButton             *m_exportBanks;

    TQPushButton             *m_copyPrograms;
    TQPushButton             *m_pastePrograms;
    std::pair<DeviceId, int> m_copyBank;

    std::map<DeviceId, std::string>  m_deviceNameMap;
    BankList                         m_bankList;
    ProgramList                      m_programList;
    ProgramList                      m_oldProgramList;

    bool                     m_modified;
    bool                     m_keepBankList;
    bool                     m_deleteAllReally;

    DeviceId     m_lastDevice;
    MidiBank     m_lastBank;

    bool                     m_updateDeviceList;
};

// ----------------------- RemapInstrumentDialog ------------------------
//
//


}

#endif
