/* -*- c-basic-offset: 4 indent-tabs-mode: nil -*- vi:set ts=8 sts=4 sw=4: */

/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.
 
    This program is Copyright 2000-2008
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <richard.bown@ferventsoftware.com>
 
    The moral rights of Guillaume Laurent, Chris Cannam, and Richard
    Bown to claim authorship of this work have been asserted.
 
    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.
 
    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/


#include "ClefInserter.h"

#include <tdelocale.h>
#include "base/Event.h"
#include "base/NotationTypes.h"
#include "base/ViewElement.h"
#include "commands/notation/ClefInsertionCommand.h"
#include "gui/general/EditTool.h"
#include "gui/general/LinedStaff.h"
#include "NotationElement.h"
#include "NotationTool.h"
#include "NotationView.h"
#include "NotePixmapFactory.h"
#include <tdeaction.h>
#include <tqiconset.h>
#include <tqstring.h>


namespace Rosegarden
{

ClefInserter::ClefInserter(NotationView* view)
        : NotationTool("ClefInserter", view),
        m_clef(Clef::Treble)
{
    TQIconSet icon = TQIconSet(NotePixmapFactory::toTQPixmap(NotePixmapFactory::
                             makeToolbarPixmap("select")));
    new TDEAction(i18n("Switch to Select Tool"), icon, 0, this,
                TQT_SLOT(slotSelectSelected()), actionCollection(),
                "select");

    new TDEAction(i18n("Switch to Erase Tool"), "eraser", 0, this,
                TQT_SLOT(slotEraseSelected()), actionCollection(),
                "erase");

    icon = TQIconSet
           (NotePixmapFactory::toTQPixmap(NotePixmapFactory::
                                         makeToolbarPixmap("crotchet")));
    new TDEAction(i18n("Switch to Inserting Notes"), icon, 0, this,
                TQT_SLOT(slotNotesSelected()), actionCollection(),
                "notes");

    createMenu("clefinserter.rc");
}

void ClefInserter::slotNotesSelected()
{
    m_nParentView->slotLastNoteAction();
}

void ClefInserter::slotEraseSelected()
{
    m_parentView->actionCollection()->action("erase")->activate();
}

void ClefInserter::slotSelectSelected()
{
    m_parentView->actionCollection()->action("select")->activate();
}

void ClefInserter::ready()
{
    m_nParentView->setCanvasCursor(TQt::crossCursor);
    m_nParentView->setHeightTracking(false);
}

void ClefInserter::setClef(std::string clefType)
{
    m_clef = clefType;
}

void ClefInserter::handleLeftButtonPress(timeT,
        int,
        int staffNo,
        TQMouseEvent* e,
        ViewElement*)
{
    if (staffNo < 0)
        return ;
    Event *clef = 0, *key = 0;

    LinedStaff *staff = m_nParentView->getLinedStaff(staffNo);

    NotationElementList::iterator closestElement =
        staff->getClosestElementToCanvasCoords(e->x(), (int)e->y(),
                                               clef, key, false, -1);

    if (closestElement == staff->getViewElementList()->end())
        return ;

    timeT time = (*closestElement)->event()->getAbsoluteTime(); // not getViewAbsoluteTime()


    ClefInsertionCommand *command =
        new ClefInsertionCommand(staff->getSegment(), time, m_clef);

    m_nParentView->addCommandToHistory(command);

    Event *event = command->getLastInsertedEvent();
    if (event)
        m_nParentView->setSingleSelectedEvent(staffNo, event);
}

const TQString ClefInserter::ToolName     = "clefinserter";

}
#include "ClefInserter.moc"
