/***************************************************************************
 *   Copyright (C) 2012-2013 by Timothy Pearson                            *
 *   kb9vqf@pearsoncomputing.net                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netdb.h>
#include <pwd.h>

#include <tqfile.h>
#include <tqcheckbox.h>
#include <tdeapplication.h>

#include <tdelocale.h>
#include <tdemessagebox.h>
#include <klineedit.h>
#include <kpassdlg.h>
#include <ksimpleconfig.h>
#include <tdesu/process.h>
#include <ksslcertificate.h>
#include <krfcdate.h>

#include <ldap.h>
#include <sasl/sasl.h>
#include <stdlib.h>
#include <sys/time.h>
#include <errno.h>

#include "libtdeldap.h"
#include "ldaplogindlg.h"
#include "ldappasswddlg.h"

#define LDAP_INSECURE_PORT 389
#define LDAP_SECURE_PORT 636

// FIXME
// Connect this to CMake/Automake
#define KDE_CONFDIR "/etc/trinity"

// FIXME
// This assumes Debian!
#define KRB5_FILE "/etc/krb5.conf"

#define NSSWITCH_FILE "/etc/nsswitch.conf"

#define PAMD_DIRECTORY "/etc/pam.d/"
#define PAMD_COMMON_ACCOUNT "common-account"
#define PAMD_COMMON_AUTH "common-auth"
#define PAMD_COMMON_SESSION "common-session"

#define LDAP_FILE "/etc/ldap/ldap.conf"
#define LDAP_SECONDARY_FILE "/etc/ldap.conf"
#define LDAP_TERTIARY_FILE "/etc/libnss-ldap.conf"

#define TDELDAP_SUDO_D_FILE "/etc/sudoers.d/tde-realm-admins"

#define CRON_UPDATE_NSS_FILE "/etc/cron.daily/upd-local-nss-db"
#define CRON_UPDATE_NSS_COMMAND "/usr/sbin/nss_updatedb ldap"

// FIXME
// This assumes Debian!
#define CRON_UPDATE_PRIMARY_REALM_CERTIFICATES_OPENLDAP_RELOAD_COMMAND "/etc/init.d/slapd force-reload"

int requested_ldap_version = LDAP_VERSION3;
char* ldap_user_and_operational_attributes[2] = {"*", "+"};

enum ErrorCauseLocation {
	ERRORCAUSE_LOCATION_BIND = 0
};

bool fileExists(const char* filename) {
	struct stat sts;
	if (stat(filename, &sts) == -1 && errno == ENOENT) {
		return false;
	}
	else {
		return true;
	}
}

LDAPManager::LDAPManager(TQString realm, TQString host, TQObject *parent, const char *name) : TQObject(parent, name), m_realm(realm), m_host(host), m_port(0), m_creds(0), m_ldap(0)
{
	TQStringList domainChunks = TQStringList::split(".", realm.lower());
	m_basedc = "dc=" + domainChunks.join(",dc=");
}

LDAPManager::LDAPManager(TQString realm, TQString host, LDAPCredentials* creds, TQObject *parent, const char *name) : TQObject(parent, name), m_realm(realm), m_host(host), m_port(0), m_creds(creds), m_ldap(0)
{
	TQStringList domainChunks = TQStringList::split(".", realm.lower());
	m_basedc = "dc=" + domainChunks.join(",dc=");
}

LDAPManager::~LDAPManager() {
	unbind(true);
}

TQString LDAPManager::detailedKAdminErrorMessage(TQString initialMessage) {
	if (initialMessage.contains("Looping detected inside krb5_get_in_tkt")) {
		initialMessage.append("<p>").append(i18n("Potential causes")).append(":<br>").append(i18n(" * Invalid credentials")).append("<br>").append(i18n(" * Clock skew between the realm's KDC and this machine")).append("<br>").append(i18n(" * Inability to negotiate a compatible encryption type with the realm's KDC")).append("<br>").append(i18n(" * No connectivity to the realm's KDC"));
	}
	return initialMessage;
}

TQString LDAPManager::ldapdnForRealm(TQString realm) {
	TQStringList domainChunks = TQStringList::split(".", realm.lower());
	TQString basedc = "dc=" + domainChunks.join(",dc=");
	return basedc;
}

TQString LDAPManager::cnFromDn(TQString dn) {
	int eqpos = dn.find("=")+1;
	int cmpos = dn.find(",", eqpos);
	if ((eqpos < 0) || (cmpos < 0)) {
		return dn;
	}
	dn.truncate(cmpos);
	dn.remove(0, eqpos);
	return dn;
}

TQString LDAPManager::basedn() {
	return m_basedc;
}

TQString LDAPManager::realm() {
	return m_realm;
}

LDAPCredentials LDAPManager::currentLDAPCredentials() {
	if (m_creds) {
		return *m_creds;
	}
	else {
		return LDAPCredentials();
	}
}

TQString ldapLikelyErrorCause(int errcode, int location) {
	TQString ret;

	if (location == ERRORCAUSE_LOCATION_BIND) {
		if (errcode == LDAP_SERVER_DOWN) {
			ret = " * LDAP server down<br> * Invalid LDAP Certificate Authority file on client";
		}
		if (LDAP_NAME_ERROR(errcode)) {
			ret = "Unknown user name or incorrect user name format";
		}
	}

	if (ret != "") {
		if (ret.contains("<br>")) {
			ret.prepend("<p>" + i18n("Potential causes") + ":<br>");
		}
		else {
			ret.prepend("<p>" + i18n("Potential cause") + ":<br>");
		}
	}

	return ret;
}

int sasl_bind_interact_callback(LDAP* ld, unsigned flags, void* defaults, void* sasl_interaction_struct) {
	// FIXME
	// This currently does nothing and hopes for the best!
	// sasl_interact* sasl_struct = (sasl_interact*)sasl_interaction_struct;

	return LDAP_SUCCESS;
}

int LDAPManager::bind(TQString* errstr) {
	if (m_ldap) {
		return 0;
	}

	KerberosTicketInfoList m_krbTickets = LDAPManager::getKerberosTicketList();

	bool using_ldapi = false;
	if (m_host.startsWith("ldapi://")) {
		using_ldapi = true;
	}
	bool havepass = false;
	if (m_creds || using_ldapi) {
		havepass = true;
	}
	else {
		LDAPPasswordDialog passdlg(0, 0, (m_krbTickets.count() > 0));
		passdlg.m_base->ldapAdminRealm->setEnabled(false);
		passdlg.m_base->ldapAdminRealm->insertItem(m_realm);
		passdlg.m_base->ldapUseTLS->setChecked(true);
		if (passdlg.exec() == TQDialog::Accepted) {
			havepass = true;
			if (!m_creds) {
				m_creds = new LDAPCredentials();
				m_creds->username = passdlg.m_base->ldapAdminUsername->text();
				m_creds->password = passdlg.m_base->ldapAdminPassword->password();
				m_creds->realm = passdlg.m_base->ldapAdminRealm->currentText();
				m_creds->use_tls = passdlg.m_base->ldapUseTLS->isOn();
				m_creds->use_gssapi = passdlg.use_gssapi;
			}
		}
		else {
			return -1;
		}
	}

	TQString uri;
	if (m_host.contains("://")) {
		uri = m_host;
		if (!m_creds) {
			m_creds = new LDAPCredentials();
			m_creds->username = "";
			m_creds->password = "";
			m_creds->realm = m_realm;
		}
	}
	else {
		if (m_creds->use_tls) {
			m_port = LDAP_SECURE_PORT;
			uri = TQString("ldaps://%1:%2").arg(m_host).arg(m_port);
		}
		else {
			m_port = LDAP_INSECURE_PORT;
			uri = TQString("ldap://%1:%2").arg(m_host).arg(m_port);
		}
	}

	int retcode = ldap_initialize(&m_ldap, uri.ascii());
	if (retcode < 0) {
		if (errstr) *errstr = i18n("<qt>Unable to connect to LDAP server %1 on port %2<p>Reason: [%3] %4%5</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode)).arg(ldapLikelyErrorCause(retcode, ERRORCAUSE_LOCATION_BIND));
		else KMessageBox::error(0, i18n("<qt>Unable to connect to LDAP server %1 on port %2<p>Reason: [%3] %4%5</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode)).arg(ldapLikelyErrorCause(retcode, ERRORCAUSE_LOCATION_BIND)), i18n("Unable to connect to server!"));
		return -1;
	}
	retcode = ldap_set_option(m_ldap, LDAP_OPT_PROTOCOL_VERSION, &requested_ldap_version);
	if (retcode != LDAP_OPT_SUCCESS) {
		if (errstr) *errstr = i18n("<qt>Unable to connect to LDAP server %1 on port %2<p>Reason: [%3] %4%5</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode)).arg(ldapLikelyErrorCause(retcode, ERRORCAUSE_LOCATION_BIND));
		else KMessageBox::error(0, i18n("<qt>Unable to connect to LDAP server %1 on port %2<p>Reason: [%3] %4%5</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode)).arg(ldapLikelyErrorCause(retcode, ERRORCAUSE_LOCATION_BIND)), i18n("Unable to connect to server!"));
		return -1;
	}

	TQString errorString;
	if (havepass == true) {
		char* mechanism = NULL;
		struct berval cred;
		TQString ldap_dn = m_creds->username;
		TQCString pass = m_creds->password;
		cred.bv_val = pass.data();
		cred.bv_len = pass.length();
		if ((!using_ldapi && !m_creds->use_gssapi)) {
			if (!ldap_dn.contains(",")) {
				// Look for a POSIX account with anonymous bind and the specified account name
				TQString uri;
				LDAP* ldapconn;
				if (m_host.contains("://")) {
					uri = m_host;
				}
				else {
					if (m_creds->use_tls) {
						m_port = LDAP_SECURE_PORT;
						uri = TQString("ldaps://%1:%2").arg(m_host).arg(m_port);
					}
					else {
						m_port = LDAP_INSECURE_PORT;
						uri = TQString("ldap://%1:%2").arg(m_host).arg(m_port);
					}
				}
				int retcode = ldap_initialize(&ldapconn, uri.ascii());
				if (retcode < 0) {
					if (errstr) *errstr = i18n("<qt>Unable to connect to LDAP server %1 on port %2<p>Reason: [%3] %4</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode));
					else KMessageBox::error(0, i18n("<qt>Unable to connect to LDAP server %1 on port %2<p>Reason: [%3] %4</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode)), i18n("Unable to connect to server!"));
					return -1;
				}
				retcode = ldap_set_option(ldapconn, LDAP_OPT_PROTOCOL_VERSION, &requested_ldap_version);
				if (retcode != LDAP_OPT_SUCCESS) {
					if (errstr) *errstr = i18n("<qt>Unable to connect to LDAP server %1 on port %2<p>Reason: [%3] %4</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode));
					else KMessageBox::error(0, i18n("<qt>Unable to connect to LDAP server %1 on port %2<p>Reason: [%3] %4</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode)), i18n("Unable to connect to server!"));
					return -1;
				}
				struct berval anoncred;
				anoncred.bv_val = "";
				anoncred.bv_len = strlen("");
				retcode = ldap_sasl_bind_s(ldapconn, "", mechanism, &anoncred, NULL, NULL, NULL);
				if (retcode == LDAP_SUCCESS ) {
					// Look for the DN for the specified user
					LDAPMessage* msg;
					TQString ldap_base_dn = m_basedc;
					TQString ldap_filter = TQString("(&(objectclass=posixAccount)(uid=%1))").arg(m_creds->username);
					retcode = ldap_search_ext_s(ldapconn, ldap_base_dn.ascii(), LDAP_SCOPE_SUBTREE, ldap_filter.ascii(), NULL, 0, NULL, NULL, NULL, 0, &msg);
					if (retcode != LDAP_SUCCESS) {
						if (errstr) *errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
						else KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					}
					else {
						// Iterate through the returned entries
						char* dn = NULL;
						LDAPMessage* entry;
						for(entry = ldap_first_entry(ldapconn, msg); entry != NULL; entry = ldap_next_entry(ldapconn, entry)) {
							if((dn = ldap_get_dn(ldapconn, entry)) != NULL) {
								ldap_dn = dn;
								ldap_memfree(dn);
							}
						}
					}
					// Clean up
					ldap_msgfree(msg);
	
					// All done!
					ldap_unbind_ext_s(ldapconn, NULL, NULL);
				}
				else {
					// Clean up
					ldap_unbind_ext_s(ldapconn, NULL, NULL);
				}
			}
		}

		if (m_creds->use_gssapi) {
#if LDAP_VENDOR_VERSION < 20425
			retcode = ldap_sasl_interactive_bind_s(m_ldap, "", "GSSAPI", NULL, NULL, LDAP_SASL_AUTOMATIC, sasl_bind_interact_callback, NULL);
#else // LDAP_VENDOR_VERSION
			const char* rmech = NULL;
			LDAPMessage* result = NULL;
			int msgid;
			retcode = LDAP_SASL_BIND_IN_PROGRESS;
			while (retcode == LDAP_SASL_BIND_IN_PROGRESS) {
				retcode = ldap_sasl_interactive_bind(m_ldap, "", "GSSAPI", NULL, NULL, LDAP_SASL_AUTOMATIC, sasl_bind_interact_callback, NULL, result, &rmech, &msgid);
				ldap_msgfree(result);

				if (retcode != LDAP_SASL_BIND_IN_PROGRESS) {
					break;
				}

				if ((ldap_result(m_ldap, msgid, LDAP_MSG_ALL, NULL, &result) == -1) || (!result)) {
					retcode = LDAP_INVALID_CREDENTIALS;
				}
			}

			if (retcode == LDAP_SUCCESS) {
				if (m_creds->username == "") {
					char* sasluser;
					ldap_get_option(m_ldap, LDAP_OPT_X_SASL_USERNAME, &sasluser);
					if (sasluser) {
						TQStringList principalParts = TQStringList::split("@", TQString(sasluser), false);
						m_creds->username = principalParts[0];
						m_creds->realm = principalParts[1];
						ldap_memfree(sasluser);
					}
				}
			}
#endif // LDAP_VENDOR_VERSION
		}
		else {
			retcode = ldap_sasl_bind_s(m_ldap, ldap_dn.ascii(), mechanism, &cred, NULL, NULL, NULL);
		}

		if (retcode != LDAP_SUCCESS ) {
			if (errstr) *errstr = i18n("<qt>Unable to connect to LDAP server %1 on port %2<p>Reason: [%3] %4%5</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode)).arg(ldapLikelyErrorCause(retcode, ERRORCAUSE_LOCATION_BIND));
			else KMessageBox::error(0, i18n("<qt>Unable to connect to LDAP server %1 on port %2<p>Reason: [%3] %4%5</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode)).arg(ldapLikelyErrorCause(retcode, ERRORCAUSE_LOCATION_BIND)), i18n("Unable to connect to server!"));
			return -1;
		}

		return 0;
	}
	else {
		return -2;
	}

	return -3;
}

int LDAPManager::unbind(bool force, TQString* errstr) {
	if (!m_ldap) {
		return 0;
	}

	int retcode = ldap_unbind_ext_s(m_ldap, NULL, NULL);
	if ((retcode < 0) && (force == false)) {
		if (errstr) *errstr = i18n("<qt>Unable to disconnect from LDAP server %1 on port %2<p>Reason: [%3] %4</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode));
		else KMessageBox::error(0, i18n("<qt>Unable to disconnect from LDAP server %1 on port %2<p>Reason: [%3] %4</qt>").arg(m_host).arg(m_port).arg(retcode).arg(ldap_err2string(retcode)), i18n("Unable to disconnect from server!"));
		return retcode;
	}
	else {
		m_ldap = 0;
	}
	return retcode;
}

LDAPUserInfo LDAPManager::parseLDAPUserRecord(LDAPMessage* entry) {
	int i;
	char* dn = NULL;
	char* attr;
	struct berval **vals;
	BerElement* ber;

	LDAPUserInfo userinfo;

	if((dn = ldap_get_dn(m_ldap, entry)) != NULL) {
		userinfo.distinguishedName = dn;
		TQStringList dnParts = TQStringList::split(",", dn);
		TQString id = dnParts[0];
		if (id.startsWith("uid=")) {
			id = id.remove(0, 4);
			userinfo.name = id;
		}
		ldap_memfree(dn);
	}

	for( attr = ldap_first_attribute(m_ldap, entry, &ber); attr != NULL; attr = ldap_next_attribute(m_ldap, entry, ber)) {
		if ((vals = ldap_get_values_len(m_ldap, entry, attr)) != NULL)  {
			userinfo.informationValid = true;
			TQString ldap_field = attr;
			i=0;
			if (ldap_field == "creatorsName") {
				userinfo.creatorsName = vals[i]->bv_val;
			}
			else if (ldap_field == "uidNumber") {
				userinfo.uid = atoi(vals[i]->bv_val);
			}
			else if (ldap_field == "loginShell") {
				userinfo.shell = vals[i]->bv_val;
			}
			else if (ldap_field == "homeDirectory") {
				userinfo.homedir = vals[i]->bv_val;
			}
			else if (ldap_field == "gidNumber") {
				userinfo.primary_gid = atoi(vals[i]->bv_val);
			}
			else if (ldap_field == "tdeBuiltinAccount") {
				userinfo.tde_builtin_account = (TQString(vals[i]->bv_val).upper() == "TRUE")?true:false;
			}
			else if (ldap_field == "krb5KDCFlags") {
				userinfo.status = (LDAPKRB5Flags)(atoi(vals[i]->bv_val));
			}
			else if (ldap_field == "createTimestamp") {		// YYYYMMDD000000Z
				TQString formattedDate = vals[i]->bv_val;
				formattedDate.insert(4,"-");
				formattedDate.insert(7,"-");
				formattedDate.insert(10,"T");
				formattedDate.insert(13,":");
				formattedDate.insert(16,":");
				formattedDate.remove(19, 1);
				userinfo.account_created = TQDateTime::fromString(formattedDate, TQt::ISODate);
			}
			else if (ldap_field == "modifyTimestamp") {		// YYYYMMDD000000Z
				TQString formattedDate = vals[i]->bv_val;
				formattedDate.insert(4,"-");
				formattedDate.insert(7,"-");
				formattedDate.insert(10,"T");
				formattedDate.insert(13,":");
				formattedDate.insert(16,":");
				formattedDate.remove(19, 1);
				userinfo.account_modified = TQDateTime::fromString(formattedDate, TQt::ISODate);
			}
				// FIXME
				// These two attributes do not seem to be available with a Heimdal KDC
				// userinfo.password_last_changed = vals[i]->bv_val;
				// userinfo.password_expires = vals[i]->bv_val;
			else if (ldap_field == "krb5PasswordEnd") {		// YYYYMMDD000000Z
				TQString formattedDate = vals[i]->bv_val;
				formattedDate.insert(4,"-");
				formattedDate.insert(7,"-");
				formattedDate.insert(10,"T");
				formattedDate.insert(13,":");
				formattedDate.insert(16,":");
				formattedDate.remove(19, 1);
				userinfo.password_expiration = TQDateTime::fromString(formattedDate, TQt::ISODate);
			}
				// FIXME
				// These six(!) attributes do not seem to be available with a Heimdal KDC
				// userinfo.password_ages = vals[i]->bv_val;
				// userinfo.new_password_interval = vals[i]->bv_val;
				// userinfo.new_password_warn_interval = vals[i]->bv_val;
				// userinfo.new_password_lockout_delay = vals[i]->bv_val;
				// userinfo.password_has_minimum_age = vals[i]->bv_val;
				// userinfo.password_minimum_age = vals[i]->bv_val;
			else if (ldap_field == "krb5MaxLife") {			// units: hours
				userinfo.maximum_ticket_lifetime = atoi(vals[i]->bv_val);
			}
			else if (ldap_field == "cn") {
				userinfo.commonName = vals[i]->bv_val;
			}
			else if (ldap_field == "givenName") {
				userinfo.givenName = vals[i]->bv_val;
			}
			else if (ldap_field == "sn") {
				userinfo.surName = vals[i]->bv_val;
			}
			else if (ldap_field == "initials") {
				userinfo.initials = vals[i]->bv_val;
			}
			else if (ldap_field == "title") {
				userinfo.title = vals[i]->bv_val;
			}
			else if (ldap_field == "mail") {
				userinfo.email = vals[i]->bv_val;
			}
			else if (ldap_field == "description") {
				userinfo.description = vals[i]->bv_val;
			}
			else if (ldap_field == "l") {
				userinfo.locality = vals[i]->bv_val;
			}
			else if (ldap_field == "telephoneNumber") {
				userinfo.telephoneNumber = vals[i]->bv_val;
			}
			else if (ldap_field == "facsimileTelephoneNumber") {
				userinfo.faxNumber = vals[i]->bv_val;
			}
			else if (ldap_field == "homePhone") {
				userinfo.homePhone = vals[i]->bv_val;
			}
			else if (ldap_field == "mobile") {
				userinfo.mobilePhone = vals[i]->bv_val;
			}
			else if (ldap_field == "pager") {
				userinfo.pagerNumber = vals[i]->bv_val;
			}
			else if (ldap_field == "websiteURL") {
				userinfo.website = vals[i]->bv_val;
			}
			else if (ldap_field == "postOfficeBox") {
				userinfo.poBox = vals[i]->bv_val;
			}
			else if (ldap_field == "street") {
				userinfo.street = vals[i]->bv_val;
			}
			else if (ldap_field == "postalAddress") {
				userinfo.address = vals[i]->bv_val;
			}
			else if (ldap_field == "st") {
				userinfo.state = vals[i]->bv_val;
			}
			else if (ldap_field == "postalCode") {
				userinfo.postcode = vals[i]->bv_val;
			}
			else if (ldap_field == "registeredAddress") {
				userinfo.registeredAddress = vals[i]->bv_val;
			}
			else if (ldap_field == "homePostalAddress") {
				userinfo.homeAddress = vals[i]->bv_val;
			}
			else if (ldap_field == "seeAlso") {
				userinfo.seeAlso = vals[i]->bv_val;
			}
			else if (ldap_field == "physicalDeliveryOfficeName") {
				userinfo.deliveryOffice = vals[i]->bv_val;
			}
			else if (ldap_field == "departmentNumber") {
				userinfo.department = vals[i]->bv_val;
			}
			else if (ldap_field == "roomNumber") {
				userinfo.roomNumber = vals[i]->bv_val;
			}
			else if (ldap_field == "employeeType") {
				userinfo.employeeType = vals[i]->bv_val;
			}
			else if (ldap_field == "employeeNumber") {
				userinfo.employeeNumber = vals[i]->bv_val;
			}
			else if (ldap_field == "managerName") {
				userinfo.manager = vals[i]->bv_val;
			}
			else if (ldap_field == "secretaryName") {
				userinfo.secretary = vals[i]->bv_val;
			}
			else if (ldap_field == "internationaliSDNNumber") {
				userinfo.isdnNumber = vals[i]->bv_val;
			}
			else if (ldap_field == "teletexId") {
				userinfo.teletexID = vals[i]->bv_val;
			}
			else if (ldap_field == "telexNumber") {
				userinfo.telexNumber = vals[i]->bv_val;
			}
			else if (ldap_field == "preferredDelivery") {
				userinfo.preferredDelivery = vals[i]->bv_val;
			}
			else if (ldap_field == "destinationIndicator") {
				userinfo.destinationIndicator = vals[i]->bv_val;
			}
			else if (ldap_field == "x121Address") {
				userinfo.x121Address = vals[i]->bv_val;
			}
			else if (ldap_field == "displayName") {
				userinfo.displayName = vals[i]->bv_val;
			}
			else if (ldap_field == "preferredLanguage") {
				userinfo.preferredLanguage = vals[i]->bv_val;
			}
			else if (ldap_field == "locallyUniqueID") {
				userinfo.uniqueIdentifier = vals[i]->bv_val;
			}
			else if (ldap_field == "businessCategory") {
				userinfo.businessCategory = vals[i]->bv_val;
			}
			else if (ldap_field == "carLicense") {
				userinfo.carLicense = vals[i]->bv_val;
			}
			else if (ldap_field == "notes") {
				userinfo.notes = vals[i]->bv_val;
			}
			ldap_value_free_len(vals);
		}
		ldap_memfree(attr);
	}

	if (ber != NULL) {
		ber_free(ber, 0);
	}

	return userinfo;
}

LDAPUserInfoList LDAPManager::users(int* mretcode, TQString *errstr) {
	int retcode;
	int errcode;
	LDAPUserInfoList users;

	if (bind() < 0) {
		if (mretcode) *mretcode = -1;
		return LDAPUserInfoList();
	}
	else {
		LDAPMessage* msg;
		TQString ldap_base_dn = m_basedc;
		TQString ldap_filter = "(objectClass=posixAccount)";

		retcode = ldap_search_ext_s(m_ldap, ldap_base_dn.ascii(), LDAP_SCOPE_SUBTREE, ldap_filter.ascii(), ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if ((retcode != LDAP_SUCCESS) && (retcode != LDAP_SIZELIMIT_EXCEEDED)) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			if (mretcode) *mretcode = -1;
			return LDAPUserInfoList();
		}
		else if (retcode == LDAP_SUCCESS) {
			// Iterate through the returned entries
			LDAPMessage* entry;
			for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
				users.append(parseLDAPUserRecord(entry));
			}

			// Clean up
			ldap_msgfree(msg);
	
			if (mretcode) *mretcode = 0;
			return users;
		}
		else if (retcode == LDAP_SIZELIMIT_EXCEEDED) {
			// Try paged access
			bool morePages = false;
			unsigned long pageSize = 100;
			struct berval cookie = {0, NULL};
			char pagingCriticality = 'T';
			LDAPControl* pageControl = NULL;
			LDAPControl* serverControls[2] = { NULL, NULL };
			LDAPControl** returnedControls = NULL;
	
			do {
				retcode = ldap_create_page_control(m_ldap, pageSize, &cookie, pagingCriticality, &pageControl);
				if (retcode != LDAP_SUCCESS) {
					if (errstr) {
						*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					}
					else {
						KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					}
					if (mretcode) *mretcode = -1;
					return LDAPUserInfoList();
				}
				serverControls[0] = pageControl;
				retcode = ldap_search_ext_s(m_ldap, ldap_base_dn.ascii(), LDAP_SCOPE_SUBTREE, ldap_filter.ascii(), ldap_user_and_operational_attributes, 0, serverControls, NULL, NULL, 0, &msg);
				if ((retcode != LDAP_SUCCESS) && (retcode != LDAP_PARTIAL_RESULTS)) {
					if (errstr) {
						*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					}
					else {
						KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					}
					if (mretcode) *mretcode = -1;
					return LDAPUserInfoList();
				}
				retcode = ldap_parse_result(m_ldap, msg, &errcode, NULL, NULL, NULL, &returnedControls, false);
				if (retcode != LDAP_SUCCESS) {
					if (errstr) {
						*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					}
					else {
						KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					}
					if (mretcode) *mretcode = -1;
					return LDAPUserInfoList();
				}
				if (cookie.bv_val != NULL) {
					ber_memfree(cookie.bv_val);
					cookie.bv_val = NULL;
					cookie.bv_len = 0;
				}
				if (!!returnedControls) {
					retcode = ldap_parse_pageresponse_control(m_ldap, returnedControls[0], NULL, &cookie);
					morePages = (cookie.bv_val && (strlen(cookie.bv_val) > 0));
				}
				else {
					morePages = false;
				}
	
				if (returnedControls != NULL) {
					ldap_controls_free(returnedControls);
					returnedControls = NULL;
				}
				serverControls[0] = NULL;
				ldap_control_free(pageControl);
				pageControl = NULL;
	
				// Iterate through the returned entries
				LDAPMessage* entry;
				for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
					users.append(parseLDAPUserRecord(entry));
				}

				// Clean up
				ldap_msgfree(msg);
			} while (morePages);
	
			if (mretcode) *mretcode = 0;
			return users;
		}
	}

	return LDAPUserInfoList();
}

LDAPUserInfo LDAPManager::getUserByDistinguishedName(TQString dn) {
	int retcode;
	LDAPUserInfo userinfo;

	if (bind() < 0) {
		return LDAPUserInfo();
	}
	else {
		LDAPMessage* msg;
		retcode = ldap_search_ext_s(m_ldap, dn.ascii(), LDAP_SCOPE_SUBTREE, NULL, ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if (retcode != LDAP_SUCCESS) {
			KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return LDAPUserInfo();
		}

		// Iterate through the returned entries
		LDAPMessage* entry;
		for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
			userinfo = parseLDAPUserRecord(entry);
		}

		// Clean up
		ldap_msgfree(msg);

		return userinfo;
	}

	return LDAPUserInfo();
}

LDAPGroupInfo LDAPManager::getGroupByDistinguishedName(TQString dn, TQString *errstr) {
	int retcode;
	LDAPGroupInfo groupinfo;

	if (bind(errstr) < 0) {
		return LDAPGroupInfo();
	}
	else {
		LDAPMessage* msg;
		retcode = ldap_search_ext_s(m_ldap, dn.ascii(), LDAP_SCOPE_SUBTREE, NULL, ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if (retcode != LDAP_SUCCESS) {
			if (errstr) *errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			else KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return LDAPGroupInfo();
		}

		// Iterate through the returned entries
		LDAPMessage* entry;
		for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
			groupinfo = parseLDAPGroupRecord(entry);
		}

		// Clean up
		ldap_msgfree(msg);

		return groupinfo;
	}

	return LDAPGroupInfo();
}

void create_single_attribute_operation(LDAPMod **mods, int *i, TQString attr, TQString value) {
	if (value != "") {
		char **values = (char**)malloc(2*sizeof(char*));
		values[0] = strdup(value.ascii());
		values[1] = NULL;
		mods[*i]->mod_op = LDAP_MOD_ADD;
		mods[*i]->mod_type = strdup(attr.ascii());
		mods[*i]->mod_values = values;
		(*i)++;
	}
}

void create_multiple_attributes_operation(LDAPMod **mods, int *i, TQString attr, TQStringList strings) {
	int j=0;
	char **values = (char**)malloc((strings.count()+1)*sizeof(char*));
	for ( TQStringList::Iterator it = strings.begin(); it != strings.end(); ++it ) {
		if ((*it) != "") {
			values[j] = strdup((*it).ascii());
			j++;
		}
	}
	values[j] = NULL;
	mods[*i]->mod_op = LDAP_MOD_ADD;
	mods[*i]->mod_type = strdup(attr.ascii());
	mods[*i]->mod_values = values;
	(*i)++;
}

void add_single_attribute_operation(LDAPMod **mods, int *i, TQString attr, TQString value) {
	if (value != "") {
		char **values = (char**)malloc(2*sizeof(char*));
		values[0] = strdup(value.ascii());
		values[1] = NULL;
		mods[*i]->mod_op = LDAP_MOD_REPLACE;
		mods[*i]->mod_type = strdup(attr.ascii());
		mods[*i]->mod_values = values;
		(*i)++;
	}
}

void add_single_binary_attribute_operation(LDAPMod **mods, int *i, TQString attr, TQByteArray &ba) {
	if (ba.size() > 0) {
		struct berval **values = (berval**)malloc(2*sizeof(berval*));
		values[0] = new berval;
		values[0]->bv_len = ba.size();
		values[0]->bv_val = ba.data();
		values[1] = NULL;
		mods[*i]->mod_op = LDAP_MOD_REPLACE|LDAP_MOD_BVALUES;
		mods[*i]->mod_type = strdup(attr.ascii());
		mods[*i]->mod_bvalues = values;
		(*i)++;
	}
}

void add_multiple_attributes_operation(LDAPMod **mods, int *i, TQString attr, TQStringList strings) {
	int j=0;
	char **values = (char**)malloc((strings.count()+1)*sizeof(char*));
	for ( TQStringList::Iterator it = strings.begin(); it != strings.end(); ++it ) {
		if ((*it) != "") {
			values[j] = strdup((*it).ascii());
			j++;
		}
	}
	values[j] = NULL;
	mods[*i]->mod_op = LDAP_MOD_REPLACE;
	mods[*i]->mod_type = strdup(attr.ascii());
	mods[*i]->mod_values = values;
	(*i)++;
}

void delete_single_attribute_operation(LDAPMod **mods, int *i, TQString attr) {
	mods[*i]->mod_op = LDAP_MOD_DELETE;
	mods[*i]->mod_type = strdup(attr.ascii());
	(*i)++;
}

void set_up_attribute_operations(LDAPMod **mods, int number_of_parameters) {
	int i;
	for (i=0;i<number_of_parameters;i++) {
		mods[i] = new LDAPMod;
		mods[i]->mod_type = NULL;
		mods[i]->mod_values = NULL;
	}
	mods[number_of_parameters] = NULL;
}

void clean_up_attribute_operations(int i, LDAPMod **mods, LDAPMod *prevterm, int number_of_parameters) {
	mods[i] = prevterm;
	for (i=0;i<number_of_parameters;i++) {
		if (mods[i]->mod_type != NULL) {
			free(mods[i]->mod_type);
		}
		if (mods[i]->mod_values != NULL) {
			int j = 0;
			while (mods[i]->mod_values[j] != NULL) {
				free(mods[i]->mod_values[j]);
				j++;
			}
			free(mods[i]->mod_values);
		}
		delete mods[i];
	}
}

int LDAPManager::updateUserInfo(LDAPUserInfo user, TQString *errstr) {
	int retcode;
	int i;
	LDAPUserInfo userinfo;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Assemble the LDAPMod structure
		// We will replace any existing attributes with the new values
		int number_of_parameters = 40;				// 40 primary attributes
		LDAPMod *mods[number_of_parameters+1];
		set_up_attribute_operations(mods, number_of_parameters);

		// Load LDAP modification requests from provided data structure
		i=0;
		add_single_attribute_operation(mods, &i, "uidNumber", TQString("%1").arg(user.uid));
		add_single_attribute_operation(mods, &i, "loginShell", user.shell);
		add_single_attribute_operation(mods, &i, "homeDirectory", user.homedir);
		add_single_attribute_operation(mods, &i, "userPassword", "{SASL}" + user.name + "@" + m_realm.upper());
		add_single_attribute_operation(mods, &i, "gidNumber", TQString("%1").arg(user.primary_gid));
		add_single_attribute_operation(mods, &i, "krb5KDCFlags", TQString("%1").arg(user.status));			// Default active user is 586 [KRB5_ACTIVE_DEFAULT] and locked out user is 7586 [KRB5_DISABLED_ACCOUNT]
// 		add_single_attribute_operation(mods, &i, "", user.password_expires);
// 		add_single_attribute_operation(mods, &i, "", user.password_expiration);
// 		add_single_attribute_operation(mods, &i, "", user.password_ages);
// 		add_single_attribute_operation(mods, &i, "", user.new_password_interval);
// 		add_single_attribute_operation(mods, &i, "", user.new_password_warn_interval);
// 		add_single_attribute_operation(mods, &i, "", user.new_password_lockout_delay);
// 		add_single_attribute_operation(mods, &i, "", user.password_has_minimum_age);
// 		add_single_attribute_operation(mods, &i, "", user.password_minimum_age);
		add_single_attribute_operation(mods, &i, "krb5MaxLife", TQString("%1").arg(user.maximum_ticket_lifetime));
		add_single_attribute_operation(mods, &i, "cn", user.commonName);
		add_single_attribute_operation(mods, &i, "givenName", user.givenName);
		add_single_attribute_operation(mods, &i, "sn", user.surName);
		add_single_attribute_operation(mods, &i, "initials", user.initials);
		add_single_attribute_operation(mods, &i, "title", user.title);
		add_single_attribute_operation(mods, &i, "mail", user.email);
		add_single_attribute_operation(mods, &i, "description", user.description);
		add_single_attribute_operation(mods, &i, "l", user.locality);
		add_single_attribute_operation(mods, &i, "telephoneNumber", user.telephoneNumber);
		add_single_attribute_operation(mods, &i, "facsimileTelephoneNumber", user.faxNumber);
		add_single_attribute_operation(mods, &i, "homePhone", user.homePhone);
		add_single_attribute_operation(mods, &i, "mobile", user.mobilePhone);
		add_single_attribute_operation(mods, &i, "pager", user.pagerNumber);
		add_single_attribute_operation(mods, &i, "websiteURL", user.website);
		add_single_attribute_operation(mods, &i, "postOfficeBox", user.poBox);
		add_single_attribute_operation(mods, &i, "street", user.street);
		add_single_attribute_operation(mods, &i, "postalAddress", user.address);
		add_single_attribute_operation(mods, &i, "st", user.state);
		add_single_attribute_operation(mods, &i, "postalCode", user.postcode);
		add_single_attribute_operation(mods, &i, "registeredAddress", user.registeredAddress);
		add_single_attribute_operation(mods, &i, "homePostalAddress", user.homeAddress);
		add_single_attribute_operation(mods, &i, "seeAlso", user.seeAlso);
		add_single_attribute_operation(mods, &i, "physicalDeliveryOfficeName", user.deliveryOffice);
		add_single_attribute_operation(mods, &i, "departmentNumber", user.department);
		add_single_attribute_operation(mods, &i, "roomNumber", user.roomNumber);
		add_single_attribute_operation(mods, &i, "employeeType", user.employeeType);
		add_single_attribute_operation(mods, &i, "employeeNumber", user.employeeNumber);
		add_single_attribute_operation(mods, &i, "managerName", user.manager);
		add_single_attribute_operation(mods, &i, "secretaryName", user.secretary);
		add_single_attribute_operation(mods, &i, "internationaliSDNNumber", user.isdnNumber);
		add_single_attribute_operation(mods, &i, "teletexId", user.teletexID);
		add_single_attribute_operation(mods, &i, "telexNumber", user.telexNumber);
		add_single_attribute_operation(mods, &i, "preferredDelivery", user.preferredDelivery);
		add_single_attribute_operation(mods, &i, "destinationIndicator", user.destinationIndicator);
		add_single_attribute_operation(mods, &i, "x121Address", user.x121Address);
		add_single_attribute_operation(mods, &i, "displayName", user.displayName);
		add_single_attribute_operation(mods, &i, "preferredLanguage", user.preferredLanguage);
		add_single_attribute_operation(mods, &i, "locallyUniqueID", user.uniqueIdentifier);
		add_single_attribute_operation(mods, &i, "businessCategory", user.businessCategory);
		add_single_attribute_operation(mods, &i, "carLicense", user.carLicense);
		add_single_attribute_operation(mods, &i, "notes", user.notes);
		LDAPMod *prevterm = mods[i];
		mods[i] = NULL;

		// Perform LDAP update
		retcode = ldap_modify_ext_s(m_ldap, user.distinguishedName.ascii(), mods, NULL, NULL);

		// Clean up
		clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

		if (retcode != LDAP_SUCCESS) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			return -2;
		}
		else {
			return 0;
		}
	}
}

// WARNING
// kadmin does not have a standard "waiting for user input" character or sequence
// To make matters worse, the colon does not uniquely designate the end of a line; for example the response "kadmin: ext openldap/foo.bar.baz: Principal does not exist"
// One way around this would be to see if the first colon is part of a "kadmin:" string; if so, then the colon is not a reliable end of line indicator for the current line
// (in fact only '\r' should be used as the end of line indicator in that case)
TQString LDAPManager::readFullLineFromPtyProcess(PtyProcess* proc) {
	TQString result = "";
	while ((!result.contains("\r")) &&
		(!result.contains(">")) &&
		(!((!result.contains("kadmin:")) && (!result.contains("kinit:")) && (!result.contains("ktutil:")) && result.contains(":"))) &&
		(!((result.contains("kadmin:")) && (!result.contains("kinit:")) && (!result.contains("ktutil:")) && result.contains("\r")))
		) {
		result = result + TQString(proc->readLine(false));
		tqApp->processEvents();
		if (!TQFile::exists(TQString("/proc/%1/exe").arg(proc->pid()))) {
			result.replace("\n", "");
			result.replace("\r", "");
			if (result == "") {
				result = "TDE process terminated";
			}
			break;
		}
	}
	result.replace("\n", "");
	result.replace("\r", "");
	result.replace("\x20\x08", "");		// Backspace + Space.  This one caused all kinds of fun with long distinguished names and/or passwords!
	return result;
}

int LDAPManager::setPasswordForUser(LDAPUserInfo user, TQString *errstr) {
	if (user.new_password == "") {
		return 0;
	}

	LDAPCredentials admincreds = currentLDAPCredentials();
	if ((admincreds.username == "") && (admincreds.password == "")) {
		// Probably GSSAPI
		// Get active ticket principal...
		KerberosTicketInfoList tickets = LDAPManager::getKerberosTicketList();
		TQStringList principalParts = TQStringList::split("@", tickets[0].cachePrincipal, false);
		admincreds.username = principalParts[0];
		admincreds.realm = principalParts[1];
	}

	TQCString command = "kadmin";
	QCStringList args;
	if (m_host.startsWith("ldapi://")) {
		args << TQCString("-l") << TQCString("-r") << TQCString(admincreds.realm.upper());
	}
	else {
		if (admincreds.username == "") {
			args << TQCString("-r") << TQCString(admincreds.realm.upper());
		}
		else {
			args << TQCString("-p") << TQCString(admincreds.username.lower()+"@"+(admincreds.realm.upper())) << TQCString("-r") << TQCString(admincreds.realm.upper());
		}
	}

	TQString prompt;
	PtyProcess kadminProc;
	kadminProc.exec(command, args);
	prompt = readFullLineFromPtyProcess(&kadminProc);
	prompt = prompt.stripWhiteSpace();
	if (prompt == "kadmin>") {
		command = TQCString("passwd "+user.name);
		kadminProc.enableLocalEcho(false);
		kadminProc.writeLine(command, true);
		do { // Discard our own input
			prompt = readFullLineFromPtyProcess(&kadminProc);
			printf("(kadmin) '%s'\n", prompt.ascii());
		} while ((prompt == TQString(command)) || (prompt == ""));
		prompt = prompt.stripWhiteSpace();
		if ((prompt.endsWith(" Password:")) && (prompt.startsWith(TQString(user.name + "@")))) {
			kadminProc.enableLocalEcho(false);
			kadminProc.writeLine(user.new_password, true);
			do { // Discard our own input
				prompt = readFullLineFromPtyProcess(&kadminProc);
				printf("(kadmin) '%s'\n", prompt.ascii());
			} while (prompt == "");
			prompt = prompt.stripWhiteSpace();
			if ((prompt.endsWith(" Password:")) && (prompt.startsWith("Verify"))) {
				kadminProc.enableLocalEcho(false);
				kadminProc.writeLine(user.new_password, true);
				do { // Discard our own input
					prompt = readFullLineFromPtyProcess(&kadminProc);
					printf("(kadmin) '%s'\n", prompt.ascii());
				} while (prompt == "");
				prompt = prompt.stripWhiteSpace();
			}
			if (prompt.endsWith(" Password:")) {
				if (admincreds.password == "") {
					if (tqApp->type() != TQApplication::Tty) {
						TQCString password;
						int result = KPasswordDialog::getPassword(password, prompt);
						if (result == KPasswordDialog::Accepted) {
							admincreds.password = password;
						}
					}
					else {
						TQFile file;
						file.open(IO_ReadOnly, stdin);
						TQTextStream qtin(&file);
						admincreds.password = qtin.readLine();
					}
				}
				if (admincreds.password != "") {
					kadminProc.enableLocalEcho(false);
					kadminProc.writeLine(admincreds.password, true);
					do { // Discard our own input
						prompt = readFullLineFromPtyProcess(&kadminProc);
						printf("(kadmin) '%s'\n", prompt.ascii());
					} while (prompt == "");
					prompt = prompt.stripWhiteSpace();
				}
			}
			if (prompt != "kadmin>") {
				if (errstr) *errstr = detailedKAdminErrorMessage(prompt);
				kadminProc.enableLocalEcho(false);
				kadminProc.writeLine("quit", true);
				return 1;
			}

			// Success!
			kadminProc.enableLocalEcho(false);
			kadminProc.writeLine("quit", true);
			return 0;
		}
		else if (prompt == "kadmin>") {
			// Success!
			kadminProc.enableLocalEcho(false);
			kadminProc.writeLine("quit", true);
			return 0;
		}

		// Failure
		if (errstr) *errstr = detailedKAdminErrorMessage(prompt);
		kadminProc.enableLocalEcho(false);
		kadminProc.writeLine("quit", true);
		return 1;
	}

	if (errstr) *errstr = "Internal error.  Verify that kadmin exists and can be executed.";
	return 1;	// Failure
}

TQString klistDateTimeToRFCDateTime(TQString datetime) {
	// HACK HACK HACK
	// FIXME
	TQString ret;
	TQString command = TQString("date -R -d \"%1\"").arg(datetime);
	FILE *output = popen(command.ascii(), "r");
	TQFile f;
	f.open(IO_ReadOnly, output);
	TQTextStream stream(&f);
	ret = stream.readLine();
	f.close();
	pclose(output);

	return ret;
}

#define KLIST_CREDENTIALS_CACHE_STRING "Credentials cache: "
#define KLIST_PRINCIPAL_STRING "Principal: "
#define KLIST_CACHE_VERSION_STRING "Cache version: "
#define KLIST_SERVER_STRING "Server: "
#define KLIST_CLIENT_STRING "Client: "
#define KLIST_ENCTYPE_STRING "Ticket etype: "
#define KLIST_TICKET_LENGTH_STRING "Ticket length: "
#define KLIST_AUTHTIME_STRING "Auth time: "
#define KLIST_STARTTIME_STRING "Start time: "
#define KLIST_STOPTIME_STRING "End time: "
#define KLIST_FLAGS_STRING "Ticket flags: "
#define KLIST_ADDRESSES_STRING "Ticket flags: "
#define KLIST_NO_TICKET_FILE "klist: No ticket file: "
#define KLIST_KVNO_STRING "kvno"

#define KLIST_ADDRESSLESS_STRING "addressless"

#define KLIST_KRB5_TICKET_RESERVED		"reserved"
#define KLIST_KRB5_TICKET_FORWARDABLE		"forwardable"
#define KLIST_KRB5_TICKET_FORWARDED		"forwarded"
#define KLIST_KRB5_TICKET_PROXIABLE		"proxiable"
#define KLIST_KRB5_TICKET_PROXY			"proxy"
#define KLIST_KRB5_TICKET_MAY_POSTDATE		"may-postdate"
#define KLIST_KRB5_TICKET_POSTDATED		"postdated"
#define KLIST_KRB5_TICKET_INVALID		"invalid"
#define KLIST_KRB5_TICKET_RENEWABLE		"renewable"
#define KLIST_KRB5_TICKET_INITIAL		"initial"
#define KLIST_KRB5_TICKET_PREAUTHENT		"pre-authent"
#define KLIST_KRB5_TICKET_HW_AUTHENT		"hw-authent"
#define KLIST_KRB5_TICKET_TRANSIT_CHECKED	"transited-policy-checked"
#define KLIST_KRB5_TICKET_OK_AS_DELEGATE	"ok-as-delegate"
#define KLIST_KRB5_TICKET_ANONYMOUS		"anonymous"
#define KLIST_KRB5_TICKET_ENC_PA_REP		"enc-pa-rep"

KerberosTicketInfoList LDAPManager::getKerberosTicketList(TQString cache, TQString *cacheFileName) {
	KerberosTicketInfo ticket;
	KerberosTicketInfoList list;

	TQString global_ccache;
	TQString global_principal;
	TQString global_cachevers;

	TQString line;
	FILE *output;
	if (cache != "") {
		output = popen((TQString("klist --cache=%1 -v 2>&1").arg(cache)).ascii(), "r");
	}
	else {
		output = popen("klist -v 2>&1", "r");
	}
	TQFile f;
	f.open(IO_ReadOnly, output);
	TQTextStream stream(&f);
	while ( !stream.atEnd() ) {
		line = stream.readLine();
		line = line.stripWhiteSpace();
		if (line == "") {
			if (ticket.informationValid) {
				ticket.cacheURL = global_ccache;
				ticket.cachePrincipal = global_principal;
				ticket.cacheVersion = global_cachevers.toInt();
				list.append(ticket);
			}
			ticket = KerberosTicketInfo();
		}
		else if (line.startsWith(KLIST_NO_TICKET_FILE)) {
			line.remove(0, strlen(KLIST_NO_TICKET_FILE));
			line.prepend("FILE:");
			if (cacheFileName) *cacheFileName = line;
		}
		else if (line.startsWith(KLIST_CREDENTIALS_CACHE_STRING)) {
			line.remove(0, strlen(KLIST_CREDENTIALS_CACHE_STRING));
			global_ccache = line;
			if (cacheFileName) *cacheFileName = line;
		}
		else if (line.startsWith(KLIST_PRINCIPAL_STRING)) {
			line.remove(0, strlen(KLIST_PRINCIPAL_STRING));
			global_principal = line;
		}
		else if (line.startsWith(KLIST_CACHE_VERSION_STRING)) {
			line.remove(0, strlen(KLIST_CACHE_VERSION_STRING));
			global_cachevers = line;
		}
		else if (line.startsWith(KLIST_SERVER_STRING)) {
			line.remove(0, strlen(KLIST_SERVER_STRING));
			ticket.serverPrincipal = line;
			ticket.informationValid = true;
		}
		else if (line.startsWith(KLIST_CLIENT_STRING)) {
			line.remove(0, strlen(KLIST_CLIENT_STRING));
			ticket.clientPrincipal = line;
			ticket.informationValid = true;
		}
		else if (line.startsWith(KLIST_ENCTYPE_STRING)) {
			line.remove(0, strlen(KLIST_ENCTYPE_STRING));
			TQString kvno = line;
			int commaloc = line.find(",");
			kvno.remove(0, commaloc+1);
			kvno.replace(KLIST_KVNO_STRING, "");
			kvno = kvno.stripWhiteSpace();
			line.truncate(commaloc);
			ticket.encryptionType = line;
			ticket.keyVersionNumber = kvno.toInt();
			ticket.informationValid = true;
		}
		else if (line.startsWith(KLIST_TICKET_LENGTH_STRING)) {
			line.remove(0, strlen(KLIST_TICKET_LENGTH_STRING));
			ticket.ticketSize = line.toInt();
			ticket.informationValid = true;
		}
		else if (line.startsWith(KLIST_AUTHTIME_STRING)) {
			line.remove(0, strlen(KLIST_AUTHTIME_STRING));
			line.replace("(expired)", "");
			line = line.simplifyWhiteSpace();
			line = klistDateTimeToRFCDateTime(line);
			ticket.authenticationTime.setTime_t(KRFCDate::parseDate(line));
			ticket.informationValid = true;
		}
		else if (line.startsWith(KLIST_STARTTIME_STRING)) {
			line.remove(0, strlen(KLIST_STARTTIME_STRING));
			line.replace("(expired)", "");
			line = line.simplifyWhiteSpace();
			line = klistDateTimeToRFCDateTime(line);
			ticket.validStartTime.setTime_t(KRFCDate::parseDate(line));
			ticket.informationValid = true;
		}
		else if (line.startsWith(KLIST_STOPTIME_STRING)) {
			line.remove(0, strlen(KLIST_STOPTIME_STRING));
			line.replace("(expired)", "");
			line = line.simplifyWhiteSpace();
			line = klistDateTimeToRFCDateTime(line);
			ticket.validEndTime.setTime_t(KRFCDate::parseDate(line));
			ticket.informationValid = true;
		}
		else if (line.startsWith(KLIST_FLAGS_STRING)) {
			line.remove(0, strlen(KLIST_FLAGS_STRING));
			TQStringList flags = TQStringList::split(",", line, FALSE);
			for (TQStringList::Iterator it = flags.begin(); it != flags.end(); ++it) {
				if ((*it) == KLIST_KRB5_TICKET_RESERVED) {
					ticket.flags = ticket.flags | KRB5_TICKET_RESERVED;
				}
				else if ((*it) == KLIST_KRB5_TICKET_FORWARDABLE) {
					ticket.flags = ticket.flags | KRB5_TICKET_FORWARDABLE;
				}
				else if ((*it) == KLIST_KRB5_TICKET_FORWARDED) {
					ticket.flags = ticket.flags | KRB5_TICKET_FORWARDED;
				}
				else if ((*it) == KLIST_KRB5_TICKET_PROXIABLE) {
					ticket.flags = ticket.flags | KRB5_TICKET_PROXIABLE;
				}
				else if ((*it) == KLIST_KRB5_TICKET_PROXY) {
					ticket.flags = ticket.flags | KRB5_TICKET_PROXY;
				}
				else if ((*it) == KLIST_KRB5_TICKET_MAY_POSTDATE) {
					ticket.flags = ticket.flags | KRB5_TICKET_MAY_POSTDATE;
				}
				else if ((*it) == KLIST_KRB5_TICKET_POSTDATED) {
					ticket.flags = ticket.flags | KRB5_TICKET_POSTDATED;
				}
				else if ((*it) == KLIST_KRB5_TICKET_INVALID) {
					ticket.flags = ticket.flags | KRB5_TICKET_INVALID;
				}
				else if ((*it) == KLIST_KRB5_TICKET_RENEWABLE) {
					ticket.flags = ticket.flags | KRB5_TICKET_RENEWABLE;
				}
				else if ((*it) == KLIST_KRB5_TICKET_INITIAL) {
					ticket.flags = ticket.flags | KRB5_TICKET_INITIAL;
				}
				else if ((*it) == KLIST_KRB5_TICKET_PREAUTHENT) {
					ticket.flags = ticket.flags | KRB5_TICKET_PREAUTHENT;
				}
				else if ((*it) == KLIST_KRB5_TICKET_HW_AUTHENT) {
					ticket.flags = ticket.flags | KRB5_TICKET_HW_AUTHENT;
				}
				else if ((*it) == KLIST_KRB5_TICKET_TRANSIT_CHECKED) {
					ticket.flags = ticket.flags | KRB5_TICKET_TRANSIT_CHECKED;
				}
				else if ((*it) == KLIST_KRB5_TICKET_OK_AS_DELEGATE) {
					ticket.flags = ticket.flags | KRB5_TICKET_OK_AS_DELEGATE;
				}
				else if ((*it) == KLIST_KRB5_TICKET_ANONYMOUS) {
					ticket.flags = ticket.flags | KRB5_TICKET_ANONYMOUS;
				}
				else if ((*it) == KLIST_KRB5_TICKET_ENC_PA_REP) {
					ticket.flags = ticket.flags | KRB5_TICKET_ENC_PA_REP;
				}
			}
			ticket.informationValid = true;
		}
		else if (line.startsWith(KLIST_ADDRESSES_STRING)) {
			line.remove(0, strlen(KLIST_ADDRESSES_STRING));
			if (line != KLIST_ADDRESSLESS_STRING) {
				// FIXME
				// What is the separator?
				ticket.addresses = TQStringList(line);
			}
			ticket.informationValid = true;
		}
	}
	f.close();
	pclose(output);

	return list;
}

int LDAPManager::getKerberosPassword(LDAPCredentials &creds, TQString prompt, bool requestServicePrincipal, TQWidget* parent)
{
	int i;

	KSimpleConfig* systemconfig = new KSimpleConfig( TQString::fromLatin1( KDE_CONFDIR "/ldap/ldapconfigrc" ));
	systemconfig->setGroup(NULL);
	TQString defaultRealm = systemconfig->readEntry("DefaultRealm", TQString::null);
	LDAPRealmConfigList realms = LDAPManager::readTDERealmList(systemconfig, false);
	delete systemconfig;

	if (creds.realm != "") {
		defaultRealm = creds.realm;
	}
	LDAPPasswordDialog passdlg(parent, 0, false);
	passdlg.m_base->ldapAdminRealm->setEnabled(true);
	LDAPRealmConfigList::Iterator it;
	i=0;
	for (it = realms.begin(); it != realms.end(); ++it) {
		passdlg.m_base->ldapAdminRealm->insertItem((*it).name);
		if ((*it).name == defaultRealm) {
			passdlg.m_base->ldapAdminRealm->setCurrentItem(i);
		}
		i++;
	}
	passdlg.m_base->passprompt->setText(prompt);
	passdlg.m_base->ldapUseTLS->hide();
	if (requestServicePrincipal) {
		passdlg.m_base->kerberosOtherInfoString->show();
		passdlg.m_base->kerberosServicePrincipal->show();
	}
	if (creds.username != "") {
		passdlg.m_base->ldapAdminUsername->setText(creds.username);
		passdlg.m_base->ldapAdminPassword->setFocus();
	}
	const int ret = passdlg.exec();
	if (ret == KDialog::Accepted) {
		creds.username = passdlg.m_base->ldapAdminUsername->text();
		creds.password = passdlg.m_base->ldapAdminPassword->password();
		creds.realm = passdlg.m_base->ldapAdminRealm->currentText();
		creds.service = passdlg.m_base->kerberosServicePrincipal->text();
		creds.use_tls = passdlg.m_base->ldapUseTLS->isOn();
	}
	return ret;
}

int LDAPManager::obtainKerberosTicket(LDAPCredentials creds, TQString principal, TQString *errstr) {
	TQCString command = "kinit";
	QCStringList args;
	if (principal == "") {
		args << TQCString(creds.username + "@" + creds.realm.upper());
	}
	else {
		args << TQCString("-S") << TQCString(principal) << TQCString(creds.username + "@" + creds.realm.upper());
	}

	TQString prompt;
	PtyProcess kadminProc;
	kadminProc.exec(command, args);
	prompt = readFullLineFromPtyProcess(&kadminProc);
	prompt = prompt.stripWhiteSpace();
	if (prompt.endsWith(" Password:")) {
		kadminProc.enableLocalEcho(false);
		kadminProc.writeLine(creds.password, true);
		do { // Discard our own input
			prompt = readFullLineFromPtyProcess(&kadminProc);
			printf("(kinit) '%s'\n", prompt.ascii());
		} while (prompt == "");
		prompt = prompt.stripWhiteSpace();
	}
	if ((prompt != "") && (prompt != "TDE process terminated")) {
		if (errstr) *errstr = detailedKAdminErrorMessage(prompt);
		return 1;
	}

	// Success!
	return 0;
}

int LDAPManager::obtainKerberosServiceTicket(TQString principal, TQString *errstr) {
	TQString ret;
	TQString command = TQString("kgetcred \"%1\"").arg(principal);
	FILE *output = popen(command.ascii(), "r");
	TQFile f;
	f.open(IO_ReadOnly, output);
	TQTextStream stream(&f);
	ret = stream.readLine();
	f.close();
	pclose(output);

	if (ret != "") {
		if (errstr) *errstr = detailedKAdminErrorMessage(ret);
		return -1;
	}
	return 0;
}

int LDAPManager::destroyKerberosTicket(TQString principal, TQString *errstr) {
	TQString ret;
	TQString command = TQString("kdestroy --credential=\"%1\"").arg(principal);
	FILE *output = popen(command.ascii(), "r");
	TQFile f;
	f.open(IO_ReadOnly, output);
	TQTextStream stream(&f);
	ret = stream.readLine();
	f.close();
	pclose(output);

	if (ret != "") {
		if (errstr) *errstr = detailedKAdminErrorMessage(ret);
		return -1;
	}
	return 0;
}

int LDAPManager::updateGroupInfo(LDAPGroupInfo group, TQString *errstr) {
	int retcode;
	int i;
	LDAPGroupInfo groupinfo;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Assemble the LDAPMod structure
		// We will replace any existing attributes with the new values
		int number_of_parameters = 3;				// 3 primary attributes
		LDAPMod *mods[number_of_parameters+1];
		set_up_attribute_operations(mods, number_of_parameters);

		// Load LDAP modification requests from provided data structure
		i=0;
		add_single_attribute_operation(mods, &i, "gidNumber", TQString("%1").arg(group.gid));
		TQStringList completeGroupList = group.userlist;
		TQString placeholderGroup = "cn=placeholder," + m_basedc;
		if (!completeGroupList.contains(placeholderGroup)) {
			completeGroupList.prepend(placeholderGroup);
		}
		add_multiple_attributes_operation(mods, &i, "member", completeGroupList);
		// Also populate memberUid attribute from the above list (minus the cn=,dc=... stuff, i.e. just the username)
		TQStringList posixGroupList;
		for ( TQStringList::Iterator it = group.userlist.begin(); it != group.userlist.end(); ++it ) {
			TQString plainUserName = *it;
			int eqpos = plainUserName.find("=")+1;
			int cmpos = plainUserName.find(",", eqpos);
			plainUserName.truncate(cmpos);
			plainUserName.remove(0, eqpos);
			posixGroupList.append(plainUserName);
		}
		add_multiple_attributes_operation(mods, &i, "memberUid", posixGroupList);

		LDAPMod *prevterm = mods[i];
		mods[i] = NULL;

		// Perform LDAP update
		retcode = ldap_modify_ext_s(m_ldap, group.distinguishedName.ascii(), mods, NULL, NULL);

		// Clean up
		clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

		if (retcode != LDAP_SUCCESS) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			return -2;
		}
		else {
			return 0;
		}
	}
}

// FIXME
int LDAPManager::updateMachineInfo(LDAPMachineInfo group, TQString *errstr) {
	if (errstr) *errstr = i18n("<qt>Not implemented yet!</qt>");
	return -1;
}

// FIXME
int LDAPManager::updateServiceInfo(LDAPServiceInfo group, TQString *errstr) {
	if (errstr) *errstr = i18n("<qt>Not implemented yet!</qt>");
	return -1;
}

int LDAPManager::addUserInfo(LDAPUserInfo user, TQString *errstr) {
	int retcode;
	int i;
	LDAPUserInfo userinfo;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Create the base DN entry
		int number_of_parameters = 14;				// 14 primary attributes
		LDAPMod *mods[number_of_parameters+1];
		set_up_attribute_operations(mods, number_of_parameters);

		// Load initial required LDAP object attributes
		i=0;
		create_single_attribute_operation(mods, &i, "uidNumber", TQString("%1").arg(user.uid));
		create_single_attribute_operation(mods, &i, "gidNumber", TQString("%1").arg(user.primary_gid));
		create_multiple_attributes_operation(mods, &i, "objectClass", TQStringList::split(" ", "inetOrgPerson krb5Realm krb5Principal krb5KDCEntry emsUser posixAccount"));
		create_single_attribute_operation(mods, &i, "uid", user.name);
		create_single_attribute_operation(mods, &i, "cn", user.commonName);
		create_single_attribute_operation(mods, &i, "sn", user.surName);
		create_single_attribute_operation(mods, &i, "homeDirectory", user.homedir);
		create_single_attribute_operation(mods, &i, "userPassword", "{SASL}" + user.name + "@" + m_realm.upper());
		// Kerberos
		create_single_attribute_operation(mods, &i, "krb5KeyVersionNumber", "1");
		create_single_attribute_operation(mods, &i, "krb5PrincipalName", TQString(user.name.lower()) + "@" + m_realm.upper());
		create_single_attribute_operation(mods, &i, "krb5RealmName", m_realm.upper());
		// Zivios specific
		create_single_attribute_operation(mods, &i, "emsdescription", "None");
		create_single_attribute_operation(mods, &i, "emsprimarygroupdn", "None");
		create_single_attribute_operation(mods, &i, "emstype", "UserEntry");
		LDAPMod *prevterm = mods[i];
		mods[i] = NULL;

		// Add new object
		retcode = ldap_add_ext_s(m_ldap, user.distinguishedName.ascii(), mods, NULL, NULL);

		// Clean up
		clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

		if (retcode != LDAP_SUCCESS) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP addition failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP addition failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			return -2;
		}
		else {
			return updateUserInfo(user);
		}
	}
}

int LDAPManager::addGroupInfo(LDAPGroupInfo group, TQString *errstr) {
	int retcode;
	int i;
	LDAPGroupInfo groupinfo;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Create the base DN entry
		int number_of_parameters = 6;				// 6 primary attributes
		LDAPMod *mods[number_of_parameters+1];
		set_up_attribute_operations(mods, number_of_parameters);

		TQString placeholderGroup = "cn=placeholder," + m_basedc;

		// Load initial required LDAP object attributes
		i=0;
		create_single_attribute_operation(mods, &i, "gidNumber", TQString("%1").arg(group.gid));
		create_multiple_attributes_operation(mods, &i, "objectClass", TQStringList::split(" ", "emsGroup groupOfNames posixGroup"));
		create_single_attribute_operation(mods, &i, "cn", group.name);
		create_multiple_attributes_operation(mods, &i, "member", TQStringList(placeholderGroup));
		// Zivios specific
		create_single_attribute_operation(mods, &i, "emsdescription", "None");
		create_single_attribute_operation(mods, &i, "emstype", "GroupEntry");
		LDAPMod *prevterm = mods[i];
		mods[i] = NULL;

		// Add new object
		retcode = ldap_add_ext_s(m_ldap, group.distinguishedName.ascii(), mods, NULL, NULL);

		// Clean up
		clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

		if (retcode != LDAP_SUCCESS) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP addition failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP addition failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			return -2;
		}
		else {
			return updateGroupInfo(group);
		}
	}
}

int LDAPManager::addMachineInfo(LDAPMachineInfo machine, TQString *errstr) {
	if (bind() < 0) {
		return -1;
	}
	else {
		// Use Kerberos kadmin to actually add the machine
		LDAPCredentials admincreds = currentLDAPCredentials();
		if ((admincreds.username == "") && (admincreds.password == "")) {
			// Probably GSSAPI
			// Get active ticket principal...
			KerberosTicketInfoList tickets = LDAPManager::getKerberosTicketList();
			TQStringList principalParts = TQStringList::split("@", tickets[0].cachePrincipal, false);
			admincreds.username = principalParts[0];
			admincreds.realm = principalParts[1];
		}
	
		TQCString command = "kadmin";
		QCStringList args;
		if (m_host.startsWith("ldapi://")) {
			args << TQCString("-l") << TQCString("-r") << TQCString(admincreds.realm.upper());
		}
		else {
			if (admincreds.username == "") {
				args << TQCString("-r") << TQCString(admincreds.realm.upper());
			}
			else {
				args << TQCString("-p") << TQCString(admincreds.username.lower()+"@"+(admincreds.realm.upper())) << TQCString("-r") << TQCString(admincreds.realm.upper());
			}
		}

		TQString hoststring = "host/"+machine.name+"."+admincreds.realm.lower();

		TQString prompt;
		PtyProcess kadminProc;
		kadminProc.exec(command, args);
		prompt = readFullLineFromPtyProcess(&kadminProc);
		prompt = prompt.stripWhiteSpace();
		if (prompt == "kadmin>") {
			if (machine.newPassword == "") {
				command = TQCString("ank --random-key "+hoststring);
			}
			else {
				command = TQCString("ank --password=\""+machine.newPassword+"\" "+hoststring);
			}
			kadminProc.enableLocalEcho(false);
			kadminProc.writeLine(command, true);
			do { // Discard our own input
				prompt = readFullLineFromPtyProcess(&kadminProc);
				printf("(kadmin) '%s'\n", prompt.ascii());
			} while ((prompt == TQString(command)) || (prompt == ""));
			prompt = prompt.stripWhiteSpace();
			// Use all defaults
			while (prompt != "kadmin>") {
				if (prompt.endsWith(" Password:")) {
					if (admincreds.password == "") {
						if (tqApp->type() != TQApplication::Tty) {
							TQCString password;
							int result = KPasswordDialog::getPassword(password, prompt);
							if (result == KPasswordDialog::Accepted) {
								admincreds.password = password;
							}
						}
						else {
							TQFile file;
							file.open(IO_ReadOnly, stdin);
							TQTextStream qtin(&file);
							admincreds.password = qtin.readLine();
						}
					}
					if (admincreds.password != "") {
						kadminProc.enableLocalEcho(false);
						kadminProc.writeLine(admincreds.password, true);
						do { // Discard our own input
							prompt = readFullLineFromPtyProcess(&kadminProc);
							printf("(kadmin) '%s'\n", prompt.ascii());
						} while (prompt == "");
						prompt = prompt.stripWhiteSpace();
					}
				}
				if (prompt.contains("authentication failed")) {
					if (errstr) *errstr = detailedKAdminErrorMessage(prompt);
					kadminProc.enableLocalEcho(false);
					kadminProc.writeLine("quit", true);
					return 1;
				}
				else {
					// Extract whatever default is in the [brackets] and feed it back to kadmin
					TQString defaultParam;
					int leftbracket = prompt.find("[");
					int rightbracket = prompt.find("]");
					if ((leftbracket >= 0) && (rightbracket >= 0)) {
						leftbracket++;
						defaultParam = prompt.mid(leftbracket, rightbracket-leftbracket);
					}
					command = TQCString(defaultParam);
					kadminProc.enableLocalEcho(false);
					kadminProc.writeLine(command, true);
					do { // Discard our own input
						prompt = readFullLineFromPtyProcess(&kadminProc);
						printf("(kadmin) '%s'\n", prompt.ascii());
					} while ((prompt == TQString(command)) || (prompt == ""));
					prompt = prompt.stripWhiteSpace();
				}
			}
			if (prompt != "kadmin>") {
				if (errstr) *errstr = detailedKAdminErrorMessage(prompt);
				kadminProc.enableLocalEcho(false);
				kadminProc.writeLine("quit", true);
				return 1;
			}

			// Success!
			kadminProc.enableLocalEcho(false);
			kadminProc.writeLine("quit", true);
			unbind(true);	// Using kadmin can disrupt our LDAP connection

			return 0;
		}

		if (errstr) *errstr = "Internal error.  Verify that kadmin exists and can be executed.";
		return 1;	// Failure

	}
}

int LDAPManager::addServiceInfo(LDAPServiceInfo service, TQString *errstr) {
	if (bind() < 0) {
		return -1;
	}
	else {
		// Use Kerberos kadmin to actually add the service
		LDAPCredentials admincreds = currentLDAPCredentials();
		if ((admincreds.username == "") && (admincreds.password == "")) {
			// Probably GSSAPI
			// Get active ticket principal...
			KerberosTicketInfoList tickets = LDAPManager::getKerberosTicketList();
			TQStringList principalParts = TQStringList::split("@", tickets[0].cachePrincipal, false);
			admincreds.username = principalParts[0];
			admincreds.realm = principalParts[1];
		}
	
		TQCString command = "kadmin";
		QCStringList args;
		if (m_host.startsWith("ldapi://")) {
			args << TQCString("-l") << TQCString("-r") << TQCString(admincreds.realm.upper());
		}
		else {
			if (admincreds.username == "") {
				args << TQCString("-r") << TQCString(admincreds.realm.upper());
			}
			else {
				args << TQCString("-p") << TQCString(admincreds.username.lower()+"@"+(admincreds.realm.upper())) << TQCString("-r") << TQCString(admincreds.realm.upper());
			}
		}

		TQString hoststring = service.name+"/"+service.machine;

		TQString prompt;
		PtyProcess kadminProc;
		kadminProc.exec(command, args);
		prompt = readFullLineFromPtyProcess(&kadminProc);
		prompt = prompt.stripWhiteSpace();
		if (prompt == "kadmin>") {
			command = TQCString("ank --random-key "+hoststring);
			kadminProc.enableLocalEcho(false);
			kadminProc.writeLine(command, true);
			do { // Discard our own input
				prompt = readFullLineFromPtyProcess(&kadminProc);
				printf("(kadmin) '%s'\n", prompt.ascii());
			} while ((prompt == TQString(command)) || (prompt == ""));
			prompt = prompt.stripWhiteSpace();
			// Use all defaults
			while (prompt != "kadmin>") {
				if (prompt.endsWith(" Password:")) {
					if (admincreds.password == "") {
						if (tqApp->type() != TQApplication::Tty) {
							TQCString password;
							int result = KPasswordDialog::getPassword(password, prompt);
							if (result == KPasswordDialog::Accepted) {
								admincreds.password = password;
							}
						}
						else {
							TQFile file;
							file.open(IO_ReadOnly, stdin);
							TQTextStream qtin(&file);
							admincreds.password = qtin.readLine();
						}
					}
					if (admincreds.password != "") {
						kadminProc.enableLocalEcho(false);
						kadminProc.writeLine(admincreds.password, true);
						do { // Discard our own input
							prompt = readFullLineFromPtyProcess(&kadminProc);
							printf("(kadmin) '%s'\n", prompt.ascii());
						} while (prompt == "");
						prompt = prompt.stripWhiteSpace();
					}
				}
				if (prompt.contains("authentication failed")) {
					if (errstr) *errstr = detailedKAdminErrorMessage(prompt);
					kadminProc.enableLocalEcho(false);
					kadminProc.writeLine("quit", true);
					return 1;
				}
				else {
					// Extract whatever default is in the [brackets] and feed it back to kadmin
					TQString defaultParam;
					int leftbracket = prompt.find("[");
					int rightbracket = prompt.find("]");
					if ((leftbracket >= 0) && (rightbracket >= 0)) {
						leftbracket++;
						defaultParam = prompt.mid(leftbracket, rightbracket-leftbracket);
					}
					command = TQCString(defaultParam);
					kadminProc.enableLocalEcho(false);
					kadminProc.writeLine(command, true);
					do { // Discard our own input
						prompt = readFullLineFromPtyProcess(&kadminProc);
						printf("(kadmin) '%s'\n", prompt.ascii());
					} while ((prompt == TQString(command)) || (prompt == ""));
					prompt = prompt.stripWhiteSpace();
				}
			}
			if (prompt != "kadmin>") {
				if (errstr) *errstr = detailedKAdminErrorMessage(prompt);
				kadminProc.enableLocalEcho(false);
				kadminProc.writeLine("quit", true);
				return 1;
			}

			// Success!
			kadminProc.enableLocalEcho(false);
			kadminProc.writeLine("quit", true);
			unbind(true);	// Using kadmin can disrupt our LDAP connection

			// Move Kerberos entries
			return moveKerberosEntries("o=kerberos,cn=kerberos control,ou=master services,ou=core,ou=realm," + m_basedc, errstr);
		}

		if (errstr) *errstr = "Internal error.  Verify that kadmin exists and can be executed.";
		return 1;	// Failure

	}
}

int LDAPManager::deleteUserInfo(LDAPUserInfo user, TQString *errstr) {
	int retcode;
	LDAPUserInfo userinfo;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Remove the user from all member groups
		LDAPGroupInfoList groupInfoList = groups(&retcode);
		LDAPGroupInfoList::Iterator it;
		for (it = groupInfoList.begin(); it != groupInfoList.end(); ++it) {
			LDAPGroupInfo group = *it;
			if (group.userlist.contains(user.distinguishedName)) {
				group.userlist.remove(user.distinguishedName);
				retcode = updateGroupInfo(group, errstr);
				if (retcode != 0) {
					return retcode;
				}
			}
		}

		// Delete the base DN entry
		retcode = ldap_delete_ext_s(m_ldap, user.distinguishedName.ascii(), NULL, NULL);
		if (retcode != LDAP_SUCCESS) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP deletion failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP deletion failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			return -2;
		}
		else {
			return 0;
		}
	}
}

int LDAPManager::deleteGroupInfo(LDAPGroupInfo group, TQString *errstr) {
	int retcode;
	LDAPGroupInfo groupinfo;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Delete the base DN entry
		retcode = ldap_delete_ext_s(m_ldap, group.distinguishedName.ascii(), NULL, NULL);
		if (retcode != LDAP_SUCCESS) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP deletion failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP deletion failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			return -2;
		}
		else {
			return 0;
		}
	}
}

int LDAPManager::deleteMachineInfo(LDAPMachineInfo machine, TQString *errstr) {
	int retcode;
	LDAPMachineInfo machineinfo;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Delete the base DN entry
		retcode = ldap_delete_ext_s(m_ldap, machine.distinguishedName.ascii(), NULL, NULL);
		if (retcode != LDAP_SUCCESS) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP deletion failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP deletion failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			return -2;
		}
		else {
			return 0;
		}
	}
}

int LDAPManager::deleteServiceInfo(LDAPServiceInfo service, TQString *errstr) {
	int retcode;
	LDAPServiceInfo serviceinfo;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Delete the base DN entry
		retcode = ldap_delete_ext_s(m_ldap, service.distinguishedName.ascii(), NULL, NULL);
		if (retcode != LDAP_SUCCESS) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP deletion failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP deletion failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			return -2;
		}
		else {
			return 0;
		}
	}
}

LDAPGroupInfo LDAPManager::parseLDAPGroupRecord(LDAPMessage* entry) {
	char* dn = NULL;
	char* attr;
	struct berval **vals;
	BerElement* ber;
	int i;

	LDAPGroupInfo groupinfo;

	if((dn = ldap_get_dn(m_ldap, entry)) != NULL) {
		groupinfo.distinguishedName = dn;
		TQStringList dnParts = TQStringList::split(",", dn);
		TQString id = dnParts[0];
		if (id.startsWith("cn=")) {
			id = id.remove(0, 3);
			groupinfo.name = id;
		}
		ldap_memfree(dn);
	}

	for( attr = ldap_first_attribute(m_ldap, entry, &ber); attr != NULL; attr = ldap_next_attribute(m_ldap, entry, ber)) {
		if ((vals = ldap_get_values_len(m_ldap, entry, attr)) != NULL)  {
			groupinfo.informationValid = true;
			TQString ldap_field = attr;
			i=0;
			if (ldap_field == "creatorsName") {
				groupinfo.creatorsName = vals[i]->bv_val;
			}
			else if (ldap_field == "member") {
				TQStringList members;
				for(i = 0; vals[i] != NULL; i++) {
					TQString userdn = vals[i]->bv_val;
					if (userdn.startsWith("cn=placeholder,dc=")) {
						continue;
					}
					members.append(userdn);
				}
				groupinfo.userlist = members;
			}
			else if (ldap_field == "gidNumber") {
				groupinfo.gid = atoi(vals[i]->bv_val);
			}
			else if (ldap_field == "tdeBuiltinAccount") {
				groupinfo.tde_builtin_account = (TQString(vals[i]->bv_val).upper() == "TRUE")?true:false;
			}
			ldap_value_free_len(vals);
		}
		ldap_memfree(attr);
	}

	if (ber != NULL) {
		ber_free(ber, 0);
	}

	return groupinfo;
}

LDAPMachineInfo LDAPManager::parseLDAPMachineRecord(LDAPMessage* entry) {
	char* dn = NULL;
	char* attr;
	struct berval **vals;
	BerElement* ber;
	int i;

	LDAPMachineInfo machineinfo;

	if((dn = ldap_get_dn(m_ldap, entry)) != NULL) {
		machineinfo.distinguishedName = dn;
		TQStringList dnParts = TQStringList::split(",", dn);
		TQString id = dnParts[0];
		if (id.startsWith("krb5PrincipalName=host/")) {
			id = id.remove(0, 23);
			id.replace("@"+m_realm, "");
			machineinfo.name = id;
		}
		ldap_memfree(dn);
	}

	for( attr = ldap_first_attribute(m_ldap, entry, &ber); attr != NULL; attr = ldap_next_attribute(m_ldap, entry, ber)) {
		if ((vals = ldap_get_values_len(m_ldap, entry, attr)) != NULL)  {
			machineinfo.informationValid = true;
			TQString ldap_field = attr;
			i=0;
			if (ldap_field == "creatorsName") {
				machineinfo.creatorsName = vals[i]->bv_val;
			}
			else if (ldap_field == "tdeBuiltinAccount") {
				machineinfo.tde_builtin_account = (TQString(vals[i]->bv_val).upper() == "TRUE")?true:false;
			}
			else if (ldap_field == "krb5KDCFlags") {
				machineinfo.status = (LDAPKRB5Flags)(atoi(vals[i]->bv_val));
			}
			ldap_value_free_len(vals);
		}
		ldap_memfree(attr);
	}

	if (ber != NULL) {
		ber_free(ber, 0);
	}

	return machineinfo;
}

LDAPServiceInfo LDAPManager::parseLDAPMachineServiceRecord(LDAPMessage* entry) {
	char* dn = NULL;
	char* attr;
	struct berval **vals;
	BerElement* ber;
	int i;

	LDAPServiceInfo machineserviceinfo;

	if((dn = ldap_get_dn(m_ldap, entry)) != NULL) {
		machineserviceinfo.distinguishedName = dn;
		TQStringList dnParts = TQStringList::split(",", dn);
		TQString id = dnParts[0];
		dnParts = TQStringList::split("/", id);
		id = dnParts[0];
		dnParts = TQStringList::split("=", id);
		machineserviceinfo.name = dnParts[1];
		ldap_memfree(dn);
	}

	for( attr = ldap_first_attribute(m_ldap, entry, &ber); attr != NULL; attr = ldap_next_attribute(m_ldap, entry, ber)) {
		if ((vals = ldap_get_values_len(m_ldap, entry, attr)) != NULL)  {
			machineserviceinfo.informationValid = true;
			TQString ldap_field = attr;
			i=0;
			if (ldap_field == "creatorsName") {
				machineserviceinfo.creatorsName = vals[i]->bv_val;
			}
			else if (ldap_field == "tdeBuiltinAccount") {
				machineserviceinfo.tde_builtin_account = (TQString(vals[i]->bv_val).upper() == "TRUE")?true:false;
			}
			else if (ldap_field == "krb5KDCFlags") {
				machineserviceinfo.status = (LDAPKRB5Flags)(atoi(vals[i]->bv_val));
			}
			ldap_value_free_len(vals);
		}
		ldap_memfree(attr);
	}

	if (ber != NULL) {
		ber_free(ber, 0);
	}

	return machineserviceinfo;
}

LDAPGroupInfoList LDAPManager::groups(int* mretcode, TQString *errstr) {
	int retcode;
	int errcode;
	LDAPGroupInfoList groups;

	if (bind() < 0) {
		if (mretcode) *mretcode = -1;
		return LDAPGroupInfoList();
	}
	else {
		LDAPMessage* msg;
		TQString ldap_base_dn = m_basedc;
		TQString ldap_filter = "(objectClass=posixGroup)";
		retcode = ldap_search_ext_s(m_ldap, ldap_base_dn.ascii(), LDAP_SCOPE_SUBTREE, ldap_filter.ascii(), ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if ((retcode != LDAP_SUCCESS) && (retcode != LDAP_SIZELIMIT_EXCEEDED)) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			if (mretcode) *mretcode = -1;
			return LDAPGroupInfoList();
		}
		else if (retcode == LDAP_SUCCESS) {
			// Iterate through the returned entries
			LDAPMessage* entry;
			for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
				groups.append(parseLDAPGroupRecord(entry));
			}

			// Clean up
			ldap_msgfree(msg);
	
			if (mretcode) *mretcode = 0;
			return groups;
		}
		else if (retcode == LDAP_SIZELIMIT_EXCEEDED) {
			// Try paged access
			bool morePages = false;
			unsigned long pageSize = 100;
			struct berval cookie = {0, NULL};
			char pagingCriticality = 'T';
			LDAPControl* pageControl = NULL;
			LDAPControl* serverControls[2] = { NULL, NULL };
			LDAPControl** returnedControls = NULL;
	
			do {
				retcode = ldap_create_page_control(m_ldap, pageSize, &cookie, pagingCriticality, &pageControl);
				if (retcode != LDAP_SUCCESS) {
					if (errstr) {
						*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					}
					else {
						KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					}
					if (mretcode) *mretcode = -1;
					return LDAPGroupInfoList();
				}
				serverControls[0] = pageControl;
				retcode = ldap_search_ext_s(m_ldap, ldap_base_dn.ascii(), LDAP_SCOPE_SUBTREE, ldap_filter.ascii(), ldap_user_and_operational_attributes, 0, serverControls, NULL, NULL, 0, &msg);
				if ((retcode != LDAP_SUCCESS) && (retcode != LDAP_PARTIAL_RESULTS)) {
					if (errstr) {
						*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					}
					else {
						KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					}
					if (mretcode) *mretcode = -1;
					return LDAPGroupInfoList();
				}
				retcode = ldap_parse_result(m_ldap, msg, &errcode, NULL, NULL, NULL, &returnedControls, false);
				if (retcode != LDAP_SUCCESS) {
					if (errstr) {
						*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					}
					else {
						KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					}
					if (mretcode) *mretcode = -1;
					return LDAPGroupInfoList();
				}
				if (cookie.bv_val != NULL) {
					ber_memfree(cookie.bv_val);
					cookie.bv_val = NULL;
					cookie.bv_len = 0;
				}
				if (!!returnedControls) {
					retcode = ldap_parse_pageresponse_control(m_ldap, returnedControls[0], NULL, &cookie);
					morePages = (cookie.bv_val && (strlen(cookie.bv_val) > 0));
				}
				else {
					morePages = false;
				}
	
				if (returnedControls != NULL) {
					ldap_controls_free(returnedControls);
					returnedControls = NULL;
				}
				serverControls[0] = NULL;
				ldap_control_free(pageControl);
				pageControl = NULL;
	
				// Iterate through the returned entries
				LDAPMessage* entry;
				for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
					groups.append(parseLDAPGroupRecord(entry));
				}

				// Clean up
				ldap_msgfree(msg);
			} while (morePages);
	
			if (mretcode) *mretcode = 0;
			return groups;
		}
	}

	return LDAPGroupInfoList();
}

LDAPMachineInfoList LDAPManager::machines(int* mretcode, TQString *errstr) {
	int retcode;
	int errcode;
	LDAPMachineInfoList machines;

	if (bind() < 0) {
		if (mretcode) *mretcode = -1;
		return LDAPMachineInfoList();
	}
	else {
		LDAPMessage* msg;
		TQString ldap_base_dn = m_basedc;
		TQString ldap_filter = "(&(objectClass=krb5Principal)(uid=host/*))";
		retcode = ldap_search_ext_s(m_ldap, ldap_base_dn.ascii(), LDAP_SCOPE_SUBTREE, ldap_filter.ascii(), ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if ((retcode != LDAP_SUCCESS) && (retcode != LDAP_SIZELIMIT_EXCEEDED)) {
			if (errstr) {
				*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			}
			else {
				KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			}
			if (mretcode) *mretcode = -1;
			return LDAPMachineInfoList();
		}
		else if (retcode == LDAP_SUCCESS) {
			// Iterate through the returned entries
			LDAPMessage* entry;
			for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
				machines.append(parseLDAPMachineRecord(entry));
			}

			// Clean up
			ldap_msgfree(msg);
	
			if (mretcode) *mretcode = 0;
			return machines;
		}
		else if (retcode == LDAP_SIZELIMIT_EXCEEDED) {
			// Try paged access
			bool morePages = false;
			unsigned long pageSize = 100;
			struct berval cookie = {0, NULL};
			char pagingCriticality = 'T';
			LDAPControl* pageControl = NULL;
			LDAPControl* serverControls[2] = { NULL, NULL };
			LDAPControl** returnedControls = NULL;
	
			do {
				retcode = ldap_create_page_control(m_ldap, pageSize, &cookie, pagingCriticality, &pageControl);
				if (retcode != LDAP_SUCCESS) {
					if (errstr) {
						*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					}
					else {
						KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					}
					if (mretcode) *mretcode = -1;
					return LDAPMachineInfoList();
				}
				serverControls[0] = pageControl;
				retcode = ldap_search_ext_s(m_ldap, ldap_base_dn.ascii(), LDAP_SCOPE_SUBTREE, ldap_filter.ascii(), ldap_user_and_operational_attributes, 0, serverControls, NULL, NULL, 0, &msg);
				if ((retcode != LDAP_SUCCESS) && (retcode != LDAP_PARTIAL_RESULTS)) {
					if (errstr) {
						*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					}
					else {
						KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					}
					if (mretcode) *mretcode = -1;
					return LDAPMachineInfoList();
				}
				retcode = ldap_parse_result(m_ldap, msg, &errcode, NULL, NULL, NULL, &returnedControls, false);
				if (retcode != LDAP_SUCCESS) {
					if (errstr) {
						*errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					}
					else {
						KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					}
					if (mretcode) *mretcode = -1;
					return LDAPMachineInfoList();
				}
				if (cookie.bv_val != NULL) {
					ber_memfree(cookie.bv_val);
					cookie.bv_val = NULL;
					cookie.bv_len = 0;
				}
				if (!!returnedControls) {
					retcode = ldap_parse_pageresponse_control(m_ldap, returnedControls[0], NULL, &cookie);
					morePages = (cookie.bv_val && (strlen(cookie.bv_val) > 0));
				}
				else {
					morePages = false;
				}
	
				if (returnedControls != NULL) {
					ldap_controls_free(returnedControls);
					returnedControls = NULL;
				}
				serverControls[0] = NULL;
				ldap_control_free(pageControl);
				pageControl = NULL;
	
				// Iterate through the returned entries
				LDAPMessage* entry;
				for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
					machines.append(parseLDAPMachineRecord(entry));
				}

				// Clean up
				ldap_msgfree(msg);
			} while (morePages);
	
			if (mretcode) *mretcode = 0;
			return machines;
		}
	}

	return LDAPMachineInfoList();
}

LDAPServiceInfoList LDAPManager::services(int* mretcode, TQString *errstr) {
	LDAPServiceInfoList services;

	if (bind() < 0) {
		if (mretcode) *mretcode = -1;
		return LDAPServiceInfoList();
	}
	else {
		int machineSearchRet;
		LDAPMachineInfoList machineList = machines(&machineSearchRet, errstr);
		if (machineSearchRet != 0) {
			if (mretcode) *mretcode = -1;
			return LDAPServiceInfoList();
		}

		LDAPMachineInfoList::Iterator it;
		for (it = machineList.begin(); it != machineList.end(); ++it) {
			LDAPMachineInfo machine = *it;
			LDAPServiceInfoList thisMachineServiceList = machineServices(machine.distinguishedName);
			LDAPServiceInfoList::Iterator it2;
			for (it2 = thisMachineServiceList.begin(); it2 != thisMachineServiceList.end(); ++it2) {
				services.append(*it2);
			}
		}

		if (mretcode) *mretcode = 0;
		return services;
	}

	return LDAPServiceInfoList();
}

LDAPServiceInfoList LDAPManager::machineServices(TQString machine_dn, int* mretcode) {
	int retcode;
	LDAPServiceInfoList services;

	if (bind() < 0) {
		if (mretcode) *mretcode = -1;
		return LDAPServiceInfoList();
	}
	else {
		LDAPMessage* msg;
		TQString ldap_base_dn = m_basedc;

		TQStringList machinednParts = TQStringList::split(",", machine_dn);
		TQString machine_name = machinednParts[0];
		if (machine_name.startsWith("krb5PrincipalName=host/")) {
			machine_name = machine_name.remove(0, 23);
			machine_name.replace("@"+m_realm, "");
		}

		TQString ldap_filter = TQString("(&(objectClass=krb5Principal)(uid=*/%1))").arg(machine_name);
		retcode = ldap_search_ext_s(m_ldap, ldap_base_dn.ascii(), LDAP_SCOPE_SUBTREE, ldap_filter.ascii(), ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if (retcode != LDAP_SUCCESS) {
			KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			if (mretcode) *mretcode = -1;
			return LDAPServiceInfoList();
		}

		// Iterate through the returned entries
		LDAPMessage* entry;
		for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
			LDAPServiceInfo sinfo = parseLDAPMachineServiceRecord(entry);
			sinfo.machine_dn = machine_dn;
			sinfo.machine = machine_name;
			if (sinfo.name != "host") {
				services.append(sinfo);
			}
		}

		// Clean up
		ldap_msgfree(msg);

		if (mretcode) *mretcode = 0;
		return services;
	}

	return LDAPServiceInfoList();
}

int LDAPManager::exportKeytabForPrincipal(TQString principal, TQString fileName, TQString *errstr) {
	if (bind() < 0) {
		return -1;
	}
	else {
		// Use Kerberos kadmin to export the keytab
		LDAPCredentials admincreds = currentLDAPCredentials();
		if ((admincreds.username == "") && (admincreds.password == "")) {
			// Probably GSSAPI
			// Get active ticket principal...
			KerberosTicketInfoList tickets = LDAPManager::getKerberosTicketList();
			TQStringList principalParts = TQStringList::split("@", tickets[0].cachePrincipal, false);
			admincreds.username = principalParts[0];
			admincreds.realm = principalParts[1];
		}
	
		TQCString command = "kadmin";
		QCStringList args;
		if (m_host.startsWith("ldapi://")) {
			args << TQCString("-l") << TQCString("-r") << TQCString(admincreds.realm.upper());
		}
		else {
			if (admincreds.username == "") {
				args << TQCString("-r") << TQCString(admincreds.realm.upper());
			}
			else {
				args << TQCString("-p") << TQCString(admincreds.username.lower()+"@"+(admincreds.realm.upper())) << TQCString("-r") << TQCString(admincreds.realm.upper());
			}
		}

		TQString prompt;
		PtyProcess kadminProc;
		kadminProc.exec(command, args);
		prompt = readFullLineFromPtyProcess(&kadminProc);
		prompt = prompt.stripWhiteSpace();
		if (prompt == "kadmin>") {
			if (fileName == "") {
				command = TQCString("ext_keytab "+principal);
			}
			else {
				command = TQCString("ext_keytab --keytab=\""+fileName+"\" "+principal);
			}
			kadminProc.enableLocalEcho(false);
			kadminProc.writeLine(command, true);
			do { // Discard our own input
				prompt = readFullLineFromPtyProcess(&kadminProc);
				printf("(kadmin) '%s'\n", prompt.ascii());
			} while ((prompt == TQString(command)) || (prompt == ""));
			prompt = prompt.stripWhiteSpace();
			// Use all defaults
			while (prompt != "kadmin>") {
				if (prompt.endsWith(" Password:")) {
					if (admincreds.password == "") {
						if (tqApp->type() != TQApplication::Tty) {
							TQCString password;
							int result = KPasswordDialog::getPassword(password, prompt);
							if (result == KPasswordDialog::Accepted) {
								admincreds.password = password;
							}
						}
						else {
							TQFile file;
							file.open(IO_ReadOnly, stdin);
							TQTextStream qtin(&file);
							admincreds.password = qtin.readLine();
						}
					}
					if (admincreds.password != "") {
						kadminProc.enableLocalEcho(false);
						kadminProc.writeLine(admincreds.password, true);
						do { // Discard our own input
							prompt = readFullLineFromPtyProcess(&kadminProc);
							printf("(kadmin) '%s'\n", prompt.ascii());
						} while (prompt == "");
						prompt = prompt.stripWhiteSpace();
					}
				}
				if (prompt.contains("authentication failed")) {
					if (errstr) *errstr = detailedKAdminErrorMessage(prompt);
					kadminProc.enableLocalEcho(false);
					kadminProc.writeLine("quit", true);
					return 1;
				}
				else {
					// Extract whatever default is in the [brackets] and feed it back to kadmin
					TQString defaultParam;
					int leftbracket = prompt.find("[");
					int rightbracket = prompt.find("]");
					if ((leftbracket >= 0) && (rightbracket >= 0)) {
						leftbracket++;
						defaultParam = prompt.mid(leftbracket, rightbracket-leftbracket);
					}
					command = TQCString(defaultParam);
					kadminProc.enableLocalEcho(false);
					kadminProc.writeLine(command, true);
					do { // Discard our own input
						prompt = readFullLineFromPtyProcess(&kadminProc);
						printf("(kadmin) '%s'\n", prompt.ascii());
					} while ((prompt == TQString(command)) || (prompt == ""));
					prompt = prompt.stripWhiteSpace();
				}
			}
			if (prompt != "kadmin>") {
				if (errstr) *errstr = detailedKAdminErrorMessage(prompt);
				kadminProc.enableLocalEcho(false);
				kadminProc.writeLine("quit", true);
				return 1;
			}

			// Success!
			kadminProc.enableLocalEcho(false);
			kadminProc.writeLine("quit", true);
			unbind(true);	// Using kadmin can disrupt our LDAP connection

			return 0;
		}

		if (errstr) *errstr = "Internal error.  Verify that kadmin exists and can be executed.";
		return 1;	// Failure

	}
}

int LDAPManager::writeCertificateFileIntoDirectory(TQByteArray cert, TQString attr, TQString* errstr) {
	int retcode;
	int i;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Assemble the LDAPMod structure
		// We will replace any existing attributes with the new values
		int number_of_parameters = 1;				// 1 primary attribute
		LDAPMod *mods[number_of_parameters+1];
		set_up_attribute_operations(mods, number_of_parameters);

		// Load LDAP modification requests from provided data structure
		i=0;
		add_single_binary_attribute_operation(mods, &i, attr, cert);
		LDAPMod *prevterm = mods[i];
		mods[i] = NULL;

		// Perform LDAP update
		retcode = ldap_modify_ext_s(m_ldap, TQString("cn=certificate store,o=tde,cn=tde realm data,ou=master services,ou=core,ou=realm,%1").arg(m_basedc).ascii(), mods, NULL, NULL);

		// Clean up
		clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

		if (retcode != LDAP_SUCCESS) {
			if (errstr) *errstr = i18n("<qt>LDAP certificate upload failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			else KMessageBox::error(0, i18n("<qt>LDAP certificate upload failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return -2;
		}
		else {
			return 0;
		}
	}
}

TQString LDAPManager::getRealmCAMaster(TQString* errstr) {
	int retcode;
	int i;
	TQString realmCAMaster;

	TQString dn = TQString("cn=certificate store,o=tde,cn=tde realm data,ou=master services,ou=core,ou=realm,%1").arg(m_basedc);

	if (bind(errstr) < 0) {
		return TQString();
	}
	else {
		LDAPMessage* msg;
		retcode = ldap_search_ext_s(m_ldap, dn.ascii(), LDAP_SCOPE_SUBTREE, NULL, ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if (retcode != LDAP_SUCCESS) {
			if (errstr) *errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			else KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return TQString();
		}

		// Iterate through the returned entries
		LDAPMessage* entry;
		for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
			TQString result;
			if (parseLDAPTDEStringAttribute(entry, "publicRootCertificateOriginServer", result)) {
				realmCAMaster = result;
			}
		}

		// Clean up
		ldap_msgfree(msg);

		return realmCAMaster;
	}
}

int LDAPManager::setRealmCAMaster(TQString masterFQDN, TQString* errstr) {
	int retcode;
	int i;

	if (bind() < 0) {
		return -1;
	}
	else {
		// Assemble the LDAPMod structure
		// We will replace any existing attributes with the new values
		int number_of_parameters = 1;				// 1 primary attribute
		LDAPMod *mods[number_of_parameters+1];
		set_up_attribute_operations(mods, number_of_parameters);

		// Load LDAP modification requests from provided data structure
		i=0;
		add_single_attribute_operation(mods, &i, "publicRootCertificateOriginServer", masterFQDN);
		LDAPMod *prevterm = mods[i];
		mods[i] = NULL;

		// Perform LDAP update
		retcode = ldap_modify_ext_s(m_ldap, TQString("cn=certificate store,o=tde,cn=tde realm data,ou=master services,ou=core,ou=realm,%1").arg(m_basedc).ascii(), mods, NULL, NULL);

		// Clean up
		clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

		if (retcode != LDAP_SUCCESS) {
			if (errstr) *errstr = i18n("<qt>LDAP CA master modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			else KMessageBox::error(0, i18n("<qt>LDAP CA master modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return -2;
		}
		else {
			return 0;
		}
	}
}

// Special method, used when creating a new Kerberos realm
int LDAPManager::moveKerberosEntries(TQString newSuffix, TQString* errstr) {
	int retcode;

	if (bind(errstr) < 0) {
		return -1;
	}
	else {
		LDAPMessage* msg;
		TQString ldap_base_dn = m_basedc;
		TQString ldap_filter = "(&(objectClass=krb5Principal)(!(objectClass=posixAccount)))";
		retcode = ldap_search_ext_s(m_ldap, ldap_base_dn.ascii(), LDAP_SCOPE_SUBTREE, ldap_filter.ascii(), ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if (retcode != LDAP_SUCCESS) {
			KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return -1;
		}

		// Iterate through the returned entries
		LDAPMessage* entry;
		for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
			char* dn = NULL;

			LDAPMachineInfo machineinfo;

			if((dn = ldap_get_dn(m_ldap, entry)) != NULL) {
				TQStringList dnParts = TQStringList::split(",", dn);
				TQString id = dnParts[0];
				retcode = ldap_rename_s(m_ldap, dn, id, newSuffix, 0, NULL, NULL);
				if (retcode != LDAP_SUCCESS) {
					if (errstr) *errstr = i18n("LDAP rename failure<p>Reason: [%3] %4").arg(retcode).arg(ldap_err2string(retcode));
					return -1;
				}
			}
		}

		// Clean up
		ldap_msgfree(msg);

		return 0;
	}

	return -1;
}

int LDAPManager::writeLDAPConfFile(LDAPRealmConfig realmcfg, LDAPMachineRole machineRole, TQString *errstr) {
	KSimpleConfig* systemconfig;
	TQString m_defaultRealm;
	int m_ldapVersion;
	int m_ldapTimeout;
	TQString m_bindPolicy;
	int m_ldapBindTimeout;
	TQString m_passwordHash;
	TQString m_ignoredUsers;
	TQString command;

	systemconfig = new KSimpleConfig( TQString::fromLatin1( KDE_CONFDIR "/ldap/ldapconfigrc" ));
	systemconfig->setGroup(NULL);
	m_defaultRealm = systemconfig->readEntry("DefaultRealm", TQString::null);

	m_ldapVersion = systemconfig->readNumEntry("ConnectionLDAPVersion", 3);
	m_ldapTimeout = systemconfig->readNumEntry("ConnectionLDAPTimeout", 2);
	m_bindPolicy = systemconfig->readEntry("ConnectionBindPolicy", "soft");
	m_ldapBindTimeout = systemconfig->readNumEntry("ConnectionBindTimeout", 2);
	m_passwordHash = systemconfig->readEntry("ConnectionPasswordHash", "exop");
	m_ignoredUsers = systemconfig->readEntry("ConnectionIgnoredUsers", DEFAULT_IGNORED_USERS_LIST);

	TQFile file(LDAP_FILE);
	if (file.open(IO_WriteOnly)) {
		TQTextStream stream( &file );

		stream << "# This file was automatically generated by TDE\n";
		stream << "# All changes will be lost!\n";
		stream << "\n";

		if (realmcfg.bonded) {
			stream << "host " << realmcfg.admin_server << "\n";
			TQStringList domainChunks = TQStringList::split(".", realmcfg.name.lower());
			stream << "base dc=" << domainChunks.join(",dc=") << "\n";
			stream << "ldap_version " << m_ldapVersion << "\n";
			stream << "timelimit " << m_ldapTimeout << "\n";
			stream << "bind_timelimit " << m_ldapBindTimeout << "\n";
			stream << "bind_policy " << m_bindPolicy.lower() << "\n";
			stream << "pam_password " << m_passwordHash.lower() << "\n";
			stream << "nss_initgroups_ignoreusers " << m_ignoredUsers << "\n";
			if (machineRole == ROLE_WORKSTATION) {
				stream << "tls_cacert " << KERBEROS_PKI_PUBLICDIR << realmcfg.admin_server << ".ldap.crt\n";
			}
			else {
				stream << "tls_cacert " << KERBEROS_PKI_PEM_FILE << "\n";
			}
		}

		file.close();
	}

	if (chmod(LDAP_FILE, S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH) < 0) {
		if (errstr) *errstr = TQString("Unable to change permissions of \"%1\"").arg(LDAP_FILE);
		return -1;
	}

	// Create symbolic link to secondary LDAP configuration file
	if (fileExists(LDAP_SECONDARY_FILE)) {
		if (unlink(LDAP_SECONDARY_FILE) < 0) {
			if (errstr) *errstr = TQString("Unable to unlink \"%s\"").arg(LDAP_SECONDARY_FILE);
			return -1;
		}
	}
	command = TQString("ln -s %1 %2").arg(LDAP_FILE).arg(LDAP_SECONDARY_FILE);
	if (system(command) < 0) {
		if (errstr) *errstr = TQString("Execution of \"%s\" failed").arg(command.ascii());
		return -1;
	}

	// Create symbolic link to tertiary LDAP configuration file
	if (fileExists(LDAP_TERTIARY_FILE)) {
		if (unlink(LDAP_TERTIARY_FILE) < 0) {
			if (errstr) *errstr = TQString("Unable to unlink \"%s\"").arg(LDAP_TERTIARY_FILE);
			return -1;
		}
	}
	command = TQString("ln -s %1 %2").arg(LDAP_FILE).arg(LDAP_TERTIARY_FILE);
	if (system(command) < 0) {
		if (errstr) *errstr = TQString("Execution of \"%s\" failed").arg(command.ascii());
		return -1;
	}

	delete systemconfig;

	return 0;
}

LDAPTDEBuiltinsInfo LDAPManager::parseLDAPTDEBuiltinsRecord(LDAPMessage* entry) {
	char* dn = NULL;
	char* attr;
	struct berval **vals;
	BerElement* ber;
	int i;

	LDAPTDEBuiltinsInfo builtininfo;

	if((dn = ldap_get_dn(m_ldap, entry)) != NULL) {
		ldap_memfree(dn);
	}

	for( attr = ldap_first_attribute(m_ldap, entry, &ber); attr != NULL; attr = ldap_next_attribute(m_ldap, entry, ber)) {
		if ((vals = ldap_get_values_len(m_ldap, entry, attr)) != NULL)  {
			builtininfo.informationValid = true;
			TQString ldap_field = attr;
			i=0;
			if (ldap_field == "builtinRealmAdminAccount") {
				builtininfo.builtinRealmAdminAccount = vals[i]->bv_val;
			}
			else if (ldap_field == "builtinRealmAdminGroup") {
				builtininfo.builtinRealmAdminGroup = vals[i]->bv_val;
			}
			else if (ldap_field == "builtinMachineAdminGroup") {
				builtininfo.builtinMachineAdminGroup = vals[i]->bv_val;
			}
			else if (ldap_field == "builtinStandardUserGroup") {
				builtininfo.builtinStandardUserGroup = vals[i]->bv_val;
			}
			ldap_value_free_len(vals);
		}
		ldap_memfree(attr);
	}

	if (ber != NULL) {
		ber_free(ber, 0);
	}

	return builtininfo;
}

bool LDAPManager::parseLDAPTDEStringAttribute(LDAPMessage* entry, TQString attribute, TQString& retval) {
	char* dn = NULL;
	char* attr;
	struct berval **vals;
	BerElement* ber;
	int i;
	bool found = false;

	LDAPTDEBuiltinsInfo builtininfo;

	if((dn = ldap_get_dn(m_ldap, entry)) != NULL) {
		ldap_memfree(dn);
	}

	for( attr = ldap_first_attribute(m_ldap, entry, &ber); attr != NULL; attr = ldap_next_attribute(m_ldap, entry, ber)) {
		if ((vals = ldap_get_values_len(m_ldap, entry, attr)) != NULL)  {
			builtininfo.informationValid = true;
			TQString ldap_field = attr;
			i=0;
			if (ldap_field == attribute) {
				retval = TQString(vals[i]->bv_val);
				found = true;
			}
			ldap_value_free_len(vals);
		}
		ldap_memfree(attr);
	}

	if (ber != NULL) {
		ber_free(ber, 0);
	}

	return found;
}

LDAPMasterReplicationInfo LDAPManager::parseLDAPMasterReplicationRecord(LDAPMasterReplicationInfo replicationinfo, LDAPMessage* entry) {
	char* dn = NULL;
	char* attr;
	struct berval **vals;
	BerElement* ber;
	int i;

	if((dn = ldap_get_dn(m_ldap, entry)) != NULL) {
		ldap_memfree(dn);
	}

	for( attr = ldap_first_attribute(m_ldap, entry, &ber); attr != NULL; attr = ldap_next_attribute(m_ldap, entry, ber)) {
		if ((vals = ldap_get_values_len(m_ldap, entry, attr)) != NULL)  {
			TQString ldap_field = attr;
			if (ldap_field == "olcServerID") {
				i=0;
				while (vals[i] != NULL) {
					TQStringList serverIDMapping = TQStringList::split(" ", TQString(vals[i]->bv_val), FALSE);
					LDAPMasterReplicationMapping mapping;
					mapping.id = serverIDMapping[0].toInt();
					mapping.fqdn = serverIDMapping[1];
					mapping.fqdn.replace("ldap:", "");
					mapping.fqdn.replace("ldaps:", "");
					mapping.fqdn.replace("/", "");
					replicationinfo.serverIDs.append(mapping);
					i++;
				}
				replicationinfo.informationValid = true;
			}
			else if (ldap_field == "olcMirrorMode") {
				i=0;
				TQString mirrorModeEnabled(vals[i]->bv_val);
				if (mirrorModeEnabled == "TRUE") {
					replicationinfo.enabled = true;
				}
				else {
					replicationinfo.enabled = false;
				}
			}
			ldap_value_free_len(vals);
		}
		ldap_memfree(attr);
	}

	if (ber != NULL) {
		ber_free(ber, 0);
	}

	return replicationinfo;
}

TQString LDAPManager::parseLDAPSyncProvOverlayConfigRecord(LDAPMessage* entry) {
	char* dn = NULL;
	char* attr;
	struct berval **vals;
	BerElement* ber;
	int i;
	TQString syncProvEntry;

	if((dn = ldap_get_dn(m_ldap, entry)) != NULL) {
		ldap_memfree(dn);
	}

	for( attr = ldap_first_attribute(m_ldap, entry, &ber); attr != NULL; attr = ldap_next_attribute(m_ldap, entry, ber)) {
		if ((vals = ldap_get_values_len(m_ldap, entry, attr)) != NULL)  {
			TQString ldap_field = attr;
			if (ldap_field == "olcOverlay") {
				i=0;
				while (vals[i] != NULL) {
					TQString value(vals[i]->bv_val);
					if (value.endsWith("}syncprov")) {
						syncProvEntry = value;
					}
					i++;
				}
			}
			ldap_value_free_len(vals);
		}
		ldap_memfree(attr);
	}

	if (ber != NULL) {
		ber_free(ber, 0);
	}

	return syncProvEntry;
}

LDAPTDEBuiltinsInfo LDAPManager::getTDEBuiltinMappings(TQString *errstr) {
	int retcode;
	LDAPTDEBuiltinsInfo builtininfo;

	TQString dn = TQString("cn=builtin mappings,o=tde,cn=tde realm data,ou=master services,ou=core,ou=realm,%1").arg(m_basedc);

	if (bind(errstr) < 0) {
		return LDAPTDEBuiltinsInfo();
	}
	else {
		LDAPMessage* msg;
		retcode = ldap_search_ext_s(m_ldap, dn.ascii(), LDAP_SCOPE_SUBTREE, NULL, ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if (retcode != LDAP_SUCCESS) {
			if (errstr) *errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			else KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return LDAPTDEBuiltinsInfo();
		}

		// Iterate through the returned entries
		LDAPMessage* entry;
		for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
			builtininfo = parseLDAPTDEBuiltinsRecord(entry);
		}

		// Clean up
		ldap_msgfree(msg);

		return builtininfo;
	}

	return LDAPTDEBuiltinsInfo();
}

LDAPMasterReplicationInfo LDAPManager::getLDAPMasterReplicationSettings(TQString *errstr) {
	int retcode;
	LDAPMasterReplicationInfo replicationinfo;

	if (bind(errstr) < 0) {
		return LDAPMasterReplicationInfo();
	}
	else {
		// Check cn=config settings
		LDAPMessage* msg;
		retcode = ldap_search_ext_s(m_ldap, "cn=config", LDAP_SCOPE_SUBTREE, NULL, ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if (retcode != LDAP_SUCCESS) {
			if (errstr) *errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			else KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return LDAPMasterReplicationInfo();
		}

		// Iterate through the returned entries
		LDAPMessage* entry;
		for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
			LDAPMasterReplicationInfo potentialReplicationInfo;
			potentialReplicationInfo = parseLDAPMasterReplicationRecord(LDAPMasterReplicationInfo(), entry);
			if (potentialReplicationInfo.informationValid) {
				replicationinfo = potentialReplicationInfo;
			}
		}

		// Clean up
		ldap_msgfree(msg);

		// Set OpenLDAP defaults
		replicationinfo.enabled = false;

		// Check olcDatabase 0 configuration settings
		retcode = ldap_search_ext_s(m_ldap, "olcDatabase={0}config,cn=config", LDAP_SCOPE_SUBTREE, NULL, ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if (retcode != LDAP_SUCCESS) {
			if (errstr) *errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			else KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return LDAPMasterReplicationInfo();
		}

		// Iterate through the returned entries
		for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
			replicationinfo = parseLDAPMasterReplicationRecord(replicationinfo, entry);
		}

		// Clean up
		ldap_msgfree(msg);

		return replicationinfo;
	}

	return LDAPMasterReplicationInfo();
}

int LDAPManager::setLDAPMasterReplicationSettings(LDAPMasterReplicationInfo replicationinfo, TQString *errstr) {
	int retcode;
	int i;

	if (bind(errstr) < 0) {
		return -1;
	}
	else {
		if (replicationinfo.enabled) {
			if (!errstr && (replicationinfo.syncDN == "")) {
				replicationinfo.syncDN = "cn=admin," + m_basedc;
			}
			if (!errstr && replicationinfo.syncPassword.isNull()) {
				LDAPPasswordDialog passdlg(0, 0, false);
				passdlg.m_base->ldapAdminRealm->setEnabled(false);
				passdlg.m_base->ldapAdminRealm->insertItem(m_realm);
				passdlg.m_base->ldapUseTLS->hide();
				passdlg.m_base->ldapAdminUsername->setEnabled(false);
				passdlg.m_base->ldapAdminUsername->setText(replicationinfo.syncDN);
				if (passdlg.exec() == TQDialog::Accepted) {
					replicationinfo.syncPassword = passdlg.m_base->ldapAdminPassword->password();
				}
			}

			if ((replicationinfo.syncDN == "") || (replicationinfo.syncPassword.isNull())) {
				if (errstr) *errstr = i18n("<qt>LDAP modification failure<p>Reason: Invalid multi-master synchronization credentials provided</qt>");
				else KMessageBox::error(0, i18n("<qt>LDAP modification failure<p>Reason: Invalid multi-master synchronization credentials provided</qt>"), i18n("User Error"));
				return -1;
			}

			// Test credentials before continuing
			LDAPCredentials* credentials = new LDAPCredentials;
			credentials->username = replicationinfo.syncDN;
			credentials->password = replicationinfo.syncPassword;
			credentials->realm = m_realm;
			LDAPManager* ldap_mgr = new LDAPManager(m_realm, "ldapi://", credentials);
			TQString errorstring;
			if (ldap_mgr->bind(&errorstring) != 0) {
				delete ldap_mgr;
				if (errstr) *errstr = i18n("<qt>LDAP modification failure<p>Reason: Invalid multi-master synchronization credentials provided</qt>");
				else KMessageBox::error(0, i18n("<qt>LDAP modification failure<p>Reason: Invalid multi-master synchronization credentials provided</qt>"), i18n("User Error"));
				return -1;
			}
			delete ldap_mgr;
		}

		if (replicationinfo.serverIDs.count() <= 0) {
			replicationinfo.enabled = false;
		}

		if (replicationinfo.serverIDs.count() > 0) {
			// Assemble the LDAPMod structure
			// We will replace any existing attributes with the new values
			int number_of_parameters = 1;				// 1 primary attribute
			LDAPMod *mods[number_of_parameters+1];
			set_up_attribute_operations(mods, number_of_parameters);
	
			// Load LDAP modification requests from provided data structure
			i=0;
			TQStringList serverMappingList;
			LDAPMasterReplicationMap::iterator it;
			for (it = replicationinfo.serverIDs.begin(); it != replicationinfo.serverIDs.end(); ++it) {
				serverMappingList.append(TQString("%1 ldaps://%2/").arg((*it).id).arg((*it).fqdn));
			}
			add_multiple_attributes_operation(mods, &i, "olcServerID", serverMappingList);
			LDAPMod *prevterm = mods[i];
			mods[i] = NULL;
	
			// Perform LDAP update
			retcode = ldap_modify_ext_s(m_ldap, "cn=config", mods, NULL, NULL);
	
			// Clean up
			clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);
	
			if (retcode == LDAP_NO_SUCH_ATTRIBUTE) {
				// Add new object instead
				// Assemble the LDAPMod structure
				// We will replace any existing attributes with the new values
				int number_of_parameters = 1;				// 1 primary attribute
				LDAPMod *mods[number_of_parameters+1];
				set_up_attribute_operations(mods, number_of_parameters);
		
				// Load LDAP modification requests from provided data structure
				i=0;
				TQStringList serverMappingList;
				LDAPMasterReplicationMap::iterator it;
				for (it = replicationinfo.serverIDs.begin(); it != replicationinfo.serverIDs.end(); ++it) {
					serverMappingList.append(TQString("%1 ldaps://%2/").arg((*it).id).arg((*it).fqdn));
				}
				create_multiple_attributes_operation(mods, &i, "olcServerID", serverMappingList);
				LDAPMod *prevterm = mods[i];
				mods[i] = NULL;
		
				// Perform LDAP update
				retcode = ldap_add_ext_s(m_ldap, "cn=config", mods, NULL, NULL);
		
				// Clean up
				clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);
			}
	
			if (retcode != LDAP_SUCCESS) {
				if (errstr) *errstr = i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
				else KMessageBox::error(0, i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
				return -2;
			}
		}
		else {
			// Delete the olcServerID entry
			// Assemble the LDAPMod structure
			int number_of_parameters = 1;				// 1 primary attribute
			LDAPMod *mods[number_of_parameters+1];
			set_up_attribute_operations(mods, number_of_parameters);
	
			// Load LDAP delete request
			i=0;
			delete_single_attribute_operation(mods, &i, "olcServerID");
			LDAPMod *prevterm = mods[i];
			mods[i] = NULL;
	
			// Perform LDAP update
			retcode = ldap_modify_ext_s(m_ldap, "cn=config", mods, NULL, NULL);
	
			// Clean up
			clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

			if ((retcode != LDAP_SUCCESS) && (retcode != LDAP_NO_SUCH_ATTRIBUTE)) {
				if (errstr) *errstr = i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
				else KMessageBox::error(0, i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
				return -2;
			}
		}

		{
			if (replicationinfo.enabled) {
				// Config Database
				{
					// Assemble the LDAPMod structure
					// We will replace any existing attributes with the new values
					int number_of_parameters = 2;				// 2 primary attributes
					LDAPMod *mods[number_of_parameters+1];
					set_up_attribute_operations(mods, number_of_parameters);

					// Load LDAP modification requests from provided data structure
					i=0;
					TQStringList syncReplServerList;
					LDAPMasterReplicationMap::iterator it;
					int rid = 1;
					for (it = replicationinfo.serverIDs.begin(); it != replicationinfo.serverIDs.end(); ++it) {
						TQString ridString;
						TQString serverSyncReplString;
						TQString databaseDN;
						ridString.sprintf("%03d", rid);
						databaseDN = "cn=config";
						serverSyncReplString = TQString("rid=%1 provider=ldaps://%2/ binddn=\"%3\" bindmethod=simple credentials=\"%4\" searchbase=\"%5\" filter=\"%6\" type=refreshAndPersist scope=\"sub\" attrs=\"*,+\" schemachecking=off retry=\"%7\" timeout=%8 tls_reqcert=%9").arg(ridString).arg((*it).fqdn).arg(replicationinfo.syncDN).arg(replicationinfo.syncPassword).arg(databaseDN).arg((replicationinfo.replicate_olcGlobal)?"(objectClass=*)":"(&(objectclass=*)(!(objectclass=olcGlobal)))").arg(replicationinfo.retryMethod).arg(replicationinfo.timeout).arg((replicationinfo.ignore_ssl_failure)?"never":"demand");
						if (replicationinfo.certificateFile != "") {
							serverSyncReplString.append(TQString(" tls_cert=\"%1\"").arg(replicationinfo.certificateFile));
						}
						if (replicationinfo.caCertificateFile != "") {
							serverSyncReplString.append(TQString(" tls_cacert=\"%1\"").arg(replicationinfo.caCertificateFile));
						}
						syncReplServerList.append(serverSyncReplString);
						rid++;
					}
					add_multiple_attributes_operation(mods, &i, "olcSyncRepl", syncReplServerList);
					add_single_attribute_operation(mods, &i, "olcMirrorMode", "TRUE");
					LDAPMod *prevterm = mods[i];
					mods[i] = NULL;

					// Perform LDAP update
					retcode = ldap_modify_ext_s(m_ldap, "olcDatabase={0}config,cn=config", mods, NULL, NULL);

					// Clean up
					clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

					if (retcode != LDAP_SUCCESS) {
						if (errstr) *errstr = i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
						else KMessageBox::error(0, i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
						return -2;
					}
				}

				// Main Database
				{
					// Assemble the LDAPMod structure
					// We will replace any existing attributes with the new values
					int number_of_parameters = 2;				// 2 primary attributes
					LDAPMod *mods[number_of_parameters+1];
					set_up_attribute_operations(mods, number_of_parameters);

					// Load LDAP modification requests from provided data structure
					i=0;
					TQStringList syncReplServerList;
					LDAPMasterReplicationMap::iterator it;
					int rid = 1;
					for (it = replicationinfo.serverIDs.begin(); it != replicationinfo.serverIDs.end(); ++it) {
						TQString ridString;
						TQString serverSyncReplString;
						TQString databaseDN;
						ridString.sprintf("%03d", rid);
						databaseDN = m_basedc;
						serverSyncReplString = TQString("rid=%1 provider=ldaps://%2/ binddn=\"%3\" bindmethod=simple credentials=\"%4\" searchbase=\"%5\" filter=\"%6\" type=refreshAndPersist scope=\"sub\" attrs=\"*,+\" schemachecking=off retry=\"%7\" timeout=%8 tls_reqcert=%9").arg(ridString).arg((*it).fqdn).arg(replicationinfo.syncDN).arg(replicationinfo.syncPassword).arg(databaseDN).arg((replicationinfo.replicate_olcGlobal)?"(objectClass=*)":"(&(objectclass=*)(!(objectclass=olcGlobal)))").arg(replicationinfo.retryMethod).arg(replicationinfo.timeout).arg((replicationinfo.ignore_ssl_failure)?"never":"demand");
						if (replicationinfo.certificateFile != "") {
							serverSyncReplString.append(TQString(" tls_cert=\"%1\"").arg(replicationinfo.certificateFile));
						}
						if (replicationinfo.caCertificateFile != "") {
							serverSyncReplString.append(TQString(" tls_cacert=\"%1\"").arg(replicationinfo.caCertificateFile));
						}
						syncReplServerList.append(serverSyncReplString);
						rid++;
					}
					add_multiple_attributes_operation(mods, &i, "olcSyncRepl", syncReplServerList);
					add_single_attribute_operation(mods, &i, "olcMirrorMode", "TRUE");
					LDAPMod *prevterm = mods[i];
					mods[i] = NULL;

					// Perform LDAP update
					retcode = ldap_modify_ext_s(m_ldap, "olcDatabase={1}hdb,cn=config", mods, NULL, NULL);

					// Clean up
					clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);

					if (retcode != LDAP_SUCCESS) {
						if (errstr) *errstr = i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
						else KMessageBox::error(0, i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
						return -2;
					}
				}
			}
			else {
				// Delete the olcSyncRepl and olcMirrorMode entries

				// Main Database
				{
					// Assemble the LDAPMod structure
					int number_of_parameters = 2;				// 2 primary attributes
					LDAPMod *mods[number_of_parameters+1];
					set_up_attribute_operations(mods, number_of_parameters);
			
					// Load LDAP delete request
					i=0;
					delete_single_attribute_operation(mods, &i, "olcSyncRepl");
					delete_single_attribute_operation(mods, &i, "olcMirrorMode");
					LDAPMod *prevterm = mods[i];
					mods[i] = NULL;
			
					// Perform LDAP update
					retcode = ldap_modify_ext_s(m_ldap, "olcDatabase={1}hdb,cn=config", mods, NULL, NULL);
			
					// Clean up
					clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);
		
					if ((retcode != LDAP_SUCCESS) && (retcode != LDAP_NO_SUCH_ATTRIBUTE)) {
						if (errstr) *errstr = i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
						else KMessageBox::error(0, i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
						return -2;
					}
				}

				// Config Database
				{
					// Assemble the LDAPMod structure
					int number_of_parameters = 2;				// 2 primary attributes
					LDAPMod *mods[number_of_parameters+1];
					set_up_attribute_operations(mods, number_of_parameters);
			
					// Load LDAP delete request
					i=0;
					delete_single_attribute_operation(mods, &i, "olcSyncRepl");
					delete_single_attribute_operation(mods, &i, "olcMirrorMode");
					LDAPMod *prevterm = mods[i];
					mods[i] = NULL;
			
					// Perform LDAP update
					retcode = ldap_modify_ext_s(m_ldap, "olcDatabase={0}config,cn=config", mods, NULL, NULL);
			
					// Clean up
					clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);
		
					if ((retcode != LDAP_SUCCESS) && (retcode != LDAP_NO_SUCH_ATTRIBUTE)) {
						if (errstr) *errstr = i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
						else KMessageBox::error(0, i18n("<qt>LDAP modification failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
						return -2;
					}
				}
			}

			// Get current active replication settings
			TQString* readOnlyErrorString = NULL;
			LDAPMasterReplicationInfo currentReplicationInfo = getLDAPMasterReplicationSettings(readOnlyErrorString);
			if (readOnlyErrorString) {
				// Uh oh
				if (errstr) *errstr = *readOnlyErrorString;
				else KMessageBox::error(0, *readOnlyErrorString, i18n("LDAP Error"));
				return -2;
			}
			if (replicationinfo.enabled) {
				// Set up replication
				// NOTE: The syncprov module itself is already loaded by the stock TDE LDAP configuration

				// Check to see if the syncprov overlay entries already exist
				bool haveOlcOverlaySyncProv = false;
				LDAPMessage* msg;
				retcode = ldap_search_ext_s(m_ldap, "olcDatabase={0}config,cn=config", LDAP_SCOPE_SUBTREE, NULL, ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
				if (retcode != LDAP_SUCCESS) {
					if (errstr) *errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					else KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					return -2;
				}

				// Iterate through the returned entries
				LDAPMessage* entry;
				for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
					if (parseLDAPSyncProvOverlayConfigRecord(entry) != "") {
						haveOlcOverlaySyncProv = true;
					}
				}

				// Clean up
				ldap_msgfree(msg);

				if (!haveOlcOverlaySyncProv) {
					// Create the base DN entry
					int number_of_parameters = 1;				// 1 primary attribute
					LDAPMod *mods[number_of_parameters+1];
					set_up_attribute_operations(mods, number_of_parameters);

					// Load initial required LDAP object attributes
					i=0;
					TQStringList objectClassList;
					objectClassList.append("olcOverlayConfig");
					objectClassList.append("olcSyncProvConfig");
					create_multiple_attributes_operation(mods, &i, "objectClass", objectClassList);
					LDAPMod *prevterm = mods[i];
					mods[i] = NULL;
			
					// Add new object
					retcode = ldap_add_ext_s(m_ldap, "olcOverlay=syncprov,olcDatabase={0}config,cn=config", mods, NULL, NULL);
			
					// Clean up
					clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);
			
					if (retcode != LDAP_SUCCESS) {
						if (errstr) {
							*errstr = i18n("<qt>LDAP overlay configuration failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
						}
						else {
							KMessageBox::error(0, i18n("<qt>LDAP overlay configuration failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
						}
						return -2;
					}
				}

				haveOlcOverlaySyncProv = false;
				retcode = ldap_search_ext_s(m_ldap, "olcDatabase={1}hdb,cn=config", LDAP_SCOPE_SUBTREE, NULL, ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
				if (retcode != LDAP_SUCCESS) {
					if (errstr) *errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
					else KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
					return -2;
				}

				// Iterate through the returned entries
				for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
					if (parseLDAPSyncProvOverlayConfigRecord(entry) != "") {
						haveOlcOverlaySyncProv = true;
					}
				}

				// Clean up
				ldap_msgfree(msg);

				if (!haveOlcOverlaySyncProv) {
					// Create the base DN entry
					int number_of_parameters = 1;				// 1 primary attribute
					LDAPMod *mods[number_of_parameters+1];
					set_up_attribute_operations(mods, number_of_parameters);

					// Load initial required LDAP object attributes
					i=0;
					TQStringList objectClassList;
					objectClassList.append("olcOverlayConfig");
					objectClassList.append("olcSyncProvConfig");
					create_multiple_attributes_operation(mods, &i, "objectClass", objectClassList);
					LDAPMod *prevterm = mods[i];
					mods[i] = NULL;
			
					// Add new object
					retcode = ldap_add_ext_s(m_ldap, "olcOverlay=syncprov,olcDatabase={1}hdb,cn=config", mods, NULL, NULL);
			
					// Clean up
					clean_up_attribute_operations(i, mods, prevterm, number_of_parameters);
			
					if (retcode != LDAP_SUCCESS) {
						if (errstr) {
							*errstr = i18n("<qt>LDAP overlay configuration failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
						}
						else {
							KMessageBox::error(0, i18n("<qt>LDAP overlay configuration failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
						}
						return -2;
					}
				}
			}
			else {
				// FIXME
				// OpenLDAP does not support removing overlays from the cn=config interface (i.e., once they are enabled above, they stay unless manually deleted from the config files)
				// See http://www.openldap.org/lists/openldap-software/200811/msg00103.html
				// If it were possible, the code would look something like this:
				// retcode = ldap_delete_ext_s(m_ldap, olcOverlaySyncProvAttr + ",olcDatabase={0}config,cn=config", NULL, NULL);
				// retcode = ldap_delete_ext_s(m_ldap, olcOverlaySyncProvAttr + ",olcDatabase={1}hdb,cn=config", NULL, NULL);
			}
			return 0;
		}
	}

	return -1;
}

int LDAPManager::getTDECertificate(TQString certificateName, TQString fileName, TQString *errstr) {
	int retcode;
	int returncode;
	LDAPTDEBuiltinsInfo builtininfo;

	TQString dn = TQString("cn=certificate store,o=tde,cn=tde realm data,ou=master services,ou=core,ou=realm,%1").arg(m_basedc);

	if (bind(errstr) < 0) {
		return -1;
	}
	else {
		LDAPMessage* msg;
		retcode = ldap_search_ext_s(m_ldap, dn.ascii(), LDAP_SCOPE_SUBTREE, NULL, ldap_user_and_operational_attributes, 0, NULL, NULL, NULL, 0, &msg);
		if (retcode != LDAP_SUCCESS) {
			if (errstr) *errstr = i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode));
			else KMessageBox::error(0, i18n("<qt>LDAP search failure<p>Reason: [%3] %4</qt>").arg(retcode).arg(ldap_err2string(retcode)), i18n("LDAP Error"));
			return -1;
		}

		returncode = -2;

		// Iterate through the returned entries
		LDAPMessage* entry;
		for(entry = ldap_first_entry(m_ldap, msg); entry != NULL; entry = ldap_next_entry(m_ldap, entry)) {
			char* attr;
			struct berval **vals;
			BerElement* ber;
			int i;

			LDAPTDEBuiltinsInfo builtininfo;

			for( attr = ldap_first_attribute(m_ldap, entry, &ber); attr != NULL; attr = ldap_next_attribute(m_ldap, entry, ber)) {
				if ((vals = ldap_get_values_len(m_ldap, entry, attr)) != NULL)  {
					builtininfo.informationValid = true;
					TQString ldap_field = attr;
					i=0;
					if (ldap_field == certificateName) {
						TQFile file(fileName);
						if (file.open(IO_WriteOnly)) {
							TQByteArray ba;
							ba.duplicate(vals[i]->bv_val, vals[i]->bv_len);
							file.writeBlock(ba);
							file.close();
							if (chmod(fileName.ascii(), S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH) < 0) {
								if (errstr) *errstr = i18n("Unable to change permissions of \"%1\"").arg(fileName.ascii());
								returncode = -1;
							}
							else {
								returncode = 0;
							}
						}
					}
					ldap_value_free_len(vals);
				}
				ldap_memfree(attr);
			}

			if (ber != NULL) {
				ber_free(ber, 0);
			}
		}

		// Clean up
		ldap_msgfree(msg);

		return returncode;
	}

	return -1;
}

int LDAPManager::writeSudoersConfFile(TQString *errstr) {
	LDAPTDEBuiltinsInfo tdebuiltins = getTDEBuiltinMappings(errstr);
	if (!tdebuiltins.informationValid) {
		if (errstr) *errstr = i18n("Unable to read builtin TDE user/group mappings");
		return -1;
	}

	TQString localadmingroup = tdebuiltins.builtinMachineAdminGroup;
	int eqpos = localadmingroup.find("=")+1;
	int cmpos = localadmingroup.find(",", eqpos);
	localadmingroup.truncate(cmpos);
	localadmingroup.remove(0, eqpos);

	TQFile file(TDELDAP_SUDO_D_FILE);
	if (file.open(IO_WriteOnly)) {
		TQTextStream stream( &file );

		stream << "# This file was automatically generated by TDE\n";
		stream << "# All changes will be lost!\n";
		stream << "\n";
		stream << "# Realm local machine administrators\n";
		stream << "%" << localadmingroup << "  ALL=NOPASSWD: ALL" << "\n";

		file.close();
	}

	if (chown(TDELDAP_SUDO_D_FILE, 0, 0) < 0) {
		printf("ERROR: Unable to change owner of \"%s\"\n", TDELDAP_SUDO_D_FILE);
		return -1;
	}
	if (chmod(TDELDAP_SUDO_D_FILE, S_IRUSR|S_IRGRP) < 0) {
		printf("ERROR: Unable to change permissions of \"%s\"\n", TDELDAP_SUDO_D_FILE);
		return -1;
	}

	return 0;
}

int LDAPManager::writeClientCronFiles(TQString *errstr) {
	TQFile file(CRON_UPDATE_NSS_FILE);
	if (file.open(IO_WriteOnly)) {
		TQTextStream stream( &file );

		stream << "#!/bin/sh" << "\n";
		stream << "\n";
		stream << "# This file was automatically generated by TDE\n";
		stream << "# All changes will be lost!\n";
		stream << "\n";
		stream << CRON_UPDATE_NSS_COMMAND << "\n";

		file.close();

		if (chmod(CRON_UPDATE_NSS_FILE, S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH) < 0) {
			if (errstr) *errstr = TQString("Unable to change permissions of \"%1\"").arg(CRON_UPDATE_NSS_FILE);
			return -1;
		}
	}
	else {
		if (errstr) *errstr = TQString("Unable to write file \"%1\"").arg(CRON_UPDATE_NSS_FILE);
		return -1;
	}

	if (system(CRON_UPDATE_NSS_COMMAND) < 0) {
		if (errstr) *errstr = TQString("Execution of \"%s\" failed").arg(CRON_UPDATE_NSS_COMMAND);
		return -1;
	}

	return 0;
}

int LDAPManager::writePrimaryRealmCertificateUpdateCronFile(TQString *errstr) {
	TQFile file(CRON_UPDATE_PRIMARY_REALM_CERTIFICATES_FILE);
	if (file.open(IO_WriteOnly)) {
		TQTextStream stream( &file );

		stream << "#!/bin/sh" << "\n";
		stream << "\n";
		stream << "# This file was automatically generated by TDE\n";
		stream << "# All changes will be lost!\n";
		stream << "\n";
		stream << CRON_UPDATE_PRIMARY_REALM_CERTIFICATES_COMMAND << "\n";
		stream << CRON_UPDATE_PRIMARY_REALM_CERTIFICATES_OPENLDAP_RELOAD_COMMAND << "\n";

		file.close();

		if (chmod(CRON_UPDATE_PRIMARY_REALM_CERTIFICATES_FILE, S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH) < 0) {
			if (errstr) *errstr = TQString("Unable to change permissions of \"%1\"").arg(CRON_UPDATE_PRIMARY_REALM_CERTIFICATES_FILE);
			return -1;
		}
	}
	else {
		if (errstr) *errstr = TQString("Unable to write file \"%1\"").arg(CRON_UPDATE_PRIMARY_REALM_CERTIFICATES_FILE);
		return -1;
	}

	if (system(CRON_UPDATE_PRIMARY_REALM_CERTIFICATES_COMMAND) < 0) {
		if (errstr) *errstr = TQString("Execution of \"%s\" failed").arg(CRON_UPDATE_PRIMARY_REALM_CERTIFICATES_COMMAND);
		return -1;
	}

	return 0;
}

LDAPRealmConfigList LDAPManager::readTDERealmList(KSimpleConfig* config, bool disableAllBonds) {
	LDAPRealmConfigList realms;

	TQStringList cfgRealms = config->groupList();
	for (TQStringList::Iterator it(cfgRealms.begin()); it != cfgRealms.end(); ++it) {
		if ((*it).startsWith("LDAPRealm-")) {
			config->setGroup(*it);
			TQString realmName=*it;
			realmName.remove(0,strlen("LDAPRealm-"));
			if (!realms.contains(realmName)) {
				// Read in realm data
				LDAPRealmConfig realmcfg;
				realmcfg.name = realmName;
				if (!disableAllBonds) {
					realmcfg.bonded = config->readBoolEntry("bonded");
				}
				else {
					realmcfg.bonded = false;
				}
				realmcfg.uid_offset = config->readNumEntry("uid_offset");
				realmcfg.gid_offset = config->readNumEntry("gid_offset");
				realmcfg.domain_mappings = config->readListEntry("domain_mappings");
				realmcfg.kdc = config->readEntry("kdc");
				realmcfg.kdc_port = config->readNumEntry("kdc_port");
				realmcfg.admin_server = config->readEntry("admin_server");
				realmcfg.admin_server_port = config->readNumEntry("admin_server_port");
				realmcfg.pkinit_require_eku = config->readBoolEntry("pkinit_require_eku");
				realmcfg.pkinit_require_krbtgt_otherName = config->readBoolEntry("pkinit_require_krbtgt_otherName");
				realmcfg.win2k_pkinit = config->readBoolEntry("win2k_pkinit");
				realmcfg.win2k_pkinit_require_binding = config->readBoolEntry("win2k_pkinit_require_binding");
				// Add realm to list
				realms.insert(realmName, realmcfg);
			}
		}
	}

	return realms;
}

int LDAPManager::writeTDERealmList(LDAPRealmConfigList realms, KSimpleConfig* config, TQString *errstr) {
	LDAPRealmConfigList::Iterator it;
	for (it = realms.begin(); it != realms.end(); ++it) {
		LDAPRealmConfig realmcfg = it.data();
		TQString configRealmName = realmcfg.name;
		configRealmName.prepend("LDAPRealm-");
		config->setGroup(configRealmName);
		// Save realm settings
		config->writeEntry("bonded", realmcfg.bonded);
		config->writeEntry("uid_offset", realmcfg.uid_offset);
		config->writeEntry("gid_offset", realmcfg.gid_offset);
		config->writeEntry("domain_mappings", realmcfg.domain_mappings);
		config->writeEntry("kdc", realmcfg.kdc);
		config->writeEntry("kdc_port", realmcfg.kdc_port);
		config->writeEntry("admin_server", realmcfg.admin_server);
		config->writeEntry("admin_server_port", realmcfg.admin_server_port);
		config->writeEntry("pkinit_require_eku", realmcfg.pkinit_require_eku);
		config->writeEntry("pkinit_require_krbtgt_otherName", realmcfg.pkinit_require_krbtgt_otherName);
		config->writeEntry("win2k_pkinit", realmcfg.win2k_pkinit);
		config->writeEntry("win2k_pkinit_require_binding", realmcfg.win2k_pkinit_require_binding);
	}

	// Delete any realms that do not exist in the realms database
	TQStringList cfgRealms = config->groupList();
	for (TQStringList::Iterator it(cfgRealms.begin()); it != cfgRealms.end(); ++it) {
		if ((*it).startsWith("LDAPRealm-")) {
			config->setGroup(*it);
			TQString realmName=*it;
			realmName.remove(0,strlen("LDAPRealm-"));
			if (!realms.contains(realmName)) {
				config->deleteGroup(*it);
			}
		}
	}

	return 0;
}

TQDateTime LDAPManager::getCertificateExpiration(TQString certfile) {
	TQDateTime ret;

	TQFile file(certfile);
	if (file.open(IO_ReadOnly)) {
		TQByteArray ba = file.readAll();
		file.close();

		TQCString ssldata(ba);
		ssldata.replace("-----BEGIN CERTIFICATE-----", "");
		ssldata.replace("-----END CERTIFICATE-----", "");
		ssldata.replace("\n", "");
		KSSLCertificate* cert = KSSLCertificate::fromString(ssldata);
		if (cert) {
			ret = cert->getQDTNotAfter();
			delete cert;
		}
	}

	return ret;
}

int LDAPManager::generatePublicKerberosCACertificate(LDAPCertConfig certinfo) {
	TQString command;

	command = TQString("openssl req -key %1 -new -x509 -out %2 -subj \"/C=%3/ST=%4/L=%5/O=%6/OU=%7/CN=%8/emailAddress=%9\"").arg(KERBEROS_PKI_PEMKEY_FILE).arg(KERBEROS_PKI_PEM_FILE).arg(certinfo.countryName).arg(certinfo.stateOrProvinceName).arg(certinfo.localityName).arg(certinfo.organizationName).arg(certinfo.orgUnitName).arg(certinfo.commonName).arg(certinfo.emailAddress);
	if (system(command) < 0) {
		printf("ERROR: Execution of \"%s\" failed!\n", command.ascii());
		return -1;
	}
	if (chmod(KERBEROS_PKI_PEM_FILE, S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH) < 0) {
		printf("ERROR: Unable to change permissions of \"%s\"\n", KERBEROS_PKI_PEM_FILE);
		return -1;
	}
	if (chown(KERBEROS_PKI_PEM_FILE, 0, 0) < 0) {
		printf("ERROR: Unable to change owner of \"%s\"\n", KERBEROS_PKI_PEM_FILE);
		return -1;
	}

	return 0;
}

int LDAPManager::generatePublicKerberosCertificate(LDAPCertConfig certinfo, LDAPRealmConfig realmcfg) {
	TQString command;

	TQString kdc_certfile = KERBEROS_PKI_KDC_FILE;
	TQString kdc_keyfile = KERBEROS_PKI_KDCKEY_FILE;
	TQString kdc_reqfile = KERBEROS_PKI_KDCREQ_FILE;
	kdc_certfile.replace("@@@KDCSERVER@@@", realmcfg.name.lower());
	kdc_keyfile.replace("@@@KDCSERVER@@@", realmcfg.name.lower());
	kdc_reqfile.replace("@@@KDCSERVER@@@", realmcfg.name.lower());

	command = TQString("openssl req -new -out %1 -key %2 -subj \"/C=%3/ST=%4/L=%5/O=%6/OU=%7/CN=%8/emailAddress=%9\"").arg(kdc_reqfile).arg(kdc_keyfile).arg(certinfo.countryName).arg(certinfo.stateOrProvinceName).arg(certinfo.localityName).arg(certinfo.organizationName).arg(certinfo.orgUnitName).arg(certinfo.commonName).arg(certinfo.emailAddress);
	if (system(command) < 0) {
		printf("ERROR: Execution of \"%s\" failed!\n", command.ascii());
		return -1;
	}
	command = TQString("openssl x509 -req -in %1 -CAkey %2 -CA %3 -out %4 -extfile %5 -extensions kdc_cert -CAcreateserial").arg(kdc_reqfile).arg(KERBEROS_PKI_PEMKEY_FILE).arg(KERBEROS_PKI_PEM_FILE).arg(kdc_certfile).arg(OPENSSL_EXTENSIONS_FILE);
	if (system(command) < 0) {
		printf("ERROR: Execution of \"%s\" failed!\n", command.ascii());
		return -1;
	}
	if (chmod(kdc_certfile.ascii(), S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH) < 0) {
		printf("ERROR: Unable to change permissions of \"%s\"\n", kdc_certfile.ascii());
		return -1;
	}
	if (chown(kdc_certfile.ascii(), 0, 0) < 0) {
		printf("ERROR: Unable to change owner of \"%s\"\n", kdc_certfile.ascii());
		return -1;
	}
	if (fileExists(kdc_reqfile.ascii())) {
		if (unlink(kdc_reqfile.ascii()) < 0) {
			printf("ERROR: Unable to unlink \"%s\"\n", kdc_reqfile.ascii());
			return -1;
		}
	}

	return 0;
}

int LDAPManager::generatePublicLDAPCertificate(LDAPCertConfig certinfo, LDAPRealmConfig realmcfg, uid_t ldap_uid, gid_t ldap_gid) {
	TQString command;

	TQString ldap_certfile = LDAP_CERT_FILE;
	TQString ldap_keyfile = LDAP_CERTKEY_FILE;
	TQString ldap_reqfile = LDAP_CERTREQ_FILE;
	ldap_certfile.replace("@@@ADMINSERVER@@@", realmcfg.name.lower());
	ldap_keyfile.replace("@@@ADMINSERVER@@@", realmcfg.name.lower());
	ldap_reqfile.replace("@@@ADMINSERVER@@@", realmcfg.name.lower());

	command = TQString("openssl req -new -out %1 -key %2 -subj \"/C=%3/ST=%4/L=%5/O=%6/OU=%7/CN=%8/emailAddress=%9\"").arg(ldap_reqfile).arg(ldap_keyfile).arg(certinfo.countryName).arg(certinfo.stateOrProvinceName).arg(certinfo.localityName).arg(certinfo.organizationName).arg(certinfo.orgUnitName).arg(realmcfg.admin_server).arg(certinfo.emailAddress);
	if (system(command) < 0) {
		printf("ERROR: Execution of \"%s\" failed!\n", command.ascii());
		return -1;
	}
	command = TQString("openssl x509 -req -in %1 -CAkey %2 -CA %3 -out %4 -CAcreateserial").arg(ldap_reqfile).arg(KERBEROS_PKI_PEMKEY_FILE).arg(KERBEROS_PKI_PEM_FILE).arg(ldap_certfile);
	if (system(command) < 0) {
		printf("ERROR: Execution of \"%s\" failed!\n", command.ascii());
		return -1;
	}
	if (chmod(ldap_certfile.ascii(), S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH) < 0) {
		printf("ERROR: Unable to change permissions of \"%s\"\n", ldap_certfile.ascii());
		return -1;
	}
	if (chown(ldap_certfile.ascii(), ldap_uid, ldap_gid) < 0) {
		printf("ERROR: Unable to change owner of \"%s\"\n", ldap_certfile.ascii());
		return -1;
	}
	if (fileExists(ldap_reqfile.ascii())) {
		if (unlink(ldap_reqfile.ascii()) < 0) {
			printf("ERROR: Unable to unlink \"%s\"\n", ldap_reqfile.ascii());
			return -1;
		}
	}

	return 0;
}

TQString LDAPManager::getMachineFQDN() {
	struct addrinfo hints, *info, *p;
	int gai_result;
	
	char hostname[1024];
	hostname[1023] = '\0';
	gethostname(hostname, 1023);
	
	memset(&hints, 0, sizeof hints);
	hints.ai_family = AF_UNSPEC; // IPV4 or IPV6
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_flags = AI_CANONNAME;
	
	if ((gai_result = getaddrinfo(hostname, NULL, &hints, &info)) != 0) {
		return TQString(hostname);
	}
	TQString fqdn = TQString(hostname);
	for (p=info; p!=NULL; p=p->ai_next) {
		fqdn = TQString(p->ai_canonname);
	}
	freeaddrinfo(info);

	return fqdn;
}

LDAPClientRealmConfig LDAPManager::loadClientRealmConfig(KSimpleConfig* config, bool useDefaults) {
	LDAPClientRealmConfig clientRealmConfig;

	config->setReadDefaults(useDefaults);

	config->setGroup(NULL);
	clientRealmConfig.enable_bonding = config->readBoolEntry("EnableLDAP", false);
	clientRealmConfig.defaultRealm = config->readEntry("DefaultRealm", TQString::null);
	clientRealmConfig.ticketLifetime = config->readNumEntry("TicketLifetime", 86400);
	clientRealmConfig.ldapRole = config->readEntry("LDAPRole", "Workstation");
	if (LDAPManager::getMachineFQDN() == config->readEntry("HostFQDN", "")) {
		clientRealmConfig.configurationVerifiedForLocalMachine = true;
	}
	else {
		clientRealmConfig.configurationVerifiedForLocalMachine = false;
	}

	clientRealmConfig.ldapVersion = config->readNumEntry("ConnectionLDAPVersion", 3);
	clientRealmConfig.ldapTimeout = config->readNumEntry("ConnectionLDAPTimeout", 2);
	clientRealmConfig.bindPolicy = config->readEntry("ConnectionBindPolicy", "soft");
	clientRealmConfig.ldapBindTimeout = config->readNumEntry("ConnectionBindTimeout", 2);
	clientRealmConfig.passwordHash = config->readEntry("ConnectionPasswordHash", "exop");
	clientRealmConfig.ignoredUsers = config->readEntry("ConnectionIgnoredUsers", DEFAULT_IGNORED_USERS_LIST);

	clientRealmConfig.pamConfig.enable_cached_credentials = config->readBoolEntry("EnableCachedCredentials", true);
	clientRealmConfig.pamConfig.autocreate_user_directories_enable = config->readBoolEntry("EnableAutoUserDir", true);
	clientRealmConfig.pamConfig.autocreate_user_directories_umask = config->readNumEntry("AutoUserDirUmask", 0022);
	clientRealmConfig.pamConfig.autocreate_user_directories_skel = config->readEntry("AutoUserDirSkelDir", "/etc/skel");

	return clientRealmConfig;
}

int LDAPManager::saveClientRealmConfig(LDAPClientRealmConfig clientRealmConfig, KSimpleConfig* config, TQString *errstr) {
	config->setGroup(NULL);
	config->writeEntry("EnableLDAP", clientRealmConfig.enable_bonding);
	config->writeEntry("HostFQDN", clientRealmConfig.hostFQDN);

	if (clientRealmConfig.defaultRealm != "") {
		config->writeEntry("DefaultRealm", clientRealmConfig.defaultRealm);
	}
	else {
		config->deleteEntry("DefaultRealm");
	}
	config->writeEntry("TicketLifetime", clientRealmConfig.ticketLifetime);

	config->writeEntry("ConnectionLDAPVersion", clientRealmConfig.ldapVersion);
	config->writeEntry("ConnectionLDAPTimeout", clientRealmConfig.ldapTimeout);
	config->writeEntry("ConnectionBindPolicy", clientRealmConfig.bindPolicy);
	config->writeEntry("ConnectionBindTimeout", clientRealmConfig.ldapBindTimeout);
	config->writeEntry("ConnectionPasswordHash", clientRealmConfig.passwordHash);
	config->writeEntry("ConnectionIgnoredUsers", clientRealmConfig.ignoredUsers);

	config->writeEntry("EnableCachedCredentials", clientRealmConfig.pamConfig.enable_cached_credentials);
	config->writeEntry("EnableAutoUserDir", clientRealmConfig.pamConfig.autocreate_user_directories_enable);
	config->writeEntry("AutoUserDirUmask", clientRealmConfig.pamConfig.autocreate_user_directories_umask);
	config->writeEntry("AutoUserDirSkelDir", clientRealmConfig.pamConfig.autocreate_user_directories_skel);

	return 0;
}

int LDAPManager::writeClientKrb5ConfFile(LDAPClientRealmConfig clientRealmConfig, LDAPRealmConfigList realmList, TQString *errstr) {
	TQFile file(KRB5_FILE);
	if (file.open(IO_WriteOnly)) {
		TQTextStream stream( &file );

		stream << "# This file was automatically generated by TDE\n";
		stream << "# All changes will be lost!\n";
		stream << "\n";

		// Defaults
		stream << "[libdefaults]\n";
		stream << "    ticket_lifetime = " << clientRealmConfig.ticketLifetime << "\n";
		if (clientRealmConfig.defaultRealm != "") {
			stream << "    default_realm = " << clientRealmConfig.defaultRealm << "\n";
		}
		stream << "\n";

		// Realms
		stream << "[realms]\n";
		LDAPRealmConfigList::Iterator it;
		for (it = realmList.begin(); it != realmList.end(); ++it) {
			LDAPRealmConfig realmcfg = it.data();
			stream << "   " << realmcfg.name << " = {\n";
			stream << "        kdc = " << realmcfg.kdc << ":" << realmcfg.kdc_port << "\n";
			stream << "        admin_server = " << realmcfg.admin_server << ":" << realmcfg.admin_server_port << "\n";
			stream << "        pkinit_require_eku = " << (realmcfg.pkinit_require_eku?"true":"false") << "\n";
			stream << "        pkinit_require_krbtgt_otherName = " << (realmcfg.pkinit_require_krbtgt_otherName?"true":"false") << "\n";
			stream << "        win2k_pkinit = " << (realmcfg.win2k_pkinit?"yes":"no") << "\n";
			stream << "        win2k_pkinit_require_binding = " << (realmcfg.win2k_pkinit_require_binding?"yes":"no") << "\n";
			stream << "   }\n";
		}
		stream << "\n";

		// Domain aliases
		stream << "[domain_realm]\n";
		LDAPRealmConfigList::Iterator it2;
		for (it2 = realmList.begin(); it2 != realmList.end(); ++it2) {
			LDAPRealmConfig realmcfg = it2.data();
			TQStringList domains = realmcfg.domain_mappings;
			for (TQStringList::Iterator it3 = domains.begin(); it3 != domains.end(); ++it3 ) {
				stream << "    " << *it3 << " = " << realmcfg.name << "\n";
			}
		}

		file.close();
	}

	return 0;
}

int LDAPManager::writeNSSwitchFile(TQString *errstr) {
	TQFile file(NSSWITCH_FILE);
	if (file.open(IO_WriteOnly)) {
		TQTextStream stream( &file );

		stream << "# This file was automatically generated by TDE\n";
		stream << "# All changes will be lost!\n";
		stream << "\n";
		stream << "passwd:         files ldap [NOTFOUND=return] db" << "\n";
		stream << "group:          files ldap [NOTFOUND=return] db" << "\n";
		stream << "shadow:         files ldap [NOTFOUND=return] db" << "\n";
		stream << "\n";
		stream << "hosts:          files mdns4_minimal [NOTFOUND=return] dns mdns4" << "\n";
		stream << "networks:       files" << "\n";
		stream << "\n";
		stream << "protocols:      db files" << "\n";
		stream << "services:       db files" << "\n";
		stream << "ethers:         db files" << "\n";
		stream << "rpc:            db files" << "\n";
		stream << "\n";
		stream << "netgroup:       nis" << "\n";

		file.close();
	}

	return 0;
}

int LDAPManager::writePAMFiles(LDAPPamConfig pamConfig, TQString *errstr) {
	TQFile file(PAMD_DIRECTORY PAMD_COMMON_ACCOUNT);
	if (file.open(IO_WriteOnly)) {
		TQTextStream stream( &file );

		stream << "# This file was automatically generated by TDE\n";
		stream << "# All changes will be lost!\n";
		stream << "\n";
		stream << "account sufficient pam_unix.so nullok_secure" << "\n";
		stream << "account sufficient pam_ldap.so" << "\n";
		stream << "account required pam_permit.so" << "\n";

		file.close();
	}

	TQFile file2(PAMD_DIRECTORY PAMD_COMMON_AUTH);
	if (file2.open(IO_WriteOnly)) {
		TQTextStream stream( &file2 );

		stream << "# This file was automatically generated by TDE\n";
		stream << "# All changes will be lost!\n";
		stream << "\n";
		stream << "auth [default=ignore success=ignore] pam_mount.so" << "\n";
		stream << "auth sufficient pam_unix.so nullok try_first_pass" << "\n";
		stream << "auth [default=ignore success=1 service_err=reset] pam_krb5.so ccache=/tmp/krb5cc_%u use_first_pass" << "\n";
		if (pamConfig.enable_cached_credentials) {
			stream << "auth [default=2 success=done] pam_ccreds.so action=validate use_first_pass" << "\n";
			stream << "auth sufficient pam_ccreds.so action=store use_first_pass" << "\n";
		}
		stream << "auth required pam_deny.so" << "\n";

		file2.close();
	}

	TQFile file3(PAMD_DIRECTORY PAMD_COMMON_SESSION);
	if (file3.open(IO_WriteOnly)) {
		TQTextStream stream( &file3 );

		stream << "# This file was automatically generated by TDE\n";
		stream << "# All changes will be lost!\n";
		stream << "\n";
		stream << "session [default=1] pam_permit.so" << "\n";
		stream << "session requisite pam_deny.so" << "\n";
		stream << "session required pam_permit.so" << "\n";
		stream << "session required pam_unix.so" << "\n";
		stream << "session optional pam_ck_connector.so nox11" << "\n";
		if (pamConfig.autocreate_user_directories_enable) {
			char modestring[8];
			sprintf(modestring, "%04o", pamConfig.autocreate_user_directories_umask);
			TQString skelstring;
			if (pamConfig.autocreate_user_directories_skel != "") {
				skelstring = " skel=" + pamConfig.autocreate_user_directories_skel;
			}
			TQString umaskString;
			if (pamConfig.autocreate_user_directories_umask != 0) {
				umaskString = " umask=";
				umaskString.append(modestring);
			}
			stream << "session required pam_mkhomedir.so" << skelstring << umaskString << "\n";
		}
		stream << "auth required pam_deny.so" << "\n";

		file3.close();
	}

	return 0;
}

int LDAPManager::bondRealm(TQString adminUserName, const char * adminPassword, TQString adminRealm, TQString *errstr) {
	TQCString command = "kadmin";
	QCStringList args;
	args << TQCString("-p") << TQCString(adminUserName+"@"+(adminRealm.upper())) << TQCString("-r") << TQCString(adminRealm.upper());

	TQString hoststring = "host/"+getMachineFQDN();

	TQString prompt;
	PtyProcess kadminProc;
	kadminProc.exec(command, args);
	prompt = readFullLineFromPtyProcess(&kadminProc);
	prompt = prompt.stripWhiteSpace();
	if (prompt == "kadmin>") {
		command = TQCString("ext "+hoststring);
		kadminProc.enableLocalEcho(false);
		kadminProc.writeLine(command, true);
		do { // Discard our own input
			prompt = readFullLineFromPtyProcess(&kadminProc);
			printf("(kadmin) '%s'\n", prompt.ascii());
		} while ((prompt == TQString(command)) || (prompt == ""));
		prompt = prompt.stripWhiteSpace();
		if (prompt.endsWith(" Password:")) {
			kadminProc.enableLocalEcho(false);
			kadminProc.writeLine(adminPassword, true);
			do { // Discard our own input
				prompt = readFullLineFromPtyProcess(&kadminProc);
				printf("(kadmin) '%s'\n", prompt.ascii());
			} while (prompt == "");
			prompt = prompt.stripWhiteSpace();
		}
		if (prompt.contains("authentication failed")) {
			if (errstr) *errstr = prompt;
			do { // Wait for command prompt
				prompt = readFullLineFromPtyProcess(&kadminProc);
				printf("(kadmin) '%s'\n", prompt.ascii());
			} while (prompt == "");
			kadminProc.enableLocalEcho(false);
			kadminProc.writeLine("quit", true);
			return 1;
		}
		else if (prompt.endsWith("Principal does not exist")) {
			do { // Wait for command prompt
				prompt = readFullLineFromPtyProcess(&kadminProc);
				printf("(kadmin) '%s'\n", prompt.ascii());
			} while (prompt == "");
			command = TQCString("ank --random-key "+hoststring);
			kadminProc.enableLocalEcho(false);
			kadminProc.writeLine(command, true);
			do { // Discard our own input
				prompt = readFullLineFromPtyProcess(&kadminProc);
				printf("(kadmin) '%s'\n", prompt.ascii());
			} while ((prompt == TQString(command)) || (prompt == ""));
			prompt = prompt.stripWhiteSpace();
			// Use all defaults
			while (prompt != "kadmin>") {
				if (prompt.endsWith(" Password:")) {
					kadminProc.enableLocalEcho(false);
					kadminProc.writeLine(adminPassword, true);
					do { // Discard our own input
						prompt = readFullLineFromPtyProcess(&kadminProc);
						printf("(kadmin) '%s'\n", prompt.ascii());
					} while (prompt == "");
					prompt = prompt.stripWhiteSpace();
				}
				if (prompt.contains("authentication failed")) {
					if (errstr) *errstr = prompt;
					do { // Wait for command prompt
						prompt = readFullLineFromPtyProcess(&kadminProc);
						printf("(kadmin) '%s'\n", prompt.ascii());
					} while (prompt == "");
					kadminProc.enableLocalEcho(false);
					kadminProc.writeLine("quit", true);
					return 1;
				}
				else {
					// Extract whatever default is in the [brackets] and feed it back to kadmin
					TQString defaultParam;
					int leftbracket = prompt.find("[");
					int rightbracket = prompt.find("]");
					if ((leftbracket >= 0) && (rightbracket >= 0)) {
						leftbracket++;
						defaultParam = prompt.mid(leftbracket, rightbracket-leftbracket);
					}
					command = TQCString(defaultParam);
					kadminProc.enableLocalEcho(false);
					kadminProc.writeLine(command, true);
					do { // Discard our own input
						prompt = readFullLineFromPtyProcess(&kadminProc);
						printf("(kadmin) '%s'\n", prompt.ascii());
					} while ((prompt == TQString(command)) || (prompt == ""));
					prompt = prompt.stripWhiteSpace();
				}
			}
			command = TQCString("ext "+hoststring);
			kadminProc.enableLocalEcho(false);
			kadminProc.writeLine(command, true);
			do { // Discard our own input
				prompt = readFullLineFromPtyProcess(&kadminProc);
				printf("(kadmin) '%s'\n", prompt.ascii());
			} while ((prompt == TQString(command)) || (prompt == ""));
			prompt = prompt.stripWhiteSpace();
			if (prompt != "kadmin>") {
				if (errstr) *errstr = prompt;
				do { // Wait for command prompt
					prompt = readFullLineFromPtyProcess(&kadminProc);
					printf("(kadmin) '%s'\n", prompt.ascii());
				} while (prompt == "");
				kadminProc.enableLocalEcho(false);
				kadminProc.writeLine("quit", true);
				return 1;
			}

			// Success!
			kadminProc.enableLocalEcho(false);
			kadminProc.writeLine("quit", true);

			return 0;
		}
		else if (prompt == "kadmin>") {
			// Success!
			kadminProc.enableLocalEcho(false);
			kadminProc.writeLine("quit", true);

			return 0;
		}

		// Failure
		if (errstr) *errstr = prompt;
		while (prompt == "") { // Wait for command prompt
			prompt = readFullLineFromPtyProcess(&kadminProc);
			printf("(kadmin) '%s'\n", prompt.ascii());
		}
		kadminProc.enableLocalEcho(false);
		kadminProc.writeLine("quit", true);
		return 1;
	}

	if (errstr) *errstr = "Internal error.  Verify that kadmin exists and can be executed.";
	return 1;	// Failure
}

int LDAPManager::unbondRealm(LDAPRealmConfig realmcfg, TQString adminUserName, const char * adminPassword, TQString adminRealm, TQString *errstr) {
	Q_UNUSED(realmcfg);

	TQCString command = "kadmin";
	QCStringList args;
	args << TQCString("-p") << TQCString(adminUserName+"@"+(adminRealm.upper()));

	TQString hoststring = "host/"+getMachineFQDN();

	TQString hostprinc = TQStringList::split(".", hoststring)[0];
	hostprinc.append("@"+(adminRealm.upper()));

	TQString prompt;
	PtyProcess kadminProc;
	kadminProc.exec(command, args);
	prompt = readFullLineFromPtyProcess(&kadminProc);
	prompt = prompt.stripWhiteSpace();
	if (prompt == "kadmin>") {
		command = TQCString("delete "+hoststring);
		kadminProc.enableLocalEcho(false);
		kadminProc.writeLine(command, true);
		do { // Discard our own input
			prompt = readFullLineFromPtyProcess(&kadminProc);
			printf("(kadmin) '%s'\n", prompt.ascii());
		} while ((prompt == TQString(command)) || (prompt == ""));
		prompt = prompt.stripWhiteSpace();
		if (prompt.endsWith(" Password:")) {
			kadminProc.enableLocalEcho(false);
			kadminProc.writeLine(adminPassword, true);
			do { // Discard our own input
				prompt = readFullLineFromPtyProcess(&kadminProc);
				printf("(kadmin) '%s'\n", prompt.ascii());
			} while (prompt == "");
			prompt = prompt.stripWhiteSpace();
		}
		if (prompt != "kadmin>") {
			if (errstr) *errstr = prompt;
			do { // Wait for command prompt
				prompt = readFullLineFromPtyProcess(&kadminProc);
				printf("(kadmin) '%s'\n", prompt.ascii());
			} while (prompt == "");
			kadminProc.enableLocalEcho(false);
			kadminProc.writeLine("quit", true);
			return 1;
		}

		// Success!
		kadminProc.enableLocalEcho(false);
		kadminProc.writeLine("quit", true);

		// Delete keys from keytab
		command = TQString("ktutil remove -p %1").arg(hoststring+"@"+adminRealm.upper());
		if (system(command) < 0) {
			printf("ERROR: Execution of \"%s\" failed!\n", command.data());
			return 1;	// Failure
		}

		// Success!
		return 0;
	}

	return 1;	// Failure
}

// ===============================================================================================================
//
// DATA CLASS CONSTRUCTORS AND DESTRUCTORS
//
// ===============================================================================================================

LDAPCredentials::LDAPCredentials() {
	// TQStrings are always initialized to TQString::null, so they don't need initialization here...
	use_tls = true;
	use_gssapi = false;
}

LDAPCredentials::~LDAPCredentials() {
	//
}

LDAPUserInfo::LDAPUserInfo() {
	// TQStrings are always initialized to TQString::null, so they don't need initialization here...
	informationValid = false;

	uid = -1;
	primary_gid = -1;
	tde_builtin_account = false;
	status = (LDAPKRB5Flags)0;
	account_created = TQDateTime::fromString("1970-01-01T00:00:00", TQt::ISODate);
	account_modified = TQDateTime::fromString("1970-01-01T00:00:00", TQt::ISODate);
	password_last_changed = TQDateTime::fromString("1970-01-01T00:00:00", TQt::ISODate);
	password_expires = false;
	password_expiration = TQDateTime::fromString("1970-01-01T00:00:00", TQt::ISODate);
	password_ages = false;
	new_password_interval = -1;
	new_password_warn_interval = -1;
	new_password_lockout_delay = -1;
	password_has_minimum_age = false;
	password_minimum_age = -1;
	maximum_ticket_lifetime = -1;
}

LDAPUserInfo::~LDAPUserInfo() {
	//
}

LDAPGroupInfo::LDAPGroupInfo() {
	// TQStrings are always initialized to TQString::null, so they don't need initialization here...
	informationValid = false;

	gid = -1;
	tde_builtin_account = false;
}

LDAPGroupInfo::~LDAPGroupInfo() {
	//
}

LDAPMachineInfo::LDAPMachineInfo() {
	// TQStrings are always initialized to TQString::null, so they don't need initialization here...
	informationValid = false;

	tde_builtin_account = false;
	status = (LDAPKRB5Flags)0;
}

LDAPMachineInfo::~LDAPMachineInfo() {
	//
}

LDAPServiceInfo::LDAPServiceInfo() {
	// TQStrings are always initialized to TQString::null, so they don't need initialization here...
	informationValid = false;

	tde_builtin_account = false;
	status = (LDAPKRB5Flags)0;
}

LDAPServiceInfo::~LDAPServiceInfo() {
	//
}

LDAPTDEBuiltinsInfo::LDAPTDEBuiltinsInfo() {
	// TQStrings are always initialized to TQString::null, so they don't need initialization here...
	informationValid = false;
}

LDAPTDEBuiltinsInfo::~LDAPTDEBuiltinsInfo() {
	//
}

LDAPMasterReplicationInfo::LDAPMasterReplicationInfo() {
	// TQStrings are always initialized to TQString::null, so they don't need initialization here...
	informationValid = false;
	enabled = false;
	// FIXME
	// Retry method and timeout should be user configurable
	// See http://www.openldap.org/doc/admin24/slapdconfig.html for syntax
	retryMethod = "5 5 300 5 600 +";
	timeout = 1;
	ignore_ssl_failure = false;
	replicate_olcGlobal = false;
}

LDAPMasterReplicationInfo::~LDAPMasterReplicationInfo() {
	//
}


LDAPMasterReplicationMapping::LDAPMasterReplicationMapping() {
	id = -1;
}

LDAPMasterReplicationMapping::~LDAPMasterReplicationMapping() {
	//
}

KerberosTicketInfo::KerberosTicketInfo() {
	// TQStrings are always initialized to TQString::null, so they don't need initialization here...
	informationValid = false;

	cacheVersion = -1;
	keyVersionNumber = -1;
	ticketSize = -1;
	flags = (KRB5TicketFlags)0;
}

KerberosTicketInfo::~KerberosTicketInfo() {
	//
}

LDAPPamConfig::LDAPPamConfig() {
	enable_cached_credentials = true;
	autocreate_user_directories_enable = true;
	autocreate_user_directories_umask = 0;
}

LDAPPamConfig::~LDAPPamConfig() {
	//
}

#include "libtdeldap.moc"
