/*
	systemconfig.cpp - A system configuration manager
	Copyright (C) 2005  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdlib.h>

#include "miscutils.h"
#include "systemconfig.h"

#include <ntqfileinfo.h>
#include <ntqheader.h>

#include <tdeapplication.h>
#include <kiconloader.h>
#include <tdelistview.h>
#include <tdelocale.h>

// http://bugs.debian.org/cgi-bin/bugreport.cgi?bug=417301
#include <cstdlib>

// private

int SystemConfig::_canShutDown = -1;

// public

bool SystemConfig::canShutDown()
{
/*
	if (_canShutDown == -1)
	{
		TQCString XDM_MANAGED = ::getenv("XDM_MANAGED");

		if (XDM_MANAGED.contains("maysd") || XDM_MANAGED.contains("classic"))
			_canShutDown = 1;
		else
			_canShutDown = 0;
	}
*/
	_canShutDown = 1;

	return _canShutDown == 1;
}

void SystemConfig::check(TQWidget *parent)
{
	SystemConfig *systemConfig = new SystemConfig(parent);
	systemConfig->exec();
	delete systemConfig;
}

// private

SystemConfig::SystemConfig(TQWidget *parent)
	: KDialogBase(
		parent,
		"SystemConfig", // name
		true, // modal
		i18n("System Configuration"),
		Close,
		Close // default button
	),
	_problems(0)
{
	_messages = new TDEListView(this, "TDEListView::_messages");
	_messages->setAllColumnsShowFocus(true);
	_messages->setItemMargin(5);
	_messages->setSorting(-1); // no sort
	_messages->addColumn("");
	_messages->addColumn(i18n("Message"));
	_messages->header()->setClickEnabled(false);
	connect(_messages, SIGNAL(executed(TQListViewItem *)), SLOT(slotExecuted(TQListViewItem *)));
	connect(_messages, SIGNAL(spacePressed(TQListViewItem *)), SLOT(slotExecuted(TQListViewItem *)));
	setMainWidget(_messages);

	_shutdownAllowItem = add(Info, i18n("Tip: Click here if you have problem with the \"/sbin/shutdown\" command."));

	checkFile("/sbin/poweroff");
	checkFile("/sbin/reboot");
	checkFile("/sbin/shutdown");
	checkTDM();
	checkTDE();

	if (_problems == 0)
		add(OK, i18n("No problems were found."));
}

TDEListViewItem *SystemConfig::add(const Type type, const TQString &message)
{
	TDEListViewItem *item = new TDEListViewItem(_messages);
	item->setMultiLinesEnabled(true);
	switch (type)
	{
		case Info:
			item->setPixmap(0, SmallIcon("messagebox_info"));
			break;
		case OK:
			item->setPixmap(0, SmallIcon("button_ok"));
			break;
		case Warning:
			_problems++;
			item->setPixmap(0, SmallIcon("messagebox_warning"));
			break;
	}
	item->setText(1, message);

	return item;
}

void SystemConfig::checkFile(const TQString &file)
{
	TQFileInfo program(file);

	if (!program.exists())
	{
		add(Warning, i18n("Program \"%1\" was not found!").arg(file));

		return;
	}

	if (!program.isExecutable())
	{
		add(Warning, i18n("No permissions to execute \"%1\".").arg(file));
	}
}

void SystemConfig::checkTDE()
{
	TQCString TDE_FULL_SESSION = ::getenv("TDE_FULL_SESSION");
	if (TDE_FULL_SESSION != "true")
	{
		add(Warning, i18n("It seems that this is not a TDE full session.\nKShutDown was designed to work with TDE.\nHowever, you can customize Actions in the KShutDown settings dialog\n(Settings -> Configure KShutDown... -> Actions)."));
	}
}

/* TODO: 2.0: TDM configurator
From old FAQ:
- Do it as root!
- Make sure you first create backup of the "/etc/sysconfig/desktop" file
- Remove all "GNOME" and "GDM" lines from "/etc/sysconfig/desktop" file
- Add new line: DISPLAYMANAGER="TDE"
- Reboot system and login again
*/
void SystemConfig::checkTDM()
{
	if (!canShutDown())
	{
// TODO: 2.0: auto configuration
		add(Info, i18n("Tip: You can customize Actions to work with GDM.\n(Settings -> Configure KShutDown... -> Actions)"));
		_tdmNotDetected = add(Warning, i18n("TDE Display Manager is not running,\nor the shut down/reboot function is disabled.\n\nClick here to configure TDM."));
	}
}

// private slots

void SystemConfig::slotExecuted(TQListViewItem *item)
{
	if (!item)
		return;

	if (item == _tdmNotDetected)
	{
		MiscUtils::runShellCommand("tdesu -c \"tdecmshell tdm\" -i \"exit\"");

		return;
	}

	if (item == _shutdownAllowItem)
	{
		kapp->invokeBrowser("http://doc.gwos.org/index.php/NonRootShutdown");

		return;
	}
}
