/*
	miscutils.h - Misc functions
	Copyright (C) 2004  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef __MISCUTILS_H__
#define __MISCUTILS_H__

#include <ntqdatetime.h>

#include <kprocess.h>

class TDEAction;
class KPushButton;

/** @short Misc functions. */
class MiscUtils
{
public:
	/**
	 * Closes the CD-ROM tray (if this option is enabled).
	 */
	static void closeCDTray();

	/**
	 * Displays a custom message.
	 * @param text A text to display
	 * @param testCommand A command to execute (used internally in settings dialog to test entered command)
	 */
	static void customMessage(const TQString &text, const TQString &testCommand = TQString::null);

	/**
	 * Converts seconds to time string.
	 * @param secs A number of seconds to convert
	 * @param format A date/time format
	 * @return Converted seconds or ">24", if @p secs >= 86400
	 */
	static TQString formatDateTime(const int secs, const TQString &format = TQString::null);

	/**
	 * Converts seconds to time string.
	 * @param dt A date/time to convert
	 * @return Converted seconds or ">24", if @p secs >= 86400
	 */
	static TQString formatDateTime(const TQDateTime &dt);

	inline static TQString HTML(const TQString &text) {
		return ("<qt>" + text + "</qt>");
	}

	inline static TQString HTML(TQString &text) {
		text.prepend("<qt>");
		text.append("</qt>");

		return text;
	}

	/**
	 * Returns @c true if function (@p key) is restricted (KDE Kiosk).
	 * The key value is read from the configuration file
	 * (usually /usr/share/config/kshutdownrc).
	 * See Handbook (F1) for details.
	 */
	static bool isRestricted(const TQString &key);

	/**
	 * Visual and sound notifications.
	 * @param secs A number of seconds to action
	 */
	static void notifyUser(const int secs);

	static void passiveMessage(const TQString &text, TQWidget *parent = 0);

	static void plug(const TDEAction *action, KPushButton *pushButton);

	/**
	 * Executes a command.
	 * @param command A command to execute
	 * @return @c true if successfull; otherwise @c false
	 */
	static bool runCommand(const TQString &command);

	/**
	 * Executes a shell command before the specified action.
	 * @param configEntry A configuration entry (e.g. <i>"Shut Down"</i>)
	 */
	static void runCommandBeforeAction(const TQString &configEntry);

	/**
	 * Executes a shell command.
	 * @param command A shell command to execute
	 * @param mode A run mode (@c TDEProcess::DontCare by default)
	 * @param pause A timeout in seconds (no timeout by default)
	 * @return @c true if successfull; otherwise @c false
	 */
	static bool runShellCommand(const TQString &command, const TDEProcess::RunMode mode = TDEProcess::DontCare, const int pause = -1);

	static void setAutostart(const bool yes);

	/**
	 * Sets "tool tip text" and "what's this text" of @p widget to @p value.
	 */
	static void setHint(TQWidget *widget, const TQString &value);

	/**
	 * Displays a "KRun" error message in a passive message.
	 * @param command A command that failed
	 */
	static void showRunErrorMessage(const TQString &command);

	/**
	 * Displays an information if KShutDown is in the <b>Test Mode</b>.
	 * @param message A message to display
	 */
	static void showTestMessage(const TQString &message);

private:
	static void notifyUser(const TQString &name, const TQString &text);
};

#endif // __MISCUTILS_H__
