/*
	links.cpp - A link creator/remover
	Copyright (C) 2004  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "actions.h"
#include "links.h"
#include "miscutils.h"

#include <ntqcombobox.h>
#include <ntqfile.h>
#include <ntqlabel.h>

#include <tdeapplication.h>
#include <kdesktopfile.h>
#include <tdeglobalsettings.h>
#include <kiconloader.h>
#include <tdelocale.h>
#include <tdemessagebox.h>
#include <kpushbutton.h>
#include <kstandarddirs.h>

// public

Links::Links(TQWidget *parent)
	: TQVBox(parent, "Links")
{
	setSpacing(5);
	int alignment = AlignVCenter;
	alignment |= kapp->reverseLayout() ? AlignLeft : AlignRight;


	// location
	TQHBox *w_location = new TQHBox(this);
	w_location->setSpacing(5);
	TQLabel *l_location = new TQLabel(i18n("Location where to create the link:"), w_location);
	l_location->setAlignment(alignment);
	cb_location = new TQComboBox(w_location, "TQComboBox::cb_location");
	cb_location->setFocusPolicy(StrongFocus);
	cb_location->insertItem(SmallIcon("desktop"), i18n("Desktop"));
	cb_location->insertItem(SmallIcon("kmenu"), i18n("K Menu"));
	l_location->setBuddy(cb_location);
	connect(cb_location, SIGNAL(activated(int)), SLOT(slotActivated(int)));

	// type
	TQHBox *w_type = new TQHBox(this);
	w_type->setSpacing(5);
	TQLabel *l_type = new TQLabel(i18n("Type of the link:"), w_type);
	l_type->setAlignment(alignment);
	cb_type = new TQComboBox(w_type, "TQComboBox::cb_type");
	cb_type->setFocusPolicy(StrongFocus);
	cb_type->insertItem(kapp->miniIcon(), "KShutDown");

	// NOTE: slotAddRemoveLink()
	cb_type->insertItem(SmallIcon("system-log-out"), i18n("Standard Logout Dialog"));

	cb_type->insertItem(ks_actions->getIcon(Action::ShutDown), ks_actions->getName(Action::ShutDown));
	cb_type->insertItem(ks_actions->getIcon(Action::Reboot), ks_actions->getName(Action::Reboot));
	cb_type->insertItem(ks_actions->getIcon(Action::LockScreen), ks_actions->getName(Action::LockScreen));
	cb_type->insertItem(ks_actions->getIcon(Action::Logout), ks_actions->getName(Action::Logout));
	l_type->setBuddy(cb_type);
	connect(cb_type, SIGNAL(activated(int)), SLOT(slotActivated(int)));

	// add/remove link
	b_addRemoveLink = new KPushButton(this, "KPushButton::b_addRemoveLink");
	connect(b_addRemoveLink, SIGNAL(clicked()), SLOT(slotAddRemoveLink()));

	updateAddRemoveButton();
}

void Links::createLink(const TQString &path, const TQString &command, const TQString &icon, const TQString &name, const TQString &comment)
{
	if (path.isNull())
		return;

	KDesktopFile *df = new KDesktopFile(path);
	df->setGroup("Desktop Entry");
	df->writeEntry("Comment", comment);
	df->writeEntry("Encoding", "UTF-8");
	df->writeEntry("Exec", command);
	df->writeEntry("GenericName", i18n("System Shut Down Utility"));
	df->writeEntry("Icon", icon);
	df->writeEntry("Name", name);
	df->writeEntry("StartupNotify", "false");
	df->writeEntry("Type", "Application");
	delete df;

	if (!TQFile::exists(path))
	{
		KMessageBox::error(
			0,
			MiscUtils::HTML(i18n("Could not create file <b>%1</b>!").arg(path))
		);
	}
}

void Links::removeLink(const TQString &path)
{
	if (TQFile::exists(path) && !TQFile::remove(path))
	{
		KMessageBox::error(
			0,
			MiscUtils::HTML(i18n("Could not remove file <b>%1</b>!").arg(path))
		);
	}
}

// private

TQString Links::getCurrentLocationPath() const
{
	TQString path;
	switch (cb_location->currentItem())
	{
		case 0:
			path = TDEGlobalSettings::desktopPath() + "/";
			break;
		case 1:
			path = locateLocal("apps", "") + "/";
			break;
		default:
			return TQString::null;
	}
	switch (cb_type->currentItem())
	{
		case 0: return path += "kshutdown-classic.desktop";
		case 1: return path += "kshutdown-standard.desktop";
		case 2: return path += "kshutdown-shutdown.desktop";
		case 3: return path += "kshutdown-reboot.desktop";
		case 4: return path += "kshutdown-lock.desktop";
		case 5: return path += "kshutdown-logout.desktop";
		default: return TQString::null;
	}
}

TQString Links::getCurrentTypeCommand() const
{
	switch (cb_type->currentItem())
	{
		case 0: return "kshutdown";
		case 1: return "kshutdown --standard";
		case 2: return "kshutdown --confirm --shutdown";
		case 3: return "kshutdown --confirm --reboot";
		case 4: return "kshutdown --confirm --lock";
		case 5: return "kshutdown --confirm --logout";
		default: return TQString::null;
	}
}

TQString Links::getCurrentTypeIcon() const
{
	switch (cb_type->currentItem())
	{
		case 0: return "kshutdown";
		case 1: return "system-log-out";
		// sync. with Action::getIcon
		case 2: return "system-log-out";
		case 3: return "reload";
		case 4: return "system-lock-screen";
		case 5: return "edit-undo";
		default: return TQString::null;
	}
}

void Links::updateAddRemoveButton() {
	if (TQFile::exists(getCurrentLocationPath())) {
		b_addRemoveLink->setIconSet(SmallIcon("edit-delete"));
		b_addRemoveLink->setText(i18n("Remove Link"));
	}
	else {
		b_addRemoveLink->setIconSet(SmallIcon("document-new"));
		b_addRemoveLink->setText(i18n("Add Link"));
	}
}

// private slots

void Links::slotActivated(int/* index*/) {
	updateAddRemoveButton();
}

void Links::slotAddRemoveLink() {
	if (TQFile::exists(getCurrentLocationPath())) {
		removeLink(getCurrentLocationPath());
	}
	else {
		createLink(
			getCurrentLocationPath(),
			getCurrentTypeCommand(),
			getCurrentTypeIcon(),
			(cb_type->currentItem() == 1) ? i18n("Logout") : cb_type->currentText(),
			cb_type->currentText()
		);
	}
	updateAddRemoveButton();
}
