/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
*/

/*
  Copyright (C) 2007 Shintaro Matsuoka <shin@shoegazed.org>
*/

#include "dcccommon.h"
#include "channel.h"
#include "config/preferences.h"
#include "server.h"

#include <arpa/inet.h>

#include <tqhostaddress.h>

#include <tdelocale.h>
#include <kresolver.h>
#include <kserversocket.h>


TQString DccCommon::textIpToNumericalIp( const TQString& ipString )
{
    TQHostAddress ip;
    ip.setAddress( ipString );

    return TQString::number( ip.ip4Addr() );
}

TQString DccCommon::numericalIpToTextIp( const TQString& numericalIp )
{
    TQHostAddress ip;
    ip.setAddress( numericalIp.toULong() );

    return ip.toString();
}

TQString DccCommon::getOwnIp( Server* server )
{
    TQString ownIp;
    int methodId = Preferences::dccMethodToGetOwnIp();

    if ( methodId == 1 && server )
    {
        // by the WELCOME message or the USERHOST message from the server
        ownIp = server->getOwnIpByServerMessage();
    }
    else if ( methodId == 2 && !Preferences::dccSpecificOwnIp().isEmpty() )
    {
        // manual
        KNetwork::KResolverResults res = KNetwork::KResolver::resolve(Preferences::dccSpecificOwnIp(), "");
        if(res.error() == KResolver::NoError && res.size() > 0)
        {
            ownIp = res.first().address().nodeName();
        }
    }

    // fallback or methodId == 3 (network interface)
    if ( ownIp.isEmpty() && server )
    {
        ownIp = server->getOwnIpByNetworkInterface();
    }

    kdDebug() << "DccCommon::getOwnIp(): " << ownIp << endl;
    return ownIp;
}

KNetwork::TDEServerSocket* DccCommon::createServerSocketAndListen( TQObject* parent, TQString* failedReason, int minPort, int maxPort )
{
    KNetwork::TDEServerSocket* socket = new KNetwork::TDEServerSocket( parent );
    socket->setFamily( KNetwork::KResolver::InetFamily );

    if ( minPort > 0 && maxPort >= minPort )  // ports are configured manually
    {
        // set port
        bool found = false;                       // whether succeeded to set port
        for ( int port = minPort; port <= maxPort ; ++port )
        {
            socket->setAddress( TQString::number( port ) );
            bool success = socket->listen();
            if ( ( found = ( success && socket->error() == KNetwork::TDESocketBase::NoError ) ) )
                break;
            socket->close();
        }
        if ( !found )
        {
            if ( failedReason )
                *failedReason = i18n( "No vacant port" );
            delete socket;
            return 0;
        }
    }
    else
    {
        // Let the operating system choose a port
        socket->setAddress( "0" );
        if ( !socket->listen() )
        {
            if ( failedReason )
                *failedReason = i18n( "Could not open a socket" );
            delete socket;
            return 0;
        }
    }

    return socket;
}

int DccCommon::getServerSocketPort( KNetwork::TDEServerSocket* serverSocket )
{
    KNetwork::TDESocketAddress ipAddr = serverSocket->localAddress();
    const struct sockaddr_in* socketAddress = (sockaddr_in*)ipAddr.address();
    return ntohs( socketAddress->sin_port );
}
