/***************************************************************************
                          kmymoneycombo.cpp  -  description
                             -------------------
    begin                : Mon Mar 12 2007
    copyright            : (C) 2007 by Thomas Baumgart
    email                : ipwizard@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// ----------------------------------------------------------------------------
// QT Includes

#include <tqrect.h>
#include <tqstyle.h>
#include <tqpainter.h>
#include <tqapplication.h>

// ----------------------------------------------------------------------------
// KDE Includes

#include <tdelocale.h>
#include <tdelistview.h>
#include <kdebug.h>
#include <tdeconfig.h>

// ----------------------------------------------------------------------------
// Project Includes

#include "kmymoneycombo.h"
#include "kmymoneyselector.h"
#include <kmymoney/kmymoneycompletion.h>
#include <kmymoney/kmymoneylineedit.h>
#include <kmymoney/mymoneysplit.h>
#include <kmymoney/registeritem.h>
#include <kmymoney/mymoneyscheduled.h>
#include "../kmymoneyutils.h"

KMyMoneyCombo::KMyMoneyCombo(TQWidget *w, const char *name) :
  KComboBox(w, name),
  m_completion(0),
  m_edit(0),
  m_canCreateObjects(false),
  m_inFocusOutEvent(false)
{
}

KMyMoneyCombo::KMyMoneyCombo(bool rw, TQWidget *w, const char *name) :
  KComboBox(rw, w, name),
  m_completion(0),
  m_edit(0),
  m_canCreateObjects(false),
  m_inFocusOutEvent(false)
{
  if(rw) {
    m_edit = new kMyMoneyLineEdit(this, "combo edit");
    setLineEdit(m_edit);
  }
}

void KMyMoneyCombo::setCurrentTextById(const TQString& id)
{
    setCurrentText();
    if(!id.isEmpty()) {
      TQListViewItem* item = selector()->item(id);
      if(item)
        setCurrentText(item->text(0));
    }
}

void KMyMoneyCombo::slotItemSelected(const TQString& id)
{
  if(editable()) {
    bool blocked = signalsBlocked();
    blockSignals(true);
    setCurrentTextById(id);
    blockSignals(blocked);
  }

  m_completion->hide();

  if(m_id != id) {
    m_id = id;
    emit itemSelected(id);
  }
}

void KMyMoneyCombo::setEditable(bool y)
{
  if(y == editable())
    return;

  KComboBox::setEditable(y);

  // make sure we use our own line edit style
  if(y) {
    m_edit = new kMyMoneyLineEdit(this, "combo edit");
    setLineEdit(m_edit);
    m_edit->setPaletteBackgroundColor(paletteBackgroundColor());

  } else {
    m_edit = 0;
  }
}

void KMyMoneyCombo::setHint(const TQString& hint) const
{
  if(m_edit)
    m_edit->setHint(hint);
}

void KMyMoneyCombo::paintEvent(TQPaintEvent* ev)
{
  KComboBox::paintEvent(ev);

  // if we don't have an edit field, we need to paint the text onto the button
  if(!m_edit) {
    if(m_completion) {
      TQStringList list;
      selector()->selectedItems(list);
      if(!list.isEmpty()) {
        TQString str = selector()->item(list[0])->text(0);
        // we only paint, if the text is longer than 1 char. Assumption
        // is that length 1 is the blank case so no need to do painting
        if(str.length() > 1) {
          TQPainter p( this );
          const TQColorGroup & g = colorGroup();
          p.setPen(g.text());

          TQRect re = style().querySubControlMetrics( TQStyle::CC_ComboBox, this,
                                                    TQStyle::SC_ComboBoxEditField );
          re = TQStyle::visualRect(re, this);
          p.setClipRect( re );
          p.save();
          p.setFont(font());
          TQFontMetrics fm(font());
          int x = re.x(), y = re.y() + fm.ascent();
          p.drawText( x, y, str );
          p.restore();
        }
      }
    }
  }
}

void KMyMoneyCombo::setPaletteBackgroundColor(const TQColor& color)
{
  KComboBox::setPaletteBackgroundColor(color);
  if(m_edit) {
    m_edit->setPaletteBackgroundColor(color);
  }
}

void KMyMoneyCombo::mousePressEvent(TQMouseEvent *e)
{
  // mostly copied from TQCombo::mousePressEvent() and adjusted for our needs
  if(e->button() != Qt::LeftButton)
    return;

  if(((!editable() || isInArrowArea(mapToGlobal(e->pos()))) && selector()->itemList().count()) && !m_completion->isVisible()) {
    m_completion->show();
  }

  if(m_timer.isActive()) {
    m_timer.stop();
    m_completion->slotMakeCompletion("");
  } else {
    TDEConfig config( "kcminputrc", true );
    config.setGroup("KDE");
    m_timer.start(config.readNumEntry("DoubleClickInterval", 400), true);
  }
}

bool KMyMoneyCombo::isInArrowArea(const TQPoint& pos) const
{
  TQRect arrowRect = style().querySubControlMetrics( TQStyle::CC_ComboBox, this,
                                                    TQStyle::SC_ComboBoxArrow);
  arrowRect = TQStyle::visualRect(arrowRect, this);

  // Correction for motif style, where arrow is smaller
  // and thus has a rect that doesn't fit the button.
  arrowRect.setHeight( TQMAX(  height() - (2 * arrowRect.y()), arrowRect.height() ) );

  // if the button is not editable, it covers the whole widget
  if(!editable())
    arrowRect = rect();

  return arrowRect.contains(mapFromGlobal(pos));
}

void KMyMoneyCombo::keyPressEvent(TQKeyEvent* e)
{
  if((e->key() == Key_F4 && e->state() == 0 ) ||
     (e->key() == Key_Down && (e->state() & AltButton)) ||
     (!editable() && e->key() == Key_Space)) {
    // if we have at least one item in the list, we open the dropdown
    if(selector()->listView()->firstChild())
      m_completion->show();
    e->ignore();
    return;
  }
  KComboBox::keyPressEvent(e);
}

void KMyMoneyCombo::connectNotify(const char* signal)
{
  if(signal && !strcmp(signal, TQT_SIGNAL(createItem(const TQString&,TQString&)))) {
    m_canCreateObjects = true;
  }
}

void KMyMoneyCombo::disconnectNotify(const char* signal)
{
  if(signal && !strcmp(signal, TQT_SIGNAL(createItem(const TQString&,TQString&)))) {
    m_canCreateObjects = false;
  }
}

void KMyMoneyCombo::focusOutEvent(TQFocusEvent* e)
{
  if(m_inFocusOutEvent) {
    KComboBox::focusOutEvent(e);
    return;
  }

  m_inFocusOutEvent = true;
  if(editable() && !currentText().isEmpty()) {
    if(m_canCreateObjects) {
      if(!m_completion->selector()->contains(currentText())) {
        TQString id;
        // annouce that we go into a possible dialog to create an object
        // This can be used by upstream widgets to disable filters etc.
        emit objectCreation(true);

        emit createItem(currentText(), id);

        // Announce that we return from object creation
        emit objectCreation(false);

        // update the field to a possibly created object
        m_id = id;
        setCurrentTextById(id);

        // make sure the completion does not show through
        m_completion->hide();
      }

    // else if we cannot create objects, and the current text is not
    // in the list, then we clear the text and the selection.
    } else if(!m_completion->selector()->contains(currentText())) {
      setCurrentText(TQString());
    }
  }

  KComboBox::focusOutEvent(e);

  // force update of hint and id if there is no text in the widget
  if(editable() && currentText().isEmpty()) {
    TQString id = m_id;
    m_id = TQString();
    if(!id.isEmpty())
      emit itemSelected(m_id);
    repaint();
  }
  m_inFocusOutEvent = false;
}

KMyMoneySelector* KMyMoneyCombo::selector(void) const
{
  return m_completion->selector();
}

kMyMoneyCompletion* KMyMoneyCombo::completion(void) const
{
  return m_completion;
}

void KMyMoneyCombo::selectedItem(TQString& id) const
{
  id = m_id;
}

void KMyMoneyCombo::selectedItems(TQStringList& list) const
{
  if(lineEdit() && lineEdit()->text().length() == 0) {
    list.clear();
  } else {
    m_completion->selector()->selectedItems(list);
  }
}

void KMyMoneyCombo::setSelectedItem(const TQString& id)
{
  m_completion->selector()->setSelected(id, true);
  blockSignals(true);
  slotItemSelected(id);
  blockSignals(false);
  update();
}

TQSize KMyMoneyCombo::sizeHint() const
{
  return KComboBox::sizeHint();

  // I wanted to use the code below to adjust the size of the combo box
  // according to the largest item in the selector list. Apparently that
  // does not work too well in the enter and edit schedule dialog for
  // the category combo box. So we just use the standard implementation for now.
#if 0
  constPolish();
  int i, w;
  TQFontMetrics fm = fontMetrics();

  int maxW = count() ? 18 : 7 * fm.width(TQChar('x')) + 18;
  int maxH = TQMAX( fm.lineSpacing(), 14 ) + 2;

  w = selector()->optimizedWidth();
  if ( w > maxW )
    maxW = w;

  TQSize sizeHint = (style().sizeFromContents(TQStyle::CT_ComboBox, this,
                 TQSize(maxW, maxH)).
      expandedTo(TQApplication::globalStrut()));

  return sizeHint;
#endif
}



KMyMoneyReconcileCombo::KMyMoneyReconcileCombo(TQWidget* w, const char* name) :
  KMyMoneyCombo(false, w, name)
{
  m_completion = new kMyMoneyCompletion(this, 0);
  // connect(m_completion, TQT_SIGNAL(itemSelected(const TQString&)), this, TQT_SIGNAL(itemSelected(const TQString&)));

  // add the items in reverse order of appearance (see KMyMoneySelector::newItem() for details)
  // selector()->newTopItem(i18n("Frozen"), TQString(), "F");
  selector()->newTopItem(i18n("Reconciled"), TQString(), "R");
  selector()->newTopItem(i18n("Cleared"), TQString(), "C");
  selector()->newTopItem(i18n("Not reconciled"), TQString(), " ");
  selector()->newTopItem(" ", TQString(), "U");

  connect(m_completion, TQT_SIGNAL(itemSelected(const TQString&)), this, TQT_SLOT(slotItemSelected(const TQString&)));
  connect(this, TQT_SIGNAL(itemSelected(const TQString&)), this, TQT_SLOT(slotSetState(const TQString&)));
}

void KMyMoneyReconcileCombo::slotSetState(const TQString& state)
{
  setSelectedItem(state);
}

void KMyMoneyReconcileCombo::removeDontCare(void)
{
  selector()->removeItem("U");
}

void KMyMoneyReconcileCombo::setState(MyMoneySplit::reconcileFlagE state)
{
  TQString id;
  switch(state) {
    case MyMoneySplit::NotReconciled:
      id = " ";
      break;
    case MyMoneySplit::Cleared:
      id = "C";
      break;
    case MyMoneySplit::Reconciled:
      id = "R";
      break;
    case MyMoneySplit::Frozen:
      id = "F";
      break;
    case MyMoneySplit::Unknown:
      id = "U";
      break;
    default:
      kdDebug(2) << "Unknown reconcile state '" << state << "' in KMyMoneyComboReconcile::setState()\n";
      break;
  }
  setSelectedItem(id);
}

MyMoneySplit::reconcileFlagE KMyMoneyReconcileCombo::state(void) const
{
  MyMoneySplit::reconcileFlagE state = MyMoneySplit::NotReconciled;

  TQStringList list;
  selector()->selectedItems(list);
  if(!list.isEmpty()) {
    if(list[0] == "C")
      state = MyMoneySplit::Cleared;
    if(list[0] == "R")
      state = MyMoneySplit::Reconciled;
    if(list[0] == "F")
      state = MyMoneySplit::Frozen;
    if(list[0] == "U")
      state = MyMoneySplit::Unknown;
  }
  return state;
}


KMyMoneyComboAction::KMyMoneyComboAction(TQWidget* w, const char* name) :
  KMyMoneyCombo(false, w, name)
{
  m_completion = new kMyMoneyCompletion(this, 0);
  TQString num;
  // add the items in reverse order of appearance (see KMyMoneySelector::newItem() for details)
  selector()->newTopItem(i18n("ATM"), TQString(), num.setNum(KMyMoneyRegister::ActionAtm));
  selector()->newTopItem(i18n("Withdrawal"), TQString(), num.setNum(KMyMoneyRegister::ActionWithdrawal));
  selector()->newTopItem(i18n("Transfer"), TQString(), num.setNum(KMyMoneyRegister::ActionTransfer));
  selector()->newTopItem(i18n("Deposit"), TQString(), num.setNum(KMyMoneyRegister::ActionDeposit));
  selector()->newTopItem(i18n("Cheque"), TQString(), num.setNum(KMyMoneyRegister::ActionCheck));
  connect(m_completion, TQT_SIGNAL(itemSelected(const TQString&)), this, TQT_SLOT(slotItemSelected(const TQString&)));
  connect(this, TQT_SIGNAL(itemSelected(const TQString&)), this, TQT_SLOT(slotSetAction(const TQString&)));
}

void KMyMoneyComboAction::protectItem(int id, bool protect)
{
  TQString num;
  selector()->protectItem(num.setNum(id), protect);
}

void KMyMoneyComboAction::slotSetAction(const TQString& act)
{
  setSelectedItem(act);
  update();
  emit actionSelected(action());
}

void KMyMoneyComboAction::setAction(int action)
{
  if(action < 0 || action > 5) {
    kdDebug(2) << "KMyMoneyComboAction::slotSetAction(" << action << ") invalid. Replaced with 2\n";
    action = 2;
  }
  TQString act;
  act.setNum(action);
  setSelectedItem(act);
}

int KMyMoneyComboAction::action(void) const
{
  TQStringList list;
  selector()->selectedItems(list);
  if(!list.isEmpty()) {
    return list[0].toInt();
  }
  kdDebug(2) << "KMyMoneyComboAction::action(void): unknown selection\n";
  return 0;
}

KMyMoneyCashFlowCombo::KMyMoneyCashFlowCombo(TQWidget* w, const char* name, MyMoneyAccount::accountTypeE accountType) :
  KMyMoneyCombo(false, w, name)
{
  m_completion = new kMyMoneyCompletion(this, 0);
  TQString num;
  // add the items in reverse order of appearance (see KMyMoneySelector::newItem() for details)
  if(accountType == MyMoneyAccount::Income || accountType == MyMoneyAccount::Expense) {
    // this is used for income/expense accounts to just show the reverse sense
    selector()->newTopItem(i18n("Activity for expense categories", "Paid"), TQString(), num.setNum(KMyMoneyRegister::Deposit));
    selector()->newTopItem(i18n("Activity for income categories", "Received"), TQString(), num.setNum(KMyMoneyRegister::Payment));
  } else {
    selector()->newTopItem(i18n("From"), TQString(), num.setNum(KMyMoneyRegister::Deposit));
    selector()->newTopItem(i18n("Pay to"), TQString(), num.setNum(KMyMoneyRegister::Payment));
  }
  selector()->newTopItem(" ", TQString(), num.setNum(KMyMoneyRegister::Unknown));
  connect(m_completion, TQT_SIGNAL(itemSelected(const TQString&)), this, TQT_SLOT(slotItemSelected(const TQString&)));
  connect(this, TQT_SIGNAL(itemSelected(const TQString&)), this, TQT_SLOT(slotSetDirection(const TQString&)));
}

void KMyMoneyCashFlowCombo::setDirection(KMyMoneyRegister::CashFlowDirection dir)
{
  m_dir = dir;
  TQString num;
  setSelectedItem(num.setNum(dir));
}

void KMyMoneyCashFlowCombo::slotSetDirection(const TQString& id)
{
  TQString num;
  for(int i = KMyMoneyRegister::Deposit; i <= KMyMoneyRegister::Unknown; ++i) {
    num.setNum(i);
    if(num == id) {
      m_dir = static_cast<KMyMoneyRegister::CashFlowDirection>(i);
      break;
    }
  }
  emit directionSelected(m_dir);
  update();
}

void KMyMoneyCashFlowCombo::removeDontCare(void)
{
  TQString num;
  selector()->removeItem(num.setNum(KMyMoneyRegister::Unknown));
}


KMyMoneyActivityCombo::KMyMoneyActivityCombo(TQWidget* w, const char* name) :
  KMyMoneyCombo(false, w, name),
  m_activity(MyMoneySplit::UnknownTransactionType)
{
  m_completion = new kMyMoneyCompletion(this, 0);
  TQString num;
  // add the items in reverse order of appearance (see KMyMoneySelector::newItem() for details)
  selector()->newTopItem(i18n("Split shares"), TQString(), num.setNum(MyMoneySplit::SplitShares));
  selector()->newTopItem(i18n("Remove shares"), TQString(), num.setNum(MyMoneySplit::RemoveShares));
  selector()->newTopItem(i18n("Add shares"), TQString(), num.setNum(MyMoneySplit::AddShares));
  selector()->newTopItem(i18n("Yield"), TQString(), num.setNum(MyMoneySplit::Yield));
  selector()->newTopItem(i18n("Reinvest dividend"), TQString(), num.setNum(MyMoneySplit::ReinvestDividend));
  selector()->newTopItem(i18n("Dividend"), TQString(), num.setNum(MyMoneySplit::Dividend));
  selector()->newTopItem(i18n("Sell shares"), TQString(), num.setNum(MyMoneySplit::SellShares));
  selector()->newTopItem(i18n("Buy shares"), TQString(), num.setNum(MyMoneySplit::BuyShares));

  connect(m_completion, TQT_SIGNAL(itemSelected(const TQString&)), this, TQT_SLOT(slotItemSelected(const TQString&)));
  connect(this, TQT_SIGNAL(itemSelected(const TQString&)), this, TQT_SLOT(slotSetActivity(const TQString&)));
}

void KMyMoneyActivityCombo::setActivity(MyMoneySplit::investTransactionTypeE activity)
{
  m_activity = activity;
  TQString num;
  setSelectedItem(num.setNum(activity));
}

void KMyMoneyActivityCombo::slotSetActivity(const TQString& id)
{
  TQString num;
  for(int i = MyMoneySplit::BuyShares; i <= MyMoneySplit::SplitShares; ++i) {
    num.setNum(i);
    if(num == id) {
      m_activity = static_cast<MyMoneySplit::investTransactionTypeE>(i);
      break;
    }
  }
  emit activitySelected(m_activity);
  update();
}

KMyMoneyPayeeCombo::KMyMoneyPayeeCombo(TQWidget* parent, const char * name) :
  KMyMoneyCombo(true, parent, name)
{
  m_completion = new kMyMoneyCompletion(this);

  // set to ascending sort
  selector()->listView()->setSorting(0);

  connect(m_completion, TQT_SIGNAL(itemSelected(const TQString&)), this, TQT_SLOT(slotItemSelected(const TQString&)));
  connect(this, TQT_SIGNAL(textChanged(const TQString&)), m_completion, TQT_SLOT(slotMakeCompletion(const TQString&)));
}

void KMyMoneyPayeeCombo::loadPayees(const TQValueList<MyMoneyPayee>& list)
{
  selector()->listView()->clear();
  TQValueList<MyMoneyPayee>::const_iterator it;
  for(it = list.begin(); it != list.end(); ++it) {
    selector()->newTopItem((*it).name(), TQString(), (*it).id());
  }
}


class KMyMoneyGeneralCombo::Private {
public:
  TQMap<TQString, int> m_strings;
  void insertItem(const TQString& s, int id) { m_strings[s] = id; }

  int itemId(const TQString& s) const {
    TQMap<TQString, int>::const_iterator it;
    it = m_strings.find(s);
    if(it != m_strings.end())
      return *it;
    return -1;
  }

  const TQString& itemText(int id) {
    TQMap<TQString, int>::const_iterator it;
    for(it = m_strings.begin(); it != m_strings.end(); ++it) {
      if(*it == id) {
        return it.key();
      }
    }
    return TQString::null;
  }
};

KMyMoneyGeneralCombo::KMyMoneyGeneralCombo(TQWidget* w, const char* name) :
  KComboBox(w, name),
  d(new Private)
{
  connect(this, TQT_SIGNAL(highlighted(int)), this, TQT_SLOT(slotChangeItem(int)));
}

KMyMoneyGeneralCombo::~KMyMoneyGeneralCombo()
{
  delete d;
}

void KMyMoneyGeneralCombo::setItem(int id)
{
  setCurrentItem(id);
}

int KMyMoneyGeneralCombo::item(void) const
{
  return currentItem();
}

void KMyMoneyGeneralCombo::setCurrentItem(int id)
{
  const TQString& txt = d->itemText(id);
  for(int idx = 0; idx < count(); ++idx) {
    if(txt == text(idx)) {
      KComboBox::setCurrentItem(idx);
      break;
    }
  }
}

int KMyMoneyGeneralCombo::currentItem(void) const
{
  return d->itemId(currentText());
}

void KMyMoneyGeneralCombo::clear(void)
{
  d->m_strings.clear();
  KComboBox::clear();
}

void KMyMoneyGeneralCombo::insertItem(const TQString& txt, int id, int idx)
{
  d->insertItem(txt, id);
  KComboBox::insertItem(txt, idx);
}

void KMyMoneyGeneralCombo::removeItem(int id)
{
  const TQString& txt = d->itemText(id);
  for(int idx = 0; idx < count(); ++idx) {
    if(txt == text(idx)) {
      KComboBox::removeItem(idx);
      break;
    }
  }
}

void KMyMoneyGeneralCombo::slotChangeItem(int idx)
{
  emit itemSelected(d->itemId(text(idx)));
}

KMyMoneyPeriodCombo::KMyMoneyPeriodCombo(TQWidget* parent, const char* name) :
  KMyMoneyGeneralCombo(parent, name)
{
  insertItem(i18n("All dates"), MyMoneyTransactionFilter::allDates);
  insertItem(i18n("As of today"), MyMoneyTransactionFilter::asOfToday);
  insertItem(i18n("Today"), MyMoneyTransactionFilter::today);
  insertItem(i18n("Current month"), MyMoneyTransactionFilter::currentMonth);
  insertItem(i18n("Current quarter"), MyMoneyTransactionFilter::currentQuarter);
  insertItem(i18n("Current year"), MyMoneyTransactionFilter::currentYear);
  insertItem(i18n("Current fiscal year"), MyMoneyTransactionFilter::currentFiscalYear);
  insertItem(i18n("Month to date"), MyMoneyTransactionFilter::monthToDate);
  insertItem(i18n("Year to date"), MyMoneyTransactionFilter::yearToDate);
  insertItem(i18n("Year to month"), MyMoneyTransactionFilter::yearToMonth);
  insertItem(i18n("Last month"), MyMoneyTransactionFilter::lastMonth);
  insertItem(i18n("Last year"), MyMoneyTransactionFilter::lastYear);
  insertItem(i18n("Last fiscal year"), MyMoneyTransactionFilter::lastFiscalYear);
  insertItem(i18n("Last 7 days"), MyMoneyTransactionFilter::last7Days);
  insertItem(i18n("Last 30 days"), MyMoneyTransactionFilter::last30Days);
  insertItem(i18n("Last 3 months"), MyMoneyTransactionFilter::last3Months);
  insertItem(i18n("Last quarter"), MyMoneyTransactionFilter::lastQuarter);
  insertItem(i18n("Last 6 months"), MyMoneyTransactionFilter::last6Months);
  insertItem(i18n("Last 11 months"), MyMoneyTransactionFilter::last11Months);
  insertItem(i18n("Last 12 months"), MyMoneyTransactionFilter::last12Months);
  insertItem(i18n("Next 7 days"), MyMoneyTransactionFilter::next7Days);
  insertItem(i18n("Next 30 days"), MyMoneyTransactionFilter::next30Days);
  insertItem(i18n("Next 3 months"), MyMoneyTransactionFilter::next3Months);
  insertItem(i18n("Next quarter"), MyMoneyTransactionFilter::lastQuarter);
  insertItem(i18n("Next 6 months"), MyMoneyTransactionFilter::next6Months);
  insertItem(i18n("Next 12 months"), MyMoneyTransactionFilter::next12Months);
  insertItem(i18n("Last 3 months to next 3 months"), MyMoneyTransactionFilter::last3ToNext3Months);
  insertItem(i18n("User defined"), MyMoneyTransactionFilter::userDefined);
}

void KMyMoneyPeriodCombo::setCurrentItem(MyMoneyTransactionFilter::dateOptionE id)
{
  if(id >= MyMoneyTransactionFilter::dateOptionCount)
    id = MyMoneyTransactionFilter::userDefined;

  KMyMoneyGeneralCombo::setCurrentItem(id);
}

MyMoneyTransactionFilter::dateOptionE KMyMoneyPeriodCombo::currentItem(void) const
{
  return static_cast<MyMoneyTransactionFilter::dateOptionE>(KMyMoneyGeneralCombo::currentItem());
}

TQDate KMyMoneyPeriodCombo::start(MyMoneyTransactionFilter::dateOptionE id)
{
  TQDate start, end;
  MyMoneyTransactionFilter::translateDateRange(id, start, end);
  return start;
}

TQDate KMyMoneyPeriodCombo::end(MyMoneyTransactionFilter::dateOptionE id)
{
  TQDate start, end;
  MyMoneyTransactionFilter::translateDateRange(id, start, end);
  return end;
}

#if 0
void KMyMoneyPeriodCombo::dates(TQDate& start, TQDate& end, MyMoneyTransactionFilter::dateOptionE id)
{
}
#endif

KMyMoneyOccurenceCombo::KMyMoneyOccurenceCombo(TQWidget* parent, const char* name) :
  KMyMoneyGeneralCombo(parent, name)
{
}

MyMoneySchedule::occurenceE KMyMoneyOccurenceCombo::currentItem(void) const
{
  return static_cast<MyMoneySchedule::occurenceE>(KMyMoneyGeneralCombo::currentItem());
}

KMyMoneyOccurencePeriodCombo::KMyMoneyOccurencePeriodCombo(TQWidget* parent, const char* name) :
  KMyMoneyOccurenceCombo(parent, name)
{
  insertItem(i18n(MyMoneySchedule::occurencePeriodToString(MyMoneySchedule::OCCUR_ONCE)), MyMoneySchedule::OCCUR_ONCE);
  insertItem(i18n(MyMoneySchedule::occurencePeriodToString(MyMoneySchedule::OCCUR_DAILY)), MyMoneySchedule::OCCUR_DAILY);
  insertItem(i18n(MyMoneySchedule::occurencePeriodToString(MyMoneySchedule::OCCUR_WEEKLY)), MyMoneySchedule::OCCUR_WEEKLY);
  insertItem(i18n(MyMoneySchedule::occurencePeriodToString(MyMoneySchedule::OCCUR_EVERYHALFMONTH)), MyMoneySchedule::OCCUR_EVERYHALFMONTH);
  insertItem(i18n(MyMoneySchedule::occurencePeriodToString(MyMoneySchedule::OCCUR_MONTHLY)), MyMoneySchedule::OCCUR_MONTHLY);
  insertItem(i18n(MyMoneySchedule::occurencePeriodToString(MyMoneySchedule::OCCUR_YEARLY)), MyMoneySchedule::OCCUR_YEARLY);
}

KMyMoneyFrequencyCombo::KMyMoneyFrequencyCombo(TQWidget* parent, const char* name) :
  KMyMoneyOccurenceCombo(parent, name)
{
  insertItem(i18n(MyMoneySchedule::occurenceToString(MyMoneySchedule::OCCUR_ONCE)), MyMoneySchedule::OCCUR_ONCE);
  insertItem(i18n(MyMoneySchedule::occurenceToString(MyMoneySchedule::OCCUR_DAILY)), MyMoneySchedule::OCCUR_DAILY);
  insertItem(i18n(MyMoneySchedule::occurenceToString(MyMoneySchedule::OCCUR_WEEKLY)), MyMoneySchedule::OCCUR_WEEKLY);
  insertItem(i18n(MyMoneySchedule::occurenceToString(MyMoneySchedule::OCCUR_EVERYOTHERWEEK)), MyMoneySchedule::OCCUR_EVERYOTHERWEEK);
  insertItem(i18n(MyMoneySchedule::occurenceToString(MyMoneySchedule::OCCUR_EVERYHALFMONTH)), MyMoneySchedule::OCCUR_EVERYHALFMONTH);
  insertItem(i18n(MyMoneySchedule::occurenceToString(MyMoneySchedule::OCCUR_EVERYTHREEWEEKS)), MyMoneySchedule::OCCUR_EVERYTHREEWEEKS);
  insertItem(i18n(MyMoneySchedule::occurenceToString(MyMoneySchedule::OCCUR_EVERYTHIRTYDAYS)), MyMoneySchedule::OCCUR_EVERYTHIRTYDAYS);
  insertItem(i18n(MyMoneySchedule::occurenceToString(MyMoneySchedule::OCCUR_EVERYFOURWEEKS)), MyMoneySchedule::OCCUR_EVERYFOURWEEKS);
  insertItem(i18n(MyMoneySchedule::occurenceToString(MyMoneySchedule::OCCUR_MONTHLY)), MyMoneySchedule::OCCUR_MONTHLY);
  insertItem(i18n(MyMoneySchedule::occurenceToString(MyMoneySchedule::OCCUR_EVERYEIGHTWEEKS)), MyMoneySchedule::OCCUR_EVERYEIGHTWEEKS);
  insertItem(i18n(MyMoneySchedule::occurenceToString(MyMoneySchedule::OCCUR_EVERYOTHERMONTH)), MyMoneySchedule::OCCUR_EVERYOTHERMONTH);
  insertItem(i18n(MyMoneySchedule::occurenceToString(MyMoneySchedule::OCCUR_EVERYTHREEMONTHS)), MyMoneySchedule::OCCUR_EVERYTHREEMONTHS);
  insertItem(i18n(MyMoneySchedule::occurenceToString(MyMoneySchedule::OCCUR_EVERYFOURMONTHS)), MyMoneySchedule::OCCUR_EVERYFOURMONTHS);
  insertItem(i18n(MyMoneySchedule::occurenceToString(MyMoneySchedule::OCCUR_TWICEYEARLY)), MyMoneySchedule::OCCUR_TWICEYEARLY);
  insertItem(i18n(MyMoneySchedule::occurenceToString(MyMoneySchedule::OCCUR_YEARLY)), MyMoneySchedule::OCCUR_YEARLY);
  insertItem(i18n(MyMoneySchedule::occurenceToString(MyMoneySchedule::OCCUR_EVERYOTHERYEAR)), MyMoneySchedule::OCCUR_EVERYOTHERYEAR);
}

int KMyMoneyFrequencyCombo::daysBetweenEvents(void) const
{
  return MyMoneySchedule::daysBetweenEvents(currentItem());
}

int KMyMoneyFrequencyCombo::eventsPerYear(void) const
{
  return MyMoneySchedule::eventsPerYear(currentItem());
}
#include "kmymoneycombo.moc"
