/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "configdialog.h"
#include "misc.h"
#include "misc/config.h"
#include "widgets/systemtray.h"

#include <tdelocale.h>
#include <tdefontdialog.h>
#include <kcolorbutton.h>
#include <kurlrequester.h>
#include <klineedit.h>
#include <knuminput.h>
#include <tdeglobal.h>
#include <kcharsets.h>
#include <kcombobox.h>

#include <ntqframe.h>
#include <ntqlayout.h>
#include <ntqcheckbox.h>

// Config layouts
#include "ui/config_general.h"
#include "ui/config_transfers.h"
#include "ui/config_log.h"
#include "ui/config_display.h"

#include "configfilter.h"

namespace KFTPWidgets {

ConfigDialog::ConfigDialog(TQWidget *parent, const char *name)
  : TDEConfigDialog(parent, name, KFTPCore::Config::self(), KDialogBase::TreeList, Help|Ok|Apply|Cancel, Ok, true)
{
  // Add all standard pages
  addPage(new GeneralSettings(0, "General"), i18n("General"), "kfm");
  addPage(new TransferSettings(0, "Transfers"), i18n("Transfers"), "queue");
  addPage(new LogSettings(0, "Log"), i18n("Log"), "text-x-log");
  addPage(new DisplaySettings(0, "Display"), i18n("Display"), "display");
  
  // Add  the actions page
  TQFrame *aFrame = new TQFrame();
  TQVBoxLayout *aLayout = new TQVBoxLayout(aFrame, 0, 0);
  aLayout->addWidget(KFTPCore::Config::self()->dActions()->getConfigWidget(aFrame));
  aLayout->addSpacing(KDialog::spacingHint());
  aLayout->addWidget(KFTPCore::Config::self()->uActions()->getConfigWidget(aFrame));
  aLayout->addSpacing(KDialog::spacingHint());
  aLayout->addWidget(KFTPCore::Config::self()->fActions()->getConfigWidget(aFrame));
  aLayout->addStretch(1);
  addPage(aFrame, i18n("Actions"), "kfm");
  
  // Add the filter page
  aFrame = new TQFrame();
  aLayout = new TQVBoxLayout(aFrame, 0, 0);
  m_configFilter = new ConfigFilter(aFrame);
  aLayout->addWidget(m_configFilter);
  addPage(aFrame, i18n("Filters"), "kfm");
  
  // Setup some stuff
  static_cast<KURLRequester*>(child("kcfg_defLocalDir"))->setMode(KFile::Directory | KFile::ExistingOnly | KFile::LocalOnly);
  
  // Setup signals
  connect(child("kcfg_globalMail"), SIGNAL(toggled(bool)), this, SLOT(slotGeneralEmailChanged(bool)));
  connect(child("kcfg_portForceIp"), SIGNAL(toggled(bool)), this, SLOT(slotTransfersForceIpChanged(bool)));
  connect(child("kcfg_activeForcePort"), SIGNAL(toggled(bool)), this, SLOT(slotTransfersForcePortChanged(bool)));
  connect(child("kcfg_diskCheckSpace"), SIGNAL(toggled(bool)), this, SLOT(slotTransfersDiskChanged(bool)));
  connect(child("kcfg_failedAutoRetry"), SIGNAL(toggled(bool)), this, SLOT(slotTransfersFailedRetryChanged(bool)));
  connect(child("kcfg_saveToFile"), SIGNAL(toggled(bool)), this, SLOT(slotLogOutputFileChanged(bool)));
  
  // Refresh states
  slotGeneralEmailChanged(static_cast<TQCheckBox*>(child("kcfg_globalMail"))->isChecked());
  slotTransfersForceIpChanged(static_cast<TQCheckBox*>(child("kcfg_portForceIp"))->isChecked());
  slotTransfersDiskChanged(static_cast<TQCheckBox*>(child("kcfg_diskCheckSpace"))->isChecked());
  slotTransfersFailedRetryChanged(static_cast<TQCheckBox*>(child("kcfg_failedAutoRetry"))->isChecked());
  slotLogOutputFileChanged(static_cast<TQCheckBox*>(child("kcfg_saveToFile"))->isChecked());
  slotTransfersForcePortChanged(static_cast<TQCheckBox*>(child("kcfg_activeForcePort"))->isChecked());
  
  // Let the config be up-to-date
  connect(this, SIGNAL(settingsChanged()), KFTPCore::Config::self(), SLOT(emitChange()));
  connect(this, SIGNAL(okClicked()), this, SLOT(slotSettingsChanged()));
}

void ConfigDialog::prepareDialog()
{
  // Update the actions
  KFTPCore::Config::self()->dActions()->updateWidget();
  KFTPCore::Config::self()->uActions()->updateWidget();
  KFTPCore::Config::self()->fActions()->updateWidget();
  
  // Populate charsets
  TQStringList charsets = TDEGlobal::charsets()->descriptiveEncodingNames();
  static_cast<KComboBox*>(child("cfg_defEncoding"))->insertStringList(charsets);
  
  TQString defEncoding = KFTPCore::Config::defEncoding();
  defEncoding = TQString("%1 ( %2 )").arg(TDEGlobal::charsets()->languageForEncoding(defEncoding)).arg(defEncoding);
  static_cast<KComboBox*>(child("cfg_defEncoding"))->setCurrentText(defEncoding);
  
  m_configFilter->loadSettings();
}

void ConfigDialog::slotSettingsChanged()
{
  // Update the actions
  KFTPCore::Config::self()->dActions()->updateConfig();
  KFTPCore::Config::self()->uActions()->updateConfig();
  KFTPCore::Config::self()->fActions()->updateConfig();
  
  m_configFilter->saveSettings();
  
  // Save encoding
  KFTPCore::Config::setDefEncoding(TDEGlobal::charsets()->encodingForName(static_cast<KComboBox*>(child("cfg_defEncoding"))->currentText()));
  
  // Show/hide the systray icon
  if (KFTPCore::Config::showSystrayIcon())
    SystemTray::self()->show();
  else
    SystemTray::self()->hide();
}

void ConfigDialog::slotGeneralEmailChanged(bool on)
{
  static_cast<TQWidget*>(child("kcfg_anonMail"))->setEnabled(!on);
}

void ConfigDialog::slotTransfersForceIpChanged(bool on)
{
  static_cast<TQWidget*>(child("kcfg_portIp"))->setEnabled(on);
}

void ConfigDialog::slotTransfersForcePortChanged(bool on)
{
  static_cast<TQWidget*>(child("kcfg_activeMinPort"))->setEnabled(on);
  static_cast<TQWidget*>(child("kcfg_activeMaxPort"))->setEnabled(on);
}

void ConfigDialog::slotTransfersDiskChanged(bool on)
{
  static_cast<TQWidget*>(child("kcfg_diskCheckInterval"))->setEnabled(on);
  static_cast<TQWidget*>(child("kcfg_diskMinFreeSpace"))->setEnabled(on);
}

void ConfigDialog::slotTransfersFailedRetryChanged(bool on)
{
  static_cast<TQWidget*>(child("kcfg_failedAutoRetryCount"))->setEnabled(on);
}

void ConfigDialog::slotLogOutputFileChanged(bool on)
{
  static_cast<TQWidget*>(child("kcfg_outputFilename"))->setEnabled(on);
}

}

#include "configdialog.moc"
