/***************************************************************************
 *   Copyright (C) 2004-2007 by Georgy Yunaev, gyunaev@ulduzsoft.com       *
 *   Please do not use email address above for bug reports; see            *
 *   the README file                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#include <tqlayout.h>
#include <tqheader.h>

#include "libchmfile.h"

#include "kchmmainwindow.h"
#include "kchmindexwindow.h"
#include "kchmlistitemtooltip.h"
#include "kchmtreeviewitem.h"

#include "kchmindexwindow.moc"

KCHMIndexWindow::KCHMIndexWindow ( TQWidget * parent, const char * name, WFlags f )
	: TQWidget (parent, name, f)
{
	TQVBoxLayout * layout = new TQVBoxLayout (this);
	layout->setMargin (5);

	m_indexFinder = new TQLineEdit (this);
	m_indexFinder->setFocus();
	
	m_indexList = new KQListView (this);
	m_indexList->addColumn( "idx" ); // it is hidden anyway
	m_indexList->header()->hide();
	m_indexList->setTreeStepSize (10);
	m_indexList->setShowToolTips(true);
	//m_indexList->setSorting( 0, true );

	layout->addWidget (m_indexFinder);
	layout->addSpacing (10);
	layout->addWidget (m_indexList);
	
	connect( m_indexFinder, 
			 TQT_SIGNAL( textChanged (const TQString &) ), 
			 this, 
			 TQT_SLOT( onTextChanged(const TQString &) ) );
	
	connect( m_indexFinder, 
			 TQT_SIGNAL( returnPressed() ), 
			 this, 
			 TQT_SLOT( onReturnPressed() ) );
	
	connect( m_indexList, 
			 TQT_SIGNAL( doubleClicked ( TQListViewItem *, const TQPoint &, int) ), 
			 this, 
			 TQT_SLOT( onDoubleClicked ( TQListViewItem *, const TQPoint &, int) ) );
	
	connect( m_indexList,
			 TQT_SIGNAL( contextMenuRequested( TQListViewItem *, const TQPoint& , int ) ),
			 this, 
			 TQT_SLOT( slotContextMenuRequested ( TQListViewItem *, const TQPoint &, int ) ) );

	m_indexListFilled = false;
	m_lastSelectedItem = 0;
	m_contextMenu = 0;
	
	new KCHMListItemTooltip( m_indexList );
}

void KCHMIndexWindow::onTextChanged ( const TQString & newvalue)
{
	m_lastSelectedItem = m_indexList->findItem (newvalue, 0, TQt::BeginsWith);
	
	if ( m_lastSelectedItem )
	{
		m_indexList->ensureItemVisible (m_lastSelectedItem);
		m_indexList->setCurrentItem (m_lastSelectedItem);
	}
}

void KCHMIndexWindow::showEvent( TQShowEvent * )
{
	if ( !::mainWindow->chmFile() || m_indexListFilled )
		return;

	m_indexListFilled = true;
	refillIndex();
}

void KCHMIndexWindow::onReturnPressed( )
{
	emit ::mainWindow->slotOnTreeClicked ( m_lastSelectedItem );
}


void KCHMIndexWindow::invalidate( )
{
	m_indexList->clear();
	m_indexListFilled = false;
}

void KCHMIndexWindow::onDoubleClicked( TQListViewItem *item, const TQPoint &, int )
{
	if ( !item )
		return;
	
	KCHMIndTocItem * treeitem = (KCHMIndTocItem*) item;
	
	TQString url = treeitem->getUrl();
	
	if ( !url )
		return;

	if ( url[0] == ':' ) // 'see also' link
	{
		m_lastSelectedItem = m_indexList->findItem (url.mid(1), 0);
		if ( m_lastSelectedItem )
		{
			m_indexList->ensureItemVisible (m_lastSelectedItem);
			m_indexList->setCurrentItem (m_lastSelectedItem);
		}
	}
	else
		::mainWindow->openPage( url, OPF_CONTENT_TREE | OPF_ADD2HISTORY );
}

void KCHMIndexWindow::slotContextMenuRequested( TQListViewItem * item, const TQPoint & point, int )
{
	if ( !m_contextMenu )
		m_contextMenu = ::mainWindow->currentBrowser()->createListItemContextMenu( this );
		
	if( item )
	{
		KCHMIndTocItem * treeitem = (KCHMIndTocItem*) item;
		
		::mainWindow->currentBrowser()->setTabKeeper( treeitem->getUrl() );
		m_contextMenu->popup( point );
	}
}

void KCHMIndexWindow::refillIndex( )
{
	TQValueVector< LCHMParsedEntry > data;
	
	if ( !::mainWindow->chmFile()->parseIndex( &data )
			   || data.size() == 0 )
	{
		tqWarning ("CHM index present but is empty; wrong parsing?");
		return;
	}
	
	kchmFillListViewWithParsedData( m_indexList, data, 0 );
}

void KCHMIndexWindow::search( const TQString & index )
{
	if ( !::mainWindow->chmFile() )
		return;

	if ( !m_indexListFilled )
	{
		m_indexListFilled = true;
		refillIndex();
	}

	m_indexFinder->setText( index );
	onTextChanged( index );
}
