/***************************************************************************
 *   Copyright (C) 2004-2009 by Thomas Fischer                             *
 *   fischer@unix-ag.uni-kl.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <stdio.h>

#include <ntqfile.h>
#include <ntqdir.h>
#include <ntqprocess.h>
#include <ntqstringlist.h>
#include <ntqbuffer.h>
#include <ntqwaitcondition.h>
#include <ntqapplication.h>

#include <file.h>
#include <encoderlatex.h>
#include "fileimporterbibutils.h"

namespace BibTeX
{

    FileImporterBibUtils::FileImporterBibUtils( BibTeX::File::FileFormat inputFormat )
            : FileImporter(), m_workingDir( createTempDir() ), m_inputFormat( inputFormat ), m_bibTeXImporter( new FileImporterBibTeX( false, "utf-8" ) )
    {
        m_processBuffer = new TQBuffer();
    }

    FileImporterBibUtils::~FileImporterBibUtils()
    {
        delete m_processBuffer;
        deleteTempDir( m_workingDir );
        delete m_bibTeXImporter;
    }

    File* FileImporterBibUtils::load( TQIODevice *iodevice )
    {
        m_cancelFlag = false;
        if ( !iodevice->isReadable() )
        {
            tqDebug( "iodevice is not readable" );
            return NULL;
        }
        if ( !iodevice->isOpen() )
        {
            tqDebug( "iodevice is not open" );
            return NULL;
        }

        if ( !iodeviceToXMLbuffer( iodevice ) || m_cancelFlag )
            return NULL;

        return xmlBufferToBibTeXFile();
    }

    bool FileImporterBibUtils::guessCanDecode( const TQString & text )
    {
        return guessInputFormat( text ) != BibTeX::File::formatUndefined;
    }

    File::FileFormat FileImporterBibUtils::guessInputFormat( const TQString &text )
    {
        if ( text.find( "TY  - " ) >= 0 )
            return BibTeX::File::formatRIS;
        else if ( text.find( "%A " ) >= 0 )
            return BibTeX::File::formatEndNote;
        else if ( text.find( "FN ISI Export Format" ) >= 0 )
            return BibTeX::File::formatISI;
        else
            return BibTeX::File::formatUndefined;
    }

    void FileImporterBibUtils::cancel()
    {
        m_bibTeXImporter->cancel();
        m_cancelFlag = true;
    }

    bool FileImporterBibUtils::iodeviceToXMLbuffer( TQIODevice *iodevice )
    {
        TQWaitCondition wc;

        m_processBuffer->open( IO_WriteOnly );
        m_process = NULL;
        switch ( m_inputFormat )
        {
        case BibTeX::File::formatISI:
            m_process = new TQProcess( TQStringList::split( ' ', "isi2xml -i utf8 -u" ) );
            break;
        case BibTeX::File::formatWordBib:
            m_process = new TQProcess( TQStringList::split( ' ', "wordbib2xml -i utf8 -u" ) );
            break;
        case BibTeX::File::formatAds:
            m_process = new TQProcess( TQStringList::split( ' ', "ads2xml -i utf8 -u" ) );
            break;
        case BibTeX::File::formatEndNote:
            m_process = new TQProcess( TQStringList::split( ' ', "end2xml -i utf8 -u" ) );
            break;
        case BibTeX::File::formatEndNoteXML:
            m_process = new TQProcess( TQStringList::split( ' ', "endx2xml -i utf8 -u" ) );
            break;
        case BibTeX::File::formatRIS:
            m_process = new TQProcess( TQStringList::split( ' ', "ris2xml -i utf8 -u" ) );
            break;
        case BibTeX::File::formatMODS:
            /* m_process = NULL; */
            break;
        default:
            tqDebug( "Cannot handle input format %i", m_inputFormat );
            return false;
        }

        if ( m_process != NULL )
        {
            m_waiting = true;
            connect( m_process, SIGNAL( processExited() ), this, SLOT( wakeUp() ) );
            connect( m_process, SIGNAL( readyReadStdout() ), this, SLOT( slotReadyStdout() ) );
            connect( m_process, SIGNAL( readyReadStderr() ), this, SLOT( slotReadyStderr() ) );

            m_process->start();
            if ( m_process->isRunning() )
            {
                TQByteArray inData = iodevice->readAll();
                m_process->writeToStdin( inData );
                tqApp->processEvents();
                m_process->closeStdin();

                int nothingHappens = 20;
                while ( m_waiting )
                {
                    wc.wait( 250 );
                    tqApp->processEvents();
                    --nothingHappens;
                }

                if ( nothingHappens <= 0 )
                    m_process->kill();

                if ( !m_process->normalExit() )
                {
                    tqDebug( "%s did not exit in a clean fashion", m_process->arguments()[0].latin1() );
                    delete m_process;
                    return false;
                }
            }
            else
            {
                tqDebug( "%s did not start", m_process->arguments()[0].latin1() );
                delete m_process;
                return false;
            }
        }
        else
        {
            m_processBuffer->writeBlock( iodevice->readAll() );
        }

        m_processBuffer->close();

        delete m_process;
        return true;
    }

    BibTeX::File* FileImporterBibUtils::xmlBufferToBibTeXFile()
    {
        TQWaitCondition wc;

        m_waiting = true;
        m_process = new TQProcess( TQStringList::split( ' ', "xml2bib -i utf8 -o utf8 -sk" ) );
        connect( m_process, SIGNAL( processExited() ), this, SLOT( wakeUp() ) );
        connect( m_process, SIGNAL( readyReadStdout() ), this, SLOT( slotReadyStdout() ) );
        connect( m_process, SIGNAL( readyReadStderr() ), this, SLOT( slotReadyStderr() ) );

        if ( m_process->start() )
        {
            TQBuffer *tempBuffer = m_processBuffer;
            m_processBuffer = new TQBuffer();

            tempBuffer->open( IO_ReadOnly );
            m_process->writeToStdin( tempBuffer->readAll() );
            tqApp->processEvents();
            m_process->closeStdin();
            tempBuffer->close();

            m_processBuffer->open( IO_WriteOnly );
            int nothingHappens = 20;
            while ( m_waiting )
            {
                wc.wait( 250 );
                tqApp->processEvents();
                --nothingHappens;
            }
            m_processBuffer->close();

            if ( nothingHappens <= 0 )
                m_process->kill();

            delete tempBuffer;

            if ( ! m_process->normalExit() )
            {
                delete m_process;
                return NULL;
            }

            m_processBuffer->open( IO_ReadOnly );
            File *bibTeXFile = m_bibTeXImporter->load( m_processBuffer );
            m_processBuffer->close();

            delete m_process;
            return bibTeXFile;
        }
        else
        {
            delete m_process;
            return NULL;
        }

        delete m_process;
        return NULL;
    }

    TQString FileImporterBibUtils::createTempDir()
    {
        TQString result = TQString::null;
        TQFile *devrandom = new TQFile( "/dev/random" );

        if ( devrandom->open( IO_ReadOnly ) )
        {
            TQ_UINT32 randomNumber;
            if ( devrandom->readBlock(( char* ) & randomNumber, sizeof( randomNumber ) ) > 0 )
            {
                randomNumber |= 0x10000000;
                result = TQString( "/tmp/bibtex-%1" ).arg( randomNumber, sizeof( randomNumber ) * 2, 16 );
                if ( !TQDir().mkdir( result ) )
                    result = TQString::null;
            }
            devrandom->close();
        }

        delete devrandom;

        return result;
    }

    void FileImporterBibUtils::deleteTempDir( const TQString& directory )
    {
        TQDir dir = TQDir( directory );
        TQStringList subDirs = dir.entryList( TQDir::Dirs );
        for ( TQStringList::Iterator it = subDirs.begin(); it != subDirs.end(); it++ )
        {
            if (( TQString::compare( *it, "." ) != 0 ) && ( TQString::compare( *it, ".." ) != 0 ) )
                deleteTempDir( *it );
        }
        TQStringList allEntries = dir.entryList( TQDir::All );
        for ( TQStringList::Iterator it = allEntries.begin(); it != allEntries.end(); it++ )
            dir.remove( *it );

        TQDir().rmdir( directory );
    }

    void FileImporterBibUtils::wakeUp()
    {
        m_waiting = false;
    }

    void FileImporterBibUtils::slotReadyStdout()
    {
        m_processBuffer->writeBlock( m_process->readStdout() );
    }

    void FileImporterBibUtils::slotReadyStderr()
    {
        TQByteArray ba = m_process->readStderr();
        TQTextStream bats( ba, IO_ReadOnly );
        bats.setEncoding( TQTextStream::UnicodeUTF8 );
        tqDebug( "%s", bats.read().latin1() );
    }

}
#include "fileimporterbibutils.moc"

